// utils/safeDownload.js

import { ElMessage } from 'element-plus'

/**
 * 安全下载函数：适用于后端返回 Blob 文件流 或 JSON 错误 的场景
 * @param {Blob} blobData - 接口返回的响应数据（必须是 responseType: 'blob'）
 * @param {string} defaultFilename - 默认文件名（如 'data.xlsx'）
 * @param {string} [mimeType] - MIME 类型（用于兜底创建 Blob）
 */
export async function safeDownload(blobData, defaultFilename, mimeType = 'application/octet-stream') {
  if (!(blobData instanceof Blob)) {
    ElMessage.error('无效的下载数据')
    return
  }

  try {
    // 👇 关键：先 peek 前 100 字节，判断是否是 JSON 错误
    const firstChunk = await blobData.slice(0, 100).text()
    const trimmed = firstChunk.trim()

    // 如果看起来像 JSON（以 { 或 [ 开头），尝试解析
    if (trimmed.startsWith('{') || trimmed.startsWith('[')) {
      const fullText = await blobData.text()
      let parsed
      try {
        parsed = JSON.parse(fullText)
      } catch (e) {
        // 解析失败，当作正常文件（比如内容就是纯文本）
        parsed = null
      }

      // 如果解析成功，且包含错误字段（根据你后端约定）
      if (parsed && (parsed.code !== undefined || parsed.msg || parsed.message)) {
        const errorMsg = parsed.msg || parsed.message || '导出失败'
        ElMessage.error(errorMsg)
        return
      }
    }

    // ✅ 是合法文件流，执行下载（使用你已验证的逻辑）
    const url = window.URL.createObjectURL(blobData)
    const link = document.createElement('a')
    link.href = url
    link.download = defaultFilename
    document.body.appendChild(link)
    link.click()
    document.body.removeChild(link)
    window.URL.revokeObjectURL(url)

    ElMessage.success('下载成功')
  } catch (error) {
    console.error('safeDownload error:', error)
    ElMessage.error('下载过程中发生错误')
  }
}