package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.MenuTypeEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.common.utils.SM3Util;
import com.yd.common.utils.TreeUtils;
import com.yd.user.api.service.ApiRelTenantUserService;
import com.yd.user.api.service.ApiSysUserService;
import com.yd.user.feign.request.sysuser.*;
import com.yd.user.feign.response.login.*;
import com.yd.user.feign.response.sysuser.ApiScopePageResponse;
import com.yd.user.feign.response.sysuser.ApiSysUserDetailResponse;
import com.yd.user.feign.response.sysuser.ApiSysUserPageResponse;
import com.yd.user.service.dto.*;
import com.yd.user.service.model.*;
import com.yd.user.service.service.*;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiSysUserServiceImpl implements ApiSysUserService {

    @Autowired
    private ISysUserService iSysUserService;

    @Autowired
    private ApiRelTenantUserService apiRelTenantUserService;

    @Autowired
    private IRelTenantUserService iRelTenantUserService;

    @Autowired
    private ISysTenantService iSysTenantService;

    @Autowired
    private ISysRoleService iSysRoleService;

    @Autowired
    private ISysMenuService iSysMenuService;

    @Autowired
    private IRelUserRoleService iRelUserRoleService;

    @Autowired
    private IRelRoleMenuService iRelRoleMenuService;

    @Autowired
    private IRelTenantProjectService iRelTenantProjectService;

    @Autowired
    private IRelProjectUserService iRelProjectUserService;

    @Autowired
    private ISysProjectService iSysProjectService;

    /**
     * 用户分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiSysUserPageResponse>> page(ApiSysUserPageRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();
        Page<ApiSysUserPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiSysUserPageResponse> iPage = iSysUserService.page(page, request,isAdmin,loginTenantBizId);
        return Result.success(iPage);
    }

    /**
     * 添加用户
     * @param request
     * @return
     */
    @Override
    public Result add(ApiSysUserAddRequest request) {
        LocalDateTime now = LocalDateTime.now();
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        //添加和编辑校验入参的唯一性
        addAndEditReqCheckCommon(
                request.getUserName(),
                request.getMobile(),
                request.getEmail(),
                CommonEnum.OPR_SOURCE_ADD.getCode(),
                "");

        //新增用户数据
        SysUser sysUser = new SysUser();
        BeanUtils.copyProperties(request,sysUser);

        //SM3国密算法加密密码
        sysUser = encryptPwd(sysUser,request.getPassword());

        //用户业务唯一id
        sysUser.setUserBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_USER.getCode()));
        sysUser.setIsSuperAdmin(0);
        sysUser.setLastPasswordTime(now);
        sysUser.setSourceType(1);
        iSysUserService.saveOrUpdate(sysUser);

        if (!isAdmin) {
            //非超级管理员，添加租户和用户的关系
            apiRelTenantUserService.addSingleRel(loginTenantBizId,sysUser.getUserBizId());
        }
        return Result.success();
    }

    /**
     * 编辑用户
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiSysUserEditRequest request) {
        LocalDateTime now = LocalDateTime.now();

        //校验用户是否存在
        Result<SysUser> result = checkSysUserIsExist(request.getUserBizId());
        SysUser sysUser = result.getData();

        //添加和编辑校验入参的唯一性
        addAndEditReqCheckCommon(
                request.getUserName(),
                request.getMobile(),
                request.getEmail(),
                CommonEnum.OPR_SOURCE_EDIT.getCode(),
                request.getUserBizId());

        //校验入参密码是否修改过
//        Boolean pwdIsEdit = checkPwdIsEdit(request.getPassword(),sysUser.getPassword(),sysUser.getPasswordSalt());
        //编辑用户
        BeanUtils.copyProperties(request,sysUser);
        sysUser.setLastPasswordTime(now);
        sysUser.setSourceType(1);

        //修改过密码，重新加密
//        if (pwdIsEdit) {
//            //SM3国密算法加密密码
//            sysUser = encryptPwd(sysUser,request.getPassword());
//        }

        iSysUserService.saveOrUpdate(sysUser);

        return Result.success();
    }

    /**
     * SM3国密算法加密密码
     * @param sysUser
     * @return
     */
    public SysUser encryptPwd(SysUser sysUser,String password) {
        //密码：SM3国密算法
        //生成盐值
        String salt = SM3Util.generateSalt();
        //生成加密密码：带盐的SM3哈希
        String hashedPassword = SM3Util.hashWithSalt(password, salt);
        sysUser.setPasswordSalt(salt);
        sysUser.setPassword(hashedPassword);

        return sysUser;
    }

    /**
     * 校验入参密码是否修改过
     * @param reqPwd 入参修改的密码
     * @param encryptPwd 库里加密密码
     * @param salt 库里盐值
     * @return
     */
    public Boolean checkPwdIsEdit(String reqPwd,String encryptPwd,String salt) {
        //入参密码加密
        String hashedPassword = SM3Util.hashWithSalt(reqPwd, salt);
        return !hashedPassword.equals(encryptPwd);
    }

    /**
     * 添加和编辑校验入参的唯一性
     * @param userName
     * @param mobile
     * @param email
     */
    public void addAndEditReqCheckCommon(String userName,
                                         String mobile,
                                         String email,
                                         String oprSource,
                                         String userBizId) {
        Boolean isExcludeMy = true;
        if (CommonEnum.OPR_SOURCE_ADD.getCode().equals(oprSource)) {
            //添加来源
            userBizId = "";
            isExcludeMy = false;
        }
        List<SysUser> userNameList = iSysUserService.queryList(SysUserDto.builder()
                .userName(userName)
                .userBizId(userBizId)
                .isExcludeMy(isExcludeMy)
                .build());
        if (!CollectionUtils.isEmpty(userNameList)) {
            throw new BusinessException("用户账号已存在");
        }

        List<SysUser> mobileList = iSysUserService.queryList(SysUserDto.builder()
                .userName(mobile)
                .userBizId(userBizId)
                .isExcludeMy(isExcludeMy)
                .build());
        if (!CollectionUtils.isEmpty(mobileList)) {
            throw new BusinessException("手机号已存在");
        }

        List<SysUser> emailList = iSysUserService.queryList(SysUserDto.builder()
                .userName(email)
                .userBizId(userBizId)
                .isExcludeMy(isExcludeMy)
                .build());
        if (!CollectionUtils.isEmpty(emailList)) {
            throw new BusinessException("邮箱已存在");
        }

    }

    /**
     * 获取用户详情
     * @param userBizId
     * @return
     */
    @Override
    public Result<ApiSysUserDetailResponse> detail(String userBizId) {
        SysUser sysUser = iSysUserService.queryOne(userBizId);
        ApiSysUserDetailResponse response = null;
        if (!Objects.isNull(sysUser)) {
            response = new ApiSysUserDetailResponse();
            BeanUtils.copyProperties(sysUser,response);
        }

        return Result.success(response);
    }

    /**
     * 编辑用户状态
     * @param userBizId
     * @param status
     * @return
     */
    @Override
    public Result editStatus(String userBizId, Integer status) {
        //校验用户是否存在
        Result<SysUser> result = checkSysUserIsExist(userBizId);
        SysUser sysUser = result.getData();

        sysUser.setStatus(status);
        iSysUserService.saveOrUpdate(sysUser);
        return Result.success();
    }

    /**
     * 编辑用户密码
     * @return
     */
    @Override
    public Result editPwd(ApiSysUserEditPwdRequest request) {
        //校验用户是否存在
        Result<SysUser> result = checkSysUserIsExist(request.getUserBizId());
        SysUser sysUser = result.getData();

        LocalDateTime now = LocalDateTime.now();
        //SM3国密算法加密密码
        sysUser = encryptPwd(sysUser,request.getPassword());
        sysUser.setLastPasswordTime(now);

        iSysUserService.saveOrUpdate(sysUser);

        return Result.success();
    }

    /**
     * 管理端-获取用户登录后权限信息
     * @return
     */
    @Override
    public Result<GetLoginInfoResponse> getLoginInfo() {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();

        GetLoginInfoResponse response = new GetLoginInfoResponse();
        //用户信息
        ApiLoginUserInfoResponse userInfoResponse = new ApiLoginUserInfoResponse();
        SysUser sysUser = iSysUserService.queryOne(authUserDto.getUserBizId());
        if (Objects.isNull(sysUser)) {
            //用户不存在
            throw new BusinessException(ResultCode.USER_NOT_EXISTS.getCode(),ResultCode.USER_NOT_EXISTS.getMessage());
        }
        BeanUtils.copyProperties(sysUser,userInfoResponse);
        response.setApiLoginUserInfoResponse(userInfoResponse);

        //用户绑定租户列表信息
        List<ApiLoginTenantResponse> apiLoginTenantResponseList = new ArrayList<>();

        List<RelTenantUser> relTenantUserList = iRelTenantUserService.queryList(RelTenantUserDto.builder()
                .userBizId(authUserDto.getUserBizId())
                .build());
        if (!CollectionUtils.isEmpty(relTenantUserList)) {
            //查询租户列表信息
            List<SysTenant> tenantList = iSysTenantService.queryList(relTenantUserList.stream().map(RelTenantUser::getTenantBizId).collect(Collectors.toList()));

            if (!CollectionUtils.isEmpty(tenantList)) {
                for (SysTenant sysTenant : tenantList) {
                    ApiLoginTenantResponse loginTenantResponse = new ApiLoginTenantResponse();
                    //租户信息
                    ApiLoginTenantInfoResponse apiLoginTenantInfoResponse = new ApiLoginTenantInfoResponse();
                    BeanUtils.copyProperties(sysTenant,apiLoginTenantInfoResponse);
                    loginTenantResponse.setApiLoginTenantInfoResponse(apiLoginTenantInfoResponse);

                    //查询租户绑定的项目列表
                    List<RelTenantProject> relTenantProjectList = iRelTenantProjectService.queryList(RelTenantProjectDto.builder()
                            .tenantBizId(sysTenant.getTenantBizId())
                            .build());
                    if (!CollectionUtils.isEmpty(relTenantProjectList)) {
                        List<String> projectBizIdList = relTenantProjectList.stream().map(RelTenantProject::getProjectBizId).collect(Collectors.toList());
                        List<SysProject> projectList = iSysProjectService.queryList(SysProjectDto.builder()
                                .projectBizIdList(projectBizIdList)
                                .build());
                        if (!CollectionUtils.isEmpty(projectList)) {
                            List<ApiLoginProjectInfoResponse> apiLoginProjectInfoResponseList = projectList.stream().map(dto -> {
                                ApiLoginProjectInfoResponse apiLoginProjectInfoResponse = new ApiLoginProjectInfoResponse();
                                BeanUtils.copyProperties(dto,apiLoginProjectInfoResponse);
                                return apiLoginProjectInfoResponse;
                            }).collect(Collectors.toList());
                            loginTenantResponse.setApiLoginProjectInfoResponseList(apiLoginProjectInfoResponseList);
                        }

                    }

                    List<RelUserRole> relUserRoleList = iRelUserRoleService.queryList(RelUserRoleDto.builder()
                            .tenantBizId(sysTenant.getTenantBizId())
                            .userBizId(authUserDto.getUserBizId())
                            .build());
                    if (!CollectionUtils.isEmpty(relUserRoleList)) {
                        //租户用户角色编码列表信息
                        List<String> roleBizIdList = relUserRoleList.stream().map(RelUserRole::getRoleBizId).collect(Collectors.toList());
                        List<SysRole> sysRoleList = iSysRoleService.queryList(SysRoleDto.builder()
                                .roleBizIdList(roleBizIdList)
                                .isExcludeMy(false)
                                .build());
                        if (!CollectionUtils.isEmpty(sysRoleList)) {
                            List<String> roleCodeList = sysRoleList.stream().map(SysRole::getRoleCode).collect(Collectors.toList());
                            loginTenantResponse.setRoles(roleCodeList);
                        }

                        //租户用户角色菜单按钮权限列表信息
                        List<RelRoleMenu> relRoleMenuList = iRelRoleMenuService.queryList(RelRoleMenuDto.builder()
                                .tenantBizId(sysTenant.getTenantBizId())
                                .roleBizIdList(roleBizIdList)
                                .build());
                        if (!CollectionUtils.isEmpty(relRoleMenuList)) {
                            //去重菜单id（多个角色绑定有重复的菜单）
                            List<String> menuBizIdList = relRoleMenuList.stream().map(RelRoleMenu::getMenuBizId).distinct().collect(Collectors.toList());
                            //查询按钮权限标识列表
                            List<SysMenu> anMenuList = iSysMenuService.queryList(SysMenuDto.builder()
                                    .menuBizIdList(menuBizIdList)
                                    .menuType(Integer.valueOf(MenuTypeEnum.AN.getCode()))
                                    .isExcludeMy(false)
                                    .build());
                            if (!CollectionUtils.isEmpty(anMenuList)) {
                                List<String> menuPermissionList = anMenuList.stream().map(SysMenu::getPermission).collect(Collectors.toList());
                                loginTenantResponse.setPermissions(menuPermissionList);
                            }

                            //租户用户角色菜单列表（树形结构）
//                            List<SysMenu> sysMenuList = iSysMenuService.queryList(SysMenuDto.builder()
//                                    .menuBizIdList(menuBizIdList)
//                                    .isExcludeMy(false)
//                                    .build());
//                            if (!CollectionUtils.isEmpty(sysMenuList)){
//                                List<ApiLoginMenuInfoResponse> apiLoginMenuInfoResponseList = sysMenuList
//                                        .stream()
//                                        .map(dto -> {
//                                            ApiLoginMenuInfoResponse menuInfoResponse = new ApiLoginMenuInfoResponse();
//                                            BeanUtils.copyProperties(dto,menuInfoResponse);
//                                            return menuInfoResponse;
//                                        }).collect(Collectors.toList());
//
//                                //构建菜单树形结构
//                                List<ApiLoginMenuInfoResponse> treeMenu = TreeUtils.buildTree(
//                                        apiLoginMenuInfoResponseList,
//                                        ApiLoginMenuInfoResponse::getMenuBizId,         // 获取节点ID
//                                        ApiLoginMenuInfoResponse::getParentBizId,   // 获取父节点ID
//                                        (parent, children) -> parent.setChildren(children), // 设置子节点
//                                        "0"                   // 根节点的父ID值
//                                );
//                                loginTenantResponse.setApiLoginMenuInfoResponseList(treeMenu);
//                            }
                        }
                    }
                    apiLoginTenantResponseList.add(loginTenantResponse);
                }
            }
        }

        response.setApiLoginTenantResponseList(apiLoginTenantResponseList);

        return Result.success(response);
    }

    /**
     * 管理端-获取当前登录用户绑定的当前租户用户角色菜单列表
     * @return
     */
    @Override
    public Result<List<ApiLoginMenuInfoResponse>> getMenuRouters(String tenantBizId) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();

        List<ApiLoginMenuInfoResponse> treeMenu = new ArrayList<>();

        List<RelUserRole> relUserRoleList = iRelUserRoleService.queryList(RelUserRoleDto.builder()
                .tenantBizId(tenantBizId)
                .userBizId(authUserDto.getUserBizId())
                .build());
        if (!CollectionUtils.isEmpty(relUserRoleList)) {
            //租户用户角色编码列表信息
            List<String> roleBizIdList = relUserRoleList.stream().map(RelUserRole::getRoleBizId).collect(Collectors.toList());

            //租户用户角色菜单按钮权限列表信息
            List<RelRoleMenu> relRoleMenuList = iRelRoleMenuService.queryList(RelRoleMenuDto.builder()
                    .tenantBizId(tenantBizId)
                    .roleBizIdList(roleBizIdList)
                    .build());
            if (!CollectionUtils.isEmpty(relRoleMenuList)) {
                //去重菜单id（多个角色绑定有重复的菜单）
                List<String> menuBizIdList = relRoleMenuList.stream().map(RelRoleMenu::getMenuBizId).distinct().collect(Collectors.toList());

                //租户用户角色菜单列表（树形结构）
                List<SysMenu> sysMenuList = iSysMenuService.queryList(SysMenuDto.builder()
                        .menuBizIdList(menuBizIdList)
                        .isExcludeMy(false)
                        .build());
                if (!CollectionUtils.isEmpty(sysMenuList)){
                    List<ApiLoginMenuInfoResponse> apiLoginMenuInfoResponseList = sysMenuList
                            .stream()
                            .map(dto -> {
                                ApiLoginMenuInfoResponse menuInfoResponse = new ApiLoginMenuInfoResponse();
                                BeanUtils.copyProperties(dto,menuInfoResponse);
                                return menuInfoResponse;
                            }).collect(Collectors.toList());

                    //构建菜单树形结构
                    treeMenu = TreeUtils.buildTree(
                            apiLoginMenuInfoResponseList,
                            ApiLoginMenuInfoResponse::getMenuBizId,         // 获取节点ID
                            ApiLoginMenuInfoResponse::getParentBizId,   // 获取父节点ID
                            (parent, children) -> parent.setChildren(children), // 设置子节点
                            "0"                   // 根节点的父ID值
                    );
                }
            }
        }
        return Result.success(treeMenu);
    }

    /**
     * 权限-管理端-根据作用域分页查询租户或项目列表
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiScopePageResponse>> scopePage(ApiScopePageRequest request) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //当前登录用户的角色类型 true是超级管理员 false是超级管理员
        Boolean isAdmin = authUserDto.getIsSuperAdmin() == 1;
        //当前所属的租户id（当前页面右上角显示的租户） 超级管理员不传值，非超级管理员根据传值来
        if (!isAdmin && StringUtils.isBlank(request.getLoginTenantBizId())) {
            //非超级管理员并且LoginTenantBizId为空，提示必须传值LoginTenantBizId
            throw new BusinessException("非超级管理员，当前登录用户所属的租户id（当前页面右上角显示的租户）必须传值");
        }
        String loginTenantBizId = isAdmin ? "" : request.getLoginTenantBizId();

        Page<ApiScopePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiScopePageResponse> iPage = iSysUserService.scopePage(page, request,isAdmin,loginTenantBizId);
        return Result.success(iPage);
    }

    /**
     * 权限-管理端-获取登录后的项目权限信息
     * @return
     */
    @Override
    public Result<GetLoginPermissionProjectResponse> getLoginPermissionProject(String tenantBizId, String projectBizId) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();

        GetLoginPermissionProjectResponse response = new GetLoginPermissionProjectResponse();
        //当前登录的用户基础信息
        ApiLoginUserInfoResponse userInfoResponse = new ApiLoginUserInfoResponse();
        SysUser sysUser = iSysUserService.queryOne(authUserDto.getUserBizId());
        if (Objects.isNull(sysUser)) {
            //用户不存在
            throw new BusinessException(ResultCode.USER_NOT_EXISTS.getCode(),ResultCode.USER_NOT_EXISTS.getMessage());
        }
        BeanUtils.copyProperties(sysUser,userInfoResponse);
        response.setApiLoginUserInfoResponse(userInfoResponse);

        //当前租户基础信息
        ApiLoginTenantInfoResponse apiLoginTenantInfoResponse = null;
        SysTenant sysTenant = iSysTenantService.queryOne(tenantBizId);
        if (!Objects.isNull(sysTenant)) {
            apiLoginTenantInfoResponse = new ApiLoginTenantInfoResponse();
            BeanUtils.copyProperties(sysTenant,apiLoginTenantInfoResponse);
            response.setApiLoginTenantInfoResponse(apiLoginTenantInfoResponse);
        }

        //当前项目基础信息
        ApiLoginProjectInfoResponse apiLoginProjectInfoResponse = null;
        SysProject sysProject = iSysProjectService.queryOne(projectBizId);
        if (!Objects.isNull(sysProject)) {
            apiLoginProjectInfoResponse = new ApiLoginProjectInfoResponse();
            BeanUtils.copyProperties(sysProject,apiLoginProjectInfoResponse);
            response.setApiLoginProjectInfoResponse(apiLoginProjectInfoResponse);
        }

        List<RelUserRole> relUserRoleList = iRelUserRoleService.queryList(RelUserRoleDto.builder()
                .projectBizId(projectBizId)
                .userBizId(authUserDto.getUserBizId())
                .build());
        if (!CollectionUtils.isEmpty(relUserRoleList)) {
            //项目用户角色编码列表信息
            List<String> roleBizIdList = relUserRoleList.stream().map(RelUserRole::getRoleBizId).collect(Collectors.toList());
            List<SysRole> sysRoleList = iSysRoleService.queryList(SysRoleDto.builder()
                    .roleBizIdList(roleBizIdList)
                    .isExcludeMy(false)
                    .build());
            if (!CollectionUtils.isEmpty(sysRoleList)) {
                List<String> roleCodeList = sysRoleList.stream().map(SysRole::getRoleCode).collect(Collectors.toList());
                response.setRoles(roleCodeList);
            }

            //项目用户角色菜单按钮权限列表信息
            List<RelRoleMenu> relRoleMenuList = iRelRoleMenuService.queryList(RelRoleMenuDto.builder()
                    .projectBizId(projectBizId)
                    .roleBizIdList(roleBizIdList)
                    .build());
            if (!CollectionUtils.isEmpty(relRoleMenuList)) {
                //去重菜单id（多个角色绑定有重复的菜单）
                List<String> menuBizIdList = relRoleMenuList.stream().map(RelRoleMenu::getMenuBizId).distinct().collect(Collectors.toList());
                //查询按钮权限标识列表
                List<SysMenu> anMenuList = iSysMenuService.queryList(SysMenuDto.builder()
                        .menuBizIdList(menuBizIdList)
                        .menuType(Integer.valueOf(MenuTypeEnum.AN.getCode()))
                        .isExcludeMy(false)
                        .build());
                if (!CollectionUtils.isEmpty(anMenuList)) {
                    List<String> menuPermissionList = anMenuList.stream().map(SysMenu::getPermission).collect(Collectors.toList());
                    response.setPermissions(menuPermissionList);
                }
            }
        }
        return Result.success(response);
    }

    /**
     * 权限-管理端-获取当前登录用户绑定的当前项目用户角色菜单列表
     * @return
     */
    @Override
    public Result<List<ApiLoginMenuInfoResponse>> getProjectMenuRouters(String projectBizId) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();

        List<ApiLoginMenuInfoResponse> treeMenu = new ArrayList<>();

        List<RelUserRole> relUserRoleList = iRelUserRoleService.queryList(RelUserRoleDto.builder()
                .tenantBizId(projectBizId)
                .userBizId(authUserDto.getUserBizId())
                .build());
        if (!CollectionUtils.isEmpty(relUserRoleList)) {
            //项目用户角色编码列表信息
            List<String> roleBizIdList = relUserRoleList.stream().map(RelUserRole::getRoleBizId).collect(Collectors.toList());

            //项目用户角色菜单按钮权限列表信息
            List<RelRoleMenu> relRoleMenuList = iRelRoleMenuService.queryList(RelRoleMenuDto.builder()
                    .tenantBizId(projectBizId)
                    .roleBizIdList(roleBizIdList)
                    .build());
            if (!CollectionUtils.isEmpty(relRoleMenuList)) {
                //去重菜单id（多个角色绑定有重复的菜单）
                List<String> menuBizIdList = relRoleMenuList.stream().map(RelRoleMenu::getMenuBizId).distinct().collect(Collectors.toList());

                //项目用户角色菜单列表（树形结构）
                List<SysMenu> sysMenuList = iSysMenuService.queryList(SysMenuDto.builder()
                        .menuBizIdList(menuBizIdList)
                        .isExcludeMy(false)
                        .build());
                if (!CollectionUtils.isEmpty(sysMenuList)){
                    List<ApiLoginMenuInfoResponse> apiLoginMenuInfoResponseList = sysMenuList
                            .stream()
                            .map(dto -> {
                                ApiLoginMenuInfoResponse menuInfoResponse = new ApiLoginMenuInfoResponse();
                                BeanUtils.copyProperties(dto,menuInfoResponse);
                                return menuInfoResponse;
                            }).collect(Collectors.toList());

                    //构建菜单树形结构
                    treeMenu = TreeUtils.buildTree(
                            apiLoginMenuInfoResponseList,
                            ApiLoginMenuInfoResponse::getMenuBizId,         // 获取节点ID
                            ApiLoginMenuInfoResponse::getParentBizId,   // 获取父节点ID
                            (parent, children) -> parent.setChildren(children), // 设置子节点
                            "0"                   // 根节点的父ID值
                    );
                }
            }
        }
        return Result.success(treeMenu);
    }

    /**
     * 权限-管理端-获取当前登录用户是否有权限访问该项目
     * @return
     */
    @Override
    public Result<Boolean> isProjectVisit(String projectBizId) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();

        List<RelProjectUser> relProjectUserList = iRelProjectUserService.queryList(RelProjectUserDto.builder()
                .projectBizId(projectBizId)
                .userBizId(authUserDto.getUserBizId())
                .build());
        if (CollectionUtils.isEmpty(relProjectUserList)) {
            throw new BusinessException("无项目用户权限，请在当前项目添加项目用户权限！");
        }

        List<RelUserRole> relUserRoleList = iRelUserRoleService.queryList(RelUserRoleDto.builder()
                .projectBizId(projectBizId)
                .userBizId(authUserDto.getUserBizId())
                .build());
        if (CollectionUtils.isEmpty(relUserRoleList)) {
            throw new BusinessException("无项目用户角色权限，请在当前项目添加用户角色权限！");
        }

        List<String> roleBizIdList = relUserRoleList.stream().map(RelUserRole::getRoleBizId).collect(Collectors.toList());
        List<RelRoleMenu> relRoleMenuList = iRelRoleMenuService.queryList(RelRoleMenuDto.builder()
                .projectBizId(projectBizId)
                .roleBizIdList(roleBizIdList)
                .build());
        if (CollectionUtils.isEmpty(relRoleMenuList)) {
            throw new BusinessException("无项目角色权限，请在当前项目添加角色菜单权限！");
        }

        return Result.success();
    }

    /**
     * 校验用户是否存在
     * @param userBizId
     * @return
     */
    public Result<SysUser> checkSysUserIsExist(String userBizId) {
        SysUser sysUser = iSysUserService.queryOne(userBizId);
        if (Objects.isNull(sysUser)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(sysUser);
    }
}
