package com.yd.user.service.service.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.user.feign.request.reltenantproject.ApiRelTenantProjectListPageRequest;
import com.yd.user.feign.request.reltenantproject.ApiRelTenantProjectSelectPageRequest;
import com.yd.user.feign.response.reltenantproject.ApiRelTenantProjectListPageResponse;
import com.yd.user.feign.response.reltenantproject.ApiRelTenantProjectSelectPageResponse;
import com.yd.user.service.dto.RelTenantProjectDto;
import com.yd.user.service.model.RelTenantProject;
import com.yd.user.service.dao.RelTenantProjectMapper;
import com.yd.user.service.model.SysProject;
import com.yd.user.service.service.IRelTenantProjectService;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Objects;

/**
 * <p>
 * 租户-项目关系表（正常来说一对多，加关系表方便扩展） 服务实现类
 * </p>
 *
 * @author zxm
 * @since 2025-07-31
 */
@Service
public class RelTenantProjectServiceImpl extends ServiceImpl<RelTenantProjectMapper, RelTenantProject> implements IRelTenantProjectService {

    /**
     * 租户和项目关系分页查询
     * @param request
     * @return
     */
    @Override
    public IPage<ApiRelTenantProjectListPageResponse> page(Page<ApiRelTenantProjectListPageResponse> page,
                                                           ApiRelTenantProjectListPageRequest request) {
        return baseMapper.page(page,request);
    }

    /**
     * 导入-选择项目分页查询
     * @param request
     * @return
     */
    @Override
    public IPage<ApiRelTenantProjectSelectPageResponse> selectProjectPage(Page<ApiRelTenantProjectSelectPageResponse> page,
                                                                          ApiRelTenantProjectSelectPageRequest request) {

        return baseMapper.selectProjectPage(page,request);
    }

    /**
     * 查询单个对象（主键id）
     * @param id
     * @return
     */
    @Override
    public RelTenantProject queryOne(Long id) {
        return this.getOne(new LambdaQueryWrapper<RelTenantProject>().eq(RelTenantProject::getId,id));
    }

    /**
     * 查询单个对象（租户业务id和项目业务id）
     * @param tenantBizId
     * @param projectBizId
     * @return
     */
    @Override
    public RelTenantProject queryOne(String tenantBizId, String projectBizId) {
        return this.getOne(new LambdaQueryWrapper<RelTenantProject>()
                .eq(RelTenantProject::getTenantBizId,tenantBizId)
                .eq(RelTenantProject::getProjectBizId,projectBizId)
        );
    }

    /**
     * 查询租户项目关系列表
     * @param dto
     * @return
     */
    @Override
    public List<RelTenantProject> queryList(RelTenantProjectDto dto) {
        List<RelTenantProject> list = baseMapper.selectList(new LambdaQueryWrapper<RelTenantProject>()
                .eq(!Objects.isNull(dto.getId()),RelTenantProject::getId,dto.getId())
                .eq(StringUtils.isNotBlank(dto.getTenantBizId()),RelTenantProject::getTenantBizId,dto.getTenantBizId())
                .eq(StringUtils.isNotBlank(dto.getProjectBizId()),RelTenantProject::getProjectBizId,dto.getProjectBizId())
                .eq(!Objects.isNull(dto.getStatus()),RelTenantProject::getStatus,dto.getStatus())
        );
        return list;
    }

    /**
     * 根据项目id删除关系
     * @param projectBizId
     * @return
     */
    @Override
    public void delRelByProjectBizId(String projectBizId) {
        this.remove(new LambdaQueryWrapper<RelTenantProject>().eq(RelTenantProject::getProjectBizId,projectBizId));
    }

    /**
     * 根据项目id删除关系（保留传入租户id列表的关系）
     * @param projectBizId
     * @return
     */
    @Override
    public void delRelByProjectBizIdNeTenantBizIdList(String projectBizId, List<String> tenantBizIdList) {
        this.remove(new LambdaQueryWrapper<RelTenantProject>()
                .eq(RelTenantProject::getProjectBizId,projectBizId)
                .notIn(RelTenantProject::getTenantBizId,tenantBizIdList)
        );
    }

    /**
     * 根据租户id删除关系
     * @param tenantBizId
     * @return
     */
    @Override
    public void delRelByTenantBizId(String tenantBizId) {
        this.remove(new LambdaQueryWrapper<RelTenantProject>().eq(RelTenantProject::getTenantBizId,tenantBizId));
    }
}
