package com.yd.user.api.service.impl;

import com.yd.common.constant.RedisConstants;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RedisUtil;
import com.yd.user.api.service.ApiSysDictService;
import com.yd.user.feign.request.sysdict.ApiUpdateRedisDictListRequest;
import com.yd.user.feign.request.sysdict.GetDictTypeListRequest;
import com.yd.user.feign.response.sysdict.ApiUpdateRedisDictListResponse;
import com.yd.user.feign.response.sysdict.GetDictItemListByDictTypeResponse;
import com.yd.user.feign.response.sysdict.GetDictTypeListResponse;
import com.yd.user.service.dto.SysDictItemDto;
import com.yd.user.service.model.SysDict;
import com.yd.user.service.model.SysDictItem;
import com.yd.user.service.service.ISysDictItemService;
import com.yd.user.service.service.ISysDictService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

@Service
public class ApiSysDictServiceImpl implements ApiSysDictService {

    @Autowired
    private ISysDictService iSysDictService;

    @Autowired
    private ISysDictItemService iSysDictItemService;

    @Autowired
    private RedisUtil redisUtil;

    /**
     * 根据字典类型获取字典数据项列表
     * @param dictType
     * @return
     */
    @Override
    public Result<List<GetDictItemListByDictTypeResponse>> getDictItemListByDictType(String dictType) {
        SysDict sysDict = iSysDictService.queryOneByType(dictType);
        if (Objects.isNull(sysDict)) {
            throw new BusinessException("字典类型数据不存在");
        }

        List<GetDictItemListByDictTypeResponse> responses = new ArrayList<>();
        //查询字典数据项列表
        List<SysDictItem> sysDictItemList = iSysDictItemService.queryList(SysDictItemDto.builder()
                .dictBizId(sysDict.getDictBizId())
                .build());
        if (!CollectionUtils.isEmpty(sysDictItemList)) {
            responses = sysDictItemList.stream().map(dto -> {
                GetDictItemListByDictTypeResponse response = new GetDictItemListByDictTypeResponse();
                BeanUtils.copyProperties(dto,response);
                return response;
            }).collect(Collectors.toList());
        }

        return Result.success(responses);
    }

    /**
     * 根据字典类型列表获取字典数据项列表
     * @return
     */
    @Override
    public Result<List<GetDictTypeListResponse>> getByDictTypeList(GetDictTypeListRequest request) {
        List<GetDictTypeListResponse> typeListResponseList = new ArrayList<>();

        // 根据字典类型列表查询字典类型信息
        List<SysDict> sysDictList = iSysDictService.queryList(request.getTypeList());

        if (!CollectionUtils.isEmpty(sysDictList)) {
            // 获取字典类型ID列表
            List<String> dictBizIdList = sysDictList.stream()
                    .map(SysDict::getDictBizId)
                    .collect(Collectors.toList());

            // 查询对应的字典项
            List<SysDictItem> sysDictItemList = iSysDictItemService.queryList(
                    SysDictItemDto.builder()
                            .dictBizIdList(dictBizIdList)
                            .build());

            // 按dictBizId分组字典项
            Map<String, List<SysDictItem>> itemsByDictId = sysDictItemList.stream()
                    .collect(Collectors.groupingBy(SysDictItem::getDictBizId));

            // 构建响应
            typeListResponseList = sysDictList.stream()
                    .map(dict -> {
                        GetDictTypeListResponse response = new GetDictTypeListResponse();
                        BeanUtils.copyProperties(dict, response);

                        // 获取该字典类型对应的字典项并转换
                        List<GetDictItemListByDictTypeResponse> itemResponses =
                                itemsByDictId.getOrDefault(dict.getDictBizId(), new ArrayList<>())
                                        .stream()
                                        .map(item -> {
                                            GetDictItemListByDictTypeResponse itemResponse = new GetDictItemListByDictTypeResponse();
                                            BeanUtils.copyProperties(item, itemResponse);
                                            return itemResponse;
                                        })
                                        .collect(Collectors.toList());

                        response.setDictItemList(itemResponses);
                        return response;
                    })
                    .collect(Collectors.toList());
        }

        return Result.success(typeListResponseList);
    }

    /**
     * 更新redis缓存中的字典列表
     * @param request
     * @return
     */
    @Override
    public Result<ApiUpdateRedisDictListResponse> updateRedisDictList(ApiUpdateRedisDictListRequest request) {
        List<GetDictItemListByDictTypeResponse> responseList = iSysDictService.queryListByTypeList(request.getTypeList());
        if (!CollectionUtils.isEmpty(responseList)) {
            //设置redis缓存
            redisUtil.setCacheObject(RedisConstants.DICT_LIST,responseList);
        }
        return Result.success();
    }


}
