package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.TreeUtils;
import com.yd.user.api.service.ApiRelTenantDeptService;
import com.yd.user.feign.request.reltenantdept.ApiRelTenantDeptImportQueryDeptTreeRequest;
import com.yd.user.feign.request.reltenantdept.ApiRelTenantDeptListAddRequest;
import com.yd.user.feign.request.reltenantdept.ApiRelTenantDeptListPageRequest;
import com.yd.user.feign.request.reltenantdept.ApiRelTenantDeptTreeListRequest;
import com.yd.user.feign.response.reltenantdept.ApiRelTenantDeptImportQueryDeptTreeResponse;
import com.yd.user.feign.response.reltenantdept.ApiRelTenantDeptListPageResponse;
import com.yd.user.feign.response.reltenantdept.ApiRelTenantDeptTreeListResponse;
import com.yd.user.service.dto.RelTenantDeptDto;
import com.yd.user.service.model.RelTenantDept;
import com.yd.user.service.service.IRelTenantDeptService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiRelTenantDeptServiceImpl implements ApiRelTenantDeptService {

    @Autowired
    private IRelTenantDeptService iRelTenantDeptService;

    /**
     * 租户和部门关系分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelTenantDeptListPageResponse>> page(ApiRelTenantDeptListPageRequest request) {
        Page<ApiRelTenantDeptListPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelTenantDeptListPageResponse> iPage = iRelTenantDeptService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 租户和部门关系树形列表查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiRelTenantDeptTreeListResponse>> tree(ApiRelTenantDeptTreeListRequest request) {
        List<ApiRelTenantDeptTreeListResponse> treeMenu = new ArrayList<>();
        //租户和部门关系列表
        List<ApiRelTenantDeptTreeListResponse> queryList = iRelTenantDeptService.treeList(request);

        if (!CollectionUtils.isEmpty(queryList)) {
            //构建部门树形结构
            treeMenu = TreeUtils.buildTree(
                    queryList,
                    ApiRelTenantDeptTreeListResponse::getDeptBizId,         // 获取节点ID
                    ApiRelTenantDeptTreeListResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }
        return Result.success(treeMenu);
    }

    /**
     * 添加租户和部门列表关系
     * @param request
     * @return
     */
    @Override
    public Result addDeptList(ApiRelTenantDeptListAddRequest request) {
        List<RelTenantDept> list = new ArrayList<>();
        //先删后增加
        iRelTenantDeptService.removeByTenantBizId(request.getTenantBizId());

        if (!CollectionUtils.isEmpty(request.getDeptBizIdList())) {
            //部门唯一标识(业务ID) 列表（选中列表）不为空
            list = request.getDeptBizIdList().stream().map(bizId -> {
                RelTenantDept relTenantDept = new RelTenantDept();
                relTenantDept.setTenantBizId(request.getTenantBizId());
                relTenantDept.setDeptBizId(bizId);
                return relTenantDept;
            }).collect(Collectors.toList());

            iRelTenantDeptService.saveOrUpdateBatch(list);
        }

        return Result.success();
    }

    /**
     * 删除租户和部门关系
     * @param id 租户和部门关系表主键id
     * @return
     */
    @Override
    public Result del(Long id) {
        RelTenantDept relTenantDept = iRelTenantDeptService.queryOne(id);
        if (Objects.isNull(relTenantDept)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        iRelTenantDeptService.removeById(relTenantDept.getId());
        return Result.success();
    }

    /**
     * 导入-树形部门查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiRelTenantDeptImportQueryDeptTreeResponse>> importQueryDeptTree(ApiRelTenantDeptImportQueryDeptTreeRequest request) {
        //TODO 条件搜索部门树后续完善
        List<ApiRelTenantDeptImportQueryDeptTreeResponse> treeMenu = new ArrayList<>();
        //查询租户和部门关系数据
        List<ApiRelTenantDeptImportQueryDeptTreeResponse> queryList = iRelTenantDeptService.importQueryDeptTreeList(request);

        if (!CollectionUtils.isEmpty(queryList)) {
            //构建部门树形结构
            treeMenu = TreeUtils.buildTree(
                    queryList,
                    ApiRelTenantDeptImportQueryDeptTreeResponse::getDeptBizId,         // 获取节点ID
                    ApiRelTenantDeptImportQueryDeptTreeResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }

        return Result.success(treeMenu);
    }

    /**
     * 导入-树形部门-查询选中的部门id集合
     * @param tenantBizId
     * @return
     */
    @Override
    public Result<List<String>> selectedList(String tenantBizId) {
        List<String> deptBizIdList = new ArrayList<>();
        List<RelTenantDept> relTenantDeptList = iRelTenantDeptService.queryList(RelTenantDeptDto.builder()
                .tenantBizId(tenantBizId)
                .build());
        if (!CollectionUtils.isEmpty(relTenantDeptList)) {
            deptBizIdList = relTenantDeptList.stream().map(RelTenantDept::getDeptBizId).collect(Collectors.toList());
        }
        return Result.success(deptBizIdList);
    }

    /**
     * 添加单个租户和部门关系
     * @param tenantBizId
     * @param deptBizId
     * @return
     */
    @Override
    public Result addSingleRel(String tenantBizId, String deptBizId) {
        //新增关系
        RelTenantDept relTenantDept = new RelTenantDept();
        relTenantDept.setTenantBizId(tenantBizId);
        relTenantDept.setDeptBizId(deptBizId);
        iRelTenantDeptService.saveOrUpdate(relTenantDept);
        return Result.success();
    }
}
