package com.yd.user.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.TreeUtils;
import com.yd.user.api.service.ApiRelProjectMenuService;
import com.yd.user.feign.request.relprojectmenu.ApiRelProjectMenuImportQueryMenuTreeRequest;
import com.yd.user.feign.request.relprojectmenu.ApiRelProjectMenuListAddRequest;
import com.yd.user.feign.request.relprojectmenu.ApiRelProjectMenuListPageRequest;
import com.yd.user.feign.request.relprojectmenu.ApiRelProjectMenuTreeListRequest;
import com.yd.user.feign.response.relprojectmenu.ApiRelProjectMenuImportQueryMenuTreeResponse;
import com.yd.user.feign.response.relprojectmenu.ApiRelProjectMenuListPageResponse;
import com.yd.user.feign.response.relprojectmenu.ApiRelProjectMenuTreeListResponse;
import com.yd.user.feign.response.reltenantmenu.ApiRelTenantMenuTreeListResponse;
import com.yd.user.service.dto.RelProjectMenuDto;
import com.yd.user.service.model.RelProjectMenu;
import com.yd.user.service.service.IRelProjectMenuService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
public class ApiRelProjectMenuServiceImpl implements ApiRelProjectMenuService {

    @Autowired
    private IRelProjectMenuService iRelProjectMenuService;

    /**
     * 项目和菜单关系分页查询r
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelProjectMenuListPageResponse>> page(ApiRelProjectMenuListPageRequest request) {
        Page<ApiRelProjectMenuListPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelProjectMenuListPageResponse> iPage = iRelProjectMenuService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 项目和菜单关系树形列表查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiRelProjectMenuTreeListResponse>> tree(ApiRelProjectMenuTreeListRequest request) {
        List<ApiRelProjectMenuTreeListResponse> treeMenu = new ArrayList<>();
        //项目和菜单关系列表
        List<ApiRelProjectMenuTreeListResponse> queryList = iRelProjectMenuService.treeList(request);

        if (!CollectionUtils.isEmpty(queryList)) {
            //构建菜单树形结构
            treeMenu = TreeUtils.buildTree(
                    queryList,
                    ApiRelProjectMenuTreeListResponse::getMenuBizId,         // 获取节点ID
                    ApiRelProjectMenuTreeListResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }
        return Result.success(treeMenu);
    }

    /**
     * 添加项目和菜单列表关系
     * @param request
     * @return
     */
    @Override
    public Result addMenuList(ApiRelProjectMenuListAddRequest request) {
        List<RelProjectMenu> list = new ArrayList<>();
        //先删后增加
        iRelProjectMenuService.removeByProjectBizId(request.getProjectBizId());

        if (!CollectionUtils.isEmpty(request.getMenuBizIdList())) {
            //菜单唯一标识(业务ID) 列表（选中列表）不为空
            list = request.getMenuBizIdList().stream().map(bizId -> {
                RelProjectMenu relProjectMenu = new RelProjectMenu();
                relProjectMenu.setTenantBizId(request.getTenantBizId());
                relProjectMenu.setProjectBizId(request.getProjectBizId());
                relProjectMenu.setMenuBizId(bizId);
                return relProjectMenu;
            }).collect(Collectors.toList());

            iRelProjectMenuService.saveOrUpdateBatch(list);
        }

        return Result.success();
    }

    /**
     * 删除项目和菜单关系
     * @param id 项目和菜单关系表主键id
     * @return
     */
    @Override
    public Result del(Long id) {
        RelProjectMenu relProjectMenu = iRelProjectMenuService.queryOne(id);
        if (Objects.isNull(relProjectMenu)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        iRelProjectMenuService.removeById(relProjectMenu.getId());
        return Result.success();
    }

    /**
     * 导入-树形菜单查询
     * @param request
     * @return
     */
    @Override
    public Result<List<ApiRelProjectMenuImportQueryMenuTreeResponse>> importQueryMenuTree(ApiRelProjectMenuImportQueryMenuTreeRequest request) {
        //TODO 条件搜索菜单树后续完善
        List<ApiRelProjectMenuImportQueryMenuTreeResponse> treeMenu = new ArrayList<>();
        //查询项目和菜单关系数据
        List<ApiRelProjectMenuImportQueryMenuTreeResponse> queryList = iRelProjectMenuService.importQueryMenuTreeList(request);

        if (!CollectionUtils.isEmpty(queryList)) {
            //构建菜单树形结构
            treeMenu = TreeUtils.buildTree(
                    queryList,
                    ApiRelProjectMenuImportQueryMenuTreeResponse::getMenuBizId,         // 获取节点ID
                    ApiRelProjectMenuImportQueryMenuTreeResponse::getParentBizId,   // 获取父节点ID
                    (parent, children) -> parent.setChildren(children), // 设置子节点
                    "0"                   // 根节点的父ID值
            );
        }

        return Result.success(treeMenu);
    }

    /**
     * 导入-树形菜单-查询选中的菜单id集合
     * @param projectBizId
     * @return
     */
    @Override
    public Result<List<String>> selectedList(String tenantBizId,String projectBizId) {
        List<String> menuBizIdList = new ArrayList<>();
        List<RelProjectMenu> relProjectMenuList = iRelProjectMenuService.queryList(RelProjectMenuDto.builder()
                .tenantBizId(tenantBizId)
                .projectBizId(projectBizId)
                .build());
        if (!CollectionUtils.isEmpty(relProjectMenuList)) {
            menuBizIdList = relProjectMenuList.stream().map(RelProjectMenu::getMenuBizId).collect(Collectors.toList());
        }
        return Result.success(menuBizIdList);
    }
    
    /**
     * 添加单个项目和菜单关系
     * @param projectBizId
     * @param menuBizId
     * @return
     */
    @Override
    public Result addSingleRel(String projectBizId, String menuBizId) {
        //新增关系
        RelProjectMenu relProjectMenu = new RelProjectMenu();
        relProjectMenu.setProjectBizId(projectBizId);
        relProjectMenu.setMenuBizId(menuBizId);
        iRelProjectMenuService.saveOrUpdate(relProjectMenu);
        return Result.success();
    }
}
