package com.yd.question.service.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

public class CategoryGroupingExample {
    
    // 假设这是您已经定义好的DTO类
    static class ApiRelQuestionsGroupDto {
        private String firstCategory;
        private String secondCategory;
        private String text;
        
        // 构造方法、getter和setter
        public ApiRelQuestionsGroupDto(String firstCategory, String secondCategory, String text) {
            this.firstCategory = firstCategory;
            this.secondCategory = secondCategory;
            this.text = text;
        }
        
        public String getFirstCategory() { return firstCategory; }
        public String getSecondCategory() { return secondCategory; }
        public String getText() { return text; }
    }
    
    // 二级分类DTO
    static class ApiSecondCategoryDto {
        private String secondCategory;
        private String secondCategoryName;
        private Integer sortOrder;
        private List<ApiRelQuestionsGroupDto> questionList;
        
        // getter和setter
        public String getSecondCategory() { return secondCategory; }
        public void setSecondCategory(String secondCategory) { this.secondCategory = secondCategory; }
        public String getSecondCategoryName() { return secondCategoryName; }
        public void setSecondCategoryName(String secondCategoryName) { this.secondCategoryName = secondCategoryName; }
        public Integer getSortOrder() { return sortOrder; }
        public void setSortOrder(Integer sortOrder) { this.sortOrder = sortOrder; }
        public List<ApiRelQuestionsGroupDto> getQuestionList() { return questionList; }
        public void setQuestionList(List<ApiRelQuestionsGroupDto> questionList) { this.questionList = questionList; }
    }
    
    // 一级分类DTO
    static class ApiFirstAndSecondCategoryDto {
        private String firstCategory;
        private String firstCategoryName;
        private Integer sortOrder;
        private List<ApiSecondCategoryDto> secondCategoryDtoList;
        
        // getter和setter
        public String getFirstCategory() { return firstCategory; }
        public void setFirstCategory(String firstCategory) { this.firstCategory = firstCategory; }
        public String getFirstCategoryName() { return firstCategoryName; }
        public void setFirstCategoryName(String firstCategoryName) { this.firstCategoryName = firstCategoryName; }
        public Integer getSortOrder() { return sortOrder; }
        public void setSortOrder(Integer sortOrder) { this.sortOrder = sortOrder; }
        public List<ApiSecondCategoryDto> getSecondCategoryDtoList() { return secondCategoryDtoList; }
        public void setSecondCategoryDtoList(List<ApiSecondCategoryDto> secondCategoryDtoList) { this.secondCategoryDtoList = secondCategoryDtoList; }
    }
    
    public static void main(String[] args) {
        // 创建测试数据
        List<ApiRelQuestionsGroupDto> list = new ArrayList<>();
        list.add(new ApiRelQuestionsGroupDto("投保人", "基本信息", "姓名"));
        list.add(new ApiRelQuestionsGroupDto("投保人", "基本信息", "年龄"));
        list.add(new ApiRelQuestionsGroupDto("投保人", "健康信息", "是否有病史"));
        list.add(new ApiRelQuestionsGroupDto("受保人", "基本信息", "姓名"));
        list.add(new ApiRelQuestionsGroupDto("受保人", "健康信息", "是否有病史"));
        list.add(new ApiRelQuestionsGroupDto("受保人", "健康信息", "是否有过敏"));
        
        // 使用Java 8 Stream API进行多级分组并转换为目标结构
        List<ApiFirstAndSecondCategoryDto> firstAndSecondCategoryDtoList = list.stream()
            // 过滤掉firstCategory为null的记录
            .filter(dto -> dto.getFirstCategory() != null)
            // 按一级分类分组
            .collect(Collectors.groupingBy(
                ApiRelQuestionsGroupDto::getFirstCategory,
                Collectors.toList()
            ))
            .entrySet().stream()
            .map(entry -> {
                // 创建一级分类DTO
                ApiFirstAndSecondCategoryDto firstCategoryDto = new ApiFirstAndSecondCategoryDto();
                firstCategoryDto.setFirstCategory(entry.getKey());
                
                // 设置一级分类名称（这里简单模拟）
                firstCategoryDto.setFirstCategoryName(entry.getKey() + "名称");
                
                // 设置排序序号（这里简单模拟）
                if("投保人".equals(entry.getKey())) {
                    firstCategoryDto.setSortOrder(1);
                } else {
                    firstCategoryDto.setSortOrder(2);
                }
                
                // 对当前一级分类下的数据进行二级分类分组
                Map<String, List<ApiRelQuestionsGroupDto>> secondCategoryMap = entry.getValue().stream()
                    // 过滤掉secondCategory为null的记录
                    .filter(dto -> dto.getSecondCategory() != null)
                    .collect(Collectors.groupingBy(ApiRelQuestionsGroupDto::getSecondCategory));
                
                // 转换为二级分类DTO列表
                List<ApiSecondCategoryDto> secondCategoryDtoList = secondCategoryMap.entrySet().stream()
                    .map(secondEntry -> {
                        ApiSecondCategoryDto secondCategoryDto = new ApiSecondCategoryDto();
                        secondCategoryDto.setSecondCategory(secondEntry.getKey());
                        
                        // 设置二级分类名称（这里简单模拟）
                        secondCategoryDto.setSecondCategoryName(secondEntry.getKey() + "名称");
                        
                        // 设置二级分类排序序号（这里简单模拟）
                        if("基本信息".equals(secondEntry.getKey())) {
                            secondCategoryDto.setSortOrder(1);
                        } else {
                            secondCategoryDto.setSortOrder(2);
                        }
                        
                        // 设置问题列表
                        secondCategoryDto.setQuestionList(secondEntry.getValue());
                        
                        return secondCategoryDto;
                    })
                    .collect(Collectors.toList());
                
                // 设置二级分类列表到一级分类DTO
                firstCategoryDto.setSecondCategoryDtoList(secondCategoryDtoList);
                
                return firstCategoryDto;
            })
            .collect(Collectors.toList());
        
        // 打印结果
        System.out.println("分组结果:");
        for (ApiFirstAndSecondCategoryDto firstCategory : firstAndSecondCategoryDtoList) {
            System.out.println("一级分类: " + firstCategory.getFirstCategory() + 
                              " (" + firstCategory.getFirstCategoryName() + "), 排序: " + firstCategory.getSortOrder());
            
            for (ApiSecondCategoryDto secondCategory : firstCategory.getSecondCategoryDtoList()) {
                System.out.println("  二级分类: " + secondCategory.getSecondCategory() + 
                                  " (" + secondCategory.getSecondCategoryName() + "), 排序: " + secondCategory.getSortOrder());
                
                for (ApiRelQuestionsGroupDto question : secondCategory.getQuestionList()) {
                    System.out.println("    问题: " + question.getText());
                }
            }
            System.out.println();
        }
    }
}
