package com.yd.product.api.service.impl;

import com.yd.base.feign.client.fieldvalue.ApiFieldValueFeignClient;
import com.yd.base.feign.request.fieldvalue.ApiFieldValueListRequest;
import com.yd.base.feign.response.fieldvalue.ApiFieldValueDetailResponse;
import com.yd.common.enums.CommonEnum;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiAttributeSettingService;
import com.yd.product.feign.response.productlaunch.ApiAttributeSettingDto;
import com.yd.product.service.dto.AttributeSettingDto;
import com.yd.product.service.model.AttributeSetting;
import com.yd.product.service.service.IAttributeSettingService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAttributeSettingServiceImpl implements ApiAttributeSettingService {

    @Autowired
    private IAttributeSettingService iAttributeSettingService;

    @Autowired
    private ApiFieldValueFeignClient apiFieldValueFeignClient;

    /**
     * 保存产品属性（参数）配置信息
     * @param apiAttributeSettingDtoList
     * @param productLaunchBizId
     * @return
     */
    @Override
    public Result save(List<ApiAttributeSettingDto> apiAttributeSettingDtoList, String productLaunchBizId) {
        //先删后新增
        iAttributeSettingService.delByProductLaunchBizId(productLaunchBizId);

        //新增
        if (!CollectionUtils.isEmpty(apiAttributeSettingDtoList)) {
            List<AttributeSetting> attributeSettingList = apiAttributeSettingDtoList.stream().map(dto -> {
                AttributeSetting attributeSetting = new AttributeSetting();
                BeanUtils.copyProperties(dto,attributeSetting);
                attributeSetting.setProductLaunchBizId(productLaunchBizId);
                attributeSetting.setAttributeSettingBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ATTRIBUTE_SETTING.getCode()));
                return attributeSetting;
            }).collect(Collectors.toList());
            iAttributeSettingService.saveOrUpdateBatch(attributeSettingList);
        }
        return Result.success();
    }

    /**
     * 根据产品上架信息id查询产品属性（参数）配置列表
     * @param productLaunchBizId
     * @return
     */
    @Override
    public List<ApiAttributeSettingDto> queryAttributeSettingDtoList(String productLaunchBizId) {
        List<ApiAttributeSettingDto> apiAttributeSettingDtoList = new ArrayList<>();
        List<AttributeSetting> attributeSettingList = iAttributeSettingService.queryList(AttributeSettingDto.builder()
                .productLaunchBizId(productLaunchBizId)
                .build());

        if (!CollectionUtils.isEmpty(attributeSettingList)) {
            //收集所有需要批量查询的字段值业务ID
            List<String> allFieldValueBizIds = new ArrayList<>();
            Map<String, List<String>> settingDtoToValueBizIdsMap = new HashMap<>();

            for (int i = 0; i < attributeSettingList.size(); i++) {
                AttributeSetting dto = attributeSettingList.get(i);
                if (StringUtils.isNotBlank(dto.getValue())) {
                    if (dto.getValue().contains(";")) {
                        // 多选情况
                        List<String> fieldValueBizIdList = Arrays.asList(dto.getValue().split(";"));
                        allFieldValueBizIds.addAll(fieldValueBizIdList);
                        settingDtoToValueBizIdsMap.put(String.valueOf(i), fieldValueBizIdList);
                    } else {
                        // 单选情况，也需要查询，因为可能需要获取字段值名称
                        allFieldValueBizIds.add(dto.getValue());
                        settingDtoToValueBizIdsMap.put(String.valueOf(i), Collections.singletonList(dto.getValue()));
                    }
                }
            }

            //批量查询字段值详情
            Map<String, String> fieldValueBizIdToNameMap = new HashMap<>();
            Map<String, String> fieldValueBizIdToCodeMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(allFieldValueBizIds)) {
                // 去重
                List<String> distinctFieldValueBizIds = allFieldValueBizIds.stream()
                        .distinct()
                        .collect(Collectors.toList());

                ApiFieldValueListRequest request = new ApiFieldValueListRequest();
                request.setFieldValueBizIdList(distinctFieldValueBizIds);

                try {
                    Result<List<ApiFieldValueDetailResponse>> result = apiFieldValueFeignClient.list(request);
                    if (result != null && result.getCode() == 200 && !CollectionUtils.isEmpty(result.getData())) {
                        log.info("查询到字段值详情数量: {}", result.getData().size());

                        // 构建映射关系：字段值业务ID -> 字段值名称 和 字段值业务ID -> 字段值编码
                        for (ApiFieldValueDetailResponse response : result.getData()) {
                            log.debug("字段值详情 - fieldValueBizId: {}, fieldValueCode: {}, value: {}",
                                    response.getFieldValueBizId(), response.getFieldValueCode(), response.getValue());

                            fieldValueBizIdToNameMap.put(response.getFieldValueBizId(), response.getValue());

                            // 如果fieldValueCode为空，则使用fieldValueBizId作为备用值
                            String fieldValueCode = StringUtils.isNotBlank(response.getFieldValueCode())
                                    ? response.getFieldValueCode()
                                    : response.getFieldValueBizId();
                            fieldValueBizIdToCodeMap.put(response.getFieldValueBizId(), fieldValueCode);
                        }

                        log.info("字段值编码映射表: {}", fieldValueBizIdToCodeMap);
                    } else {
                        log.warn("查询字段值详情失败，fieldValueBizIdList: {}", distinctFieldValueBizIds);
                    }
                } catch (Exception e) {
                    log.error("调用字段值服务异常", e);
                }
            }

            //构建返回结果
            for (int i = 0; i < attributeSettingList.size(); i++) {
                AttributeSetting dto = attributeSettingList.get(i);
                ApiAttributeSettingDto settingDto = new ApiAttributeSettingDto();
                BeanUtils.copyProperties(dto, settingDto);

                List<String> valueBizIds = settingDtoToValueBizIdsMap.get(String.valueOf(i));
                if (!CollectionUtils.isEmpty(valueBizIds)) {
                    // 构建字段值名称（多个值用分号分隔）
                    Map<String, String> finalFieldValueBizIdToNameMap = fieldValueBizIdToNameMap;
                    List<String> valueNames = valueBizIds.stream()
                            .map(fieldValueBizId -> {
                                String name = finalFieldValueBizIdToNameMap.get(fieldValueBizId);
                                return StringUtils.defaultIfBlank(name, fieldValueBizId);
                            })
                            .collect(Collectors.toList());
                    settingDto.setValueNames(StringUtils.join(valueNames, ";"));

                    // 构建字段值编码（多个值用分号分隔）
                    Map<String, String> finalFieldValueBizIdToCodeMap = fieldValueBizIdToCodeMap;
                    List<String> valueCodes = valueBizIds.stream()
                            .map(fieldValueBizId -> {
                                String code = finalFieldValueBizIdToCodeMap.get(fieldValueBizId);
                                return StringUtils.defaultIfBlank(code, fieldValueBizId);
                            })
                            .collect(Collectors.toList());
                    settingDto.setValueCodes(StringUtils.join(valueCodes, ";"));
                }

                apiAttributeSettingDtoList.add(settingDto);
            }
        }
        return apiAttributeSettingDtoList;
    }
}