package com.yd.product.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiSpeciesSettingService;
import com.yd.product.feign.dto.ApiSpeciesSettingDto;
import com.yd.product.feign.dto.ApiSpeciesTypeDto;
import com.yd.product.service.dto.SpeciesSettingDto;
import com.yd.product.service.model.SpeciesSetting;
import com.yd.product.service.service.ISpeciesSettingService;
import com.yd.user.feign.response.mdcountry.ApiMdCountryVO;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiSpeciesSettingServiceImpl implements ApiSpeciesSettingService {

    @Autowired
    private ISpeciesSettingService iSpeciesSettingService;

    /**
     * 保存产品上架规格配置列表
     * @param apiSpeciesSettingDtoList
     * @param productLaunchBizId
     * @return
     */
    @Override
    public Result save(List<ApiSpeciesSettingDto> apiSpeciesSettingDtoList,
                       String productLaunchBizId) {
        //先删后新增
        iSpeciesSettingService.delByProductLaunchBizId(productLaunchBizId);

        //新增
        if (!CollectionUtils.isEmpty(apiSpeciesSettingDtoList)) {
            List<SpeciesSetting> speciesSettingList= apiSpeciesSettingDtoList.stream().map(dto -> {
                SpeciesSetting speciesSetting = new SpeciesSetting();
                BeanUtils.copyProperties(dto,speciesSetting);
                //产品上架信息表唯一业务ID
                speciesSetting.setProductLaunchBizId(productLaunchBizId);
                //产品上架规格配置表唯一业务ID
                speciesSetting.setSpeciesSettingBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_SPECIES_SETTING.getCode()));
                return speciesSetting;
            }).collect(Collectors.toList());
            iSpeciesSettingService.saveOrUpdateBatch(speciesSettingList);
        }
        return Result.success();
    }

    /**
     * 查询产品上架规格类型分组列表
     * @param productLaunchBizId
     * @return
     */
    @Override
    public List<ApiSpeciesTypeDto> querySpeciesTypeDtoList(String productLaunchBizId) {
        List<ApiSpeciesTypeDto> apiSpeciesTypeDtoList = new ArrayList<>();
        List<SpeciesSetting> speciesSettingList = iSpeciesSettingService
                .queryList(SpeciesSettingDto.builder().productLaunchBizId(productLaunchBizId).build());

        if (!CollectionUtils.isEmpty(speciesSettingList)) {
            // 首先按ID排序
            List<SpeciesSetting> sortedList = speciesSettingList.stream()
                    .sorted(Comparator.comparing(SpeciesSetting::getId))
                    .collect(Collectors.toList());

            // 使用LinkedHashMap保持插入顺序，按规格类型分组
            Map<String, List<SpeciesSetting>> speciesSettingByType = new LinkedHashMap<>();

            for (SpeciesSetting setting : sortedList) {
                String typeName = setting.getTypeName();
                speciesSettingByType.computeIfAbsent(typeName, k -> new ArrayList<>()).add(setting);
            }

            // 为每个规格类型创建DTO
            apiSpeciesTypeDtoList = speciesSettingByType.entrySet().stream().map(entry -> {
                List<SpeciesSetting> settingsInType = entry.getValue();
                if (CollectionUtils.isEmpty(settingsInType)) {
                    return null;
                }

                // 取第一个设置项获取规格类型信息（同一类型下这些信息应该相同）
                SpeciesSetting firstSetting = settingsInType.get(0);

                ApiSpeciesTypeDto typeDto = new ApiSpeciesTypeDto();
                typeDto.setSpeciesTypeBizId(firstSetting.getSpeciesTypeBizId());
                typeDto.setTypeCode(firstSetting.getTypeCode());
                typeDto.setTypeName(firstSetting.getTypeName());

                // 转换该类型下的所有规格配置
                List<ApiSpeciesSettingDto> speciesSettingDtos = settingsInType.stream().map(setting -> {
                    ApiSpeciesSettingDto speciesSettingDto = new ApiSpeciesSettingDto();
                    BeanUtils.copyProperties(setting, speciesSettingDto);
                    return speciesSettingDto;
                }).collect(Collectors.toList());

                typeDto.setApiSpeciesSettingDtoList(speciesSettingDtos);
                return typeDto;
            }).filter(Objects::nonNull).collect(Collectors.toList());
        }
        return apiSpeciesTypeDtoList;
    }

    /**
     * 根据规格类型编码获取apiSpeciesSettingDtoList中的规格值
     * @param apiSpeciesSettingDtoList
     * @param typeCode
     * @return
     */
    @Override
    public String getSpeciesValue(List<ApiSpeciesSettingDto> apiSpeciesSettingDtoList,String typeCode) {
        String speciesValue = "";
        if (!CollectionUtils.isEmpty(apiSpeciesSettingDtoList)){
            List<ApiSpeciesSettingDto> dtoList = apiSpeciesSettingDtoList
                    .stream()
                    .filter(d -> d.getTypeCode().equals(typeCode))
                    .collect(Collectors.toList());
            if (!CollectionUtils.isEmpty(dtoList)) {
                ApiSpeciesSettingDto settingDto = dtoList.get(0);
                speciesValue = settingDto.getValue();
            }
        }
        return speciesValue;
    }
}
