package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.base.feign.enums.SpeciesTypeEnum;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiAnnouncementSpeciesService;
import com.yd.product.api.service.ApiProductLaunchService;
import com.yd.product.api.service.ApiSpeciesPriceService;
import com.yd.product.api.service.ApiSpeciesSettingService;
import com.yd.product.feign.dto.ApiSpeciesPriceDto;
import com.yd.product.feign.request.announcementspecies.ApiAnnouncementSpeciesImportRequest;
import com.yd.product.feign.request.announcementspecies.ApiAnnouncementSpeciesPageRequest;
import com.yd.product.feign.response.announcementspecies.ApiAnnouncementSpeciesPageResponse;
import com.yd.product.service.model.AnnouncementSpecies;
import com.yd.product.service.service.IAnnouncementSpeciesService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAnnouncementSpeciesServiceImpl implements ApiAnnouncementSpeciesService {

    @Autowired
    private IAnnouncementSpeciesService iAnnouncementSpeciesService;

    @Autowired
    private ApiProductLaunchService apiProductLaunchService;

    @Autowired
    private ApiSpeciesPriceService apiSpeciesPriceService;

    @Autowired
    private ApiSpeciesSettingService apiSpeciesSettingService;

    /**
     * 分页列表查询-公告佣比率规格信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiAnnouncementSpeciesPageResponse>> page(ApiAnnouncementSpeciesPageRequest request) {
        Page<ApiAnnouncementSpeciesPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiAnnouncementSpeciesPageResponse> iPage = iAnnouncementSpeciesService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 导入规格数据
     * @param request
     * @return
     */
    @Override
    public Result importSpecies(ApiAnnouncementSpeciesImportRequest request) {
        //校验产品上架信息是否存在
        apiProductLaunchService.checkProductLaunchIsExist(request.getProductLaunchBizId());
        //查询规格价格配置列表
        List<ApiSpeciesPriceDto> apiSpeciesPriceDtoList = apiSpeciesPriceService.querySpeciesPriceDtoList(request.getProductLaunchBizId());
        if (CollectionUtils.isEmpty(apiSpeciesPriceDtoList)) {
            throw new BusinessException("导入的数据不存在");
        }
        //查询规格价格配置列表 -> 新增公告佣比率规格数据
        List<AnnouncementSpecies> saveList = apiSpeciesPriceDtoList.stream().map(dto -> {
            AnnouncementSpecies species = new AnnouncementSpecies();
            //规格条件列表JSON串
            species.setSpeciesJson(dto.getSpeciesJson());
            //公告佣比率规格表唯一业务ID
            species.setAnnouncementSpeciesBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ANNOUNCEMENT_SPECIES.getCode()));
            //供款年期（5、10、20年期等）(固定规格条件)
            species.setPaymentTerm(apiSpeciesSettingService.getSpeciesValue(dto.getApiSpeciesSettingDtoList(), SpeciesTypeEnum.PAYMENT_TERM.getItemValue()));
            species.setProductLaunchBizId(request.getProductLaunchBizId());
            //保障计划（0-64岁等）(固定规格条件)
            species.setProtectionPlan(apiSpeciesSettingService.getSpeciesValue(dto.getApiSpeciesSettingDtoList(), SpeciesTypeEnum.PROTECTION_PLAN.getItemValue()));
            return species;
        }).collect(Collectors.toList());
        iAnnouncementSpeciesService.saveOrUpdateBatch(saveList);
        return Result.success();
    }

    /**
     * 校验公告佣比率规格信息是否存在
     * @param announcementSpeciesBizId
     * @return
     */
    @Override
    public Result<AnnouncementSpecies> checkAnnouncementSpeciesIsExist(String announcementSpeciesBizId) {
        AnnouncementSpecies announcementSpecies = iAnnouncementSpeciesService.queryOne(announcementSpeciesBizId);
        if (Objects.isNull(announcementSpecies)) {
            //数据不存在
            throw new BusinessException("公告佣比率规格信息不存在");
        }
        return Result.success(announcementSpecies);
    }
}
