package com.yd.product.api.service.impl;

import com.yd.base.feign.client.fieldvalue.ApiFieldValueFeignClient;
import com.yd.base.feign.request.fieldvalue.ApiFieldValueListRequest;
import com.yd.base.feign.response.fieldvalue.ApiFieldValueDetailResponse;
import com.yd.common.enums.CommonEnum;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiAttributeSettingService;
import com.yd.product.feign.response.productlaunch.ApiAttributeSettingDto;
import com.yd.product.service.dto.AttributeSettingDto;
import com.yd.product.service.model.AttributeSetting;
import com.yd.product.service.service.IAttributeSettingService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiAttributeSettingServiceImpl implements ApiAttributeSettingService {

    @Autowired
    private IAttributeSettingService iAttributeSettingService;

    @Autowired
    private ApiFieldValueFeignClient apiFieldValueFeignClient;

    /**
     * 保存产品属性（参数）配置信息
     * @param apiAttributeSettingDtoList
     * @param productLaunchBizId
     * @return
     */
    @Override
    public Result save(List<ApiAttributeSettingDto> apiAttributeSettingDtoList, String productLaunchBizId) {
        //先删后新增
        iAttributeSettingService.delByProductLaunchBizId(productLaunchBizId);

        //新增
        if (!CollectionUtils.isEmpty(apiAttributeSettingDtoList)) {
            List<AttributeSetting> attributeSettingList = apiAttributeSettingDtoList.stream().map(dto -> {
                AttributeSetting attributeSetting = new AttributeSetting();
                BeanUtils.copyProperties(dto,attributeSetting);
                attributeSetting.setProductLaunchBizId(productLaunchBizId);
                attributeSetting.setAttributeSettingBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_ATTRIBUTE_SETTING.getCode()));
                return attributeSetting;
            }).collect(Collectors.toList());
            iAttributeSettingService.saveOrUpdateBatch(attributeSettingList);
        }
        return Result.success();
    }

    /**
     * 根据产品上架信息id查询产品属性（参数）配置列表
     * @param productLaunchBizId
     * @return
     */
    @Override
    public List<ApiAttributeSettingDto> queryAttributeSettingDtoList(String productLaunchBizId) {
        List<ApiAttributeSettingDto> apiAttributeSettingDtoList = new ArrayList<>();
        List<AttributeSetting> attributeSettingList = iAttributeSettingService.queryList(AttributeSettingDto.builder()
                .productLaunchBizId(productLaunchBizId)
                .build());

        if (!CollectionUtils.isEmpty(attributeSettingList)) {
            //收集所有需要批量查询的字段值业务ID
            List<String> allFieldValueBizIds = new ArrayList<>();
            Map<String, List<String>> settingDtoToValueBizIdsMap = new HashMap<>();

            for (int i = 0; i < attributeSettingList.size(); i++) {
                AttributeSetting dto = attributeSettingList.get(i);
                List<String> fieldValueBizIds = new ArrayList<>();

                // 收集fieldValueBizId中的业务ID
                if (StringUtils.isNotBlank(dto.getFieldValueBizId())) {
                    if (dto.getFieldValueBizId().contains(";")) {
                        // 多选情况，多个字段值业务ID用分号分隔
                        fieldValueBizIds.addAll(Arrays.asList(dto.getFieldValueBizId().split(";")));
                    } else {
                        // 单选情况
                        fieldValueBizIds.add(dto.getFieldValueBizId());
                    }
                }
                // 对于fieldValueBizId为空的情况，但value字段可能包含业务ID
                else if (StringUtils.isNotBlank(dto.getValue())) {
                    String value = dto.getValue();
                    // 尝试从value字段解析可能的业务ID
                    List<String> possibleBizIds = parsePossibleBizIdsFromValue(value);
                    if (!possibleBizIds.isEmpty()) {
                        fieldValueBizIds.addAll(possibleBizIds);
                    }
                }

                if (!fieldValueBizIds.isEmpty()) {
                    allFieldValueBizIds.addAll(fieldValueBizIds);
                    settingDtoToValueBizIdsMap.put(String.valueOf(i), fieldValueBizIds);
                }
            }

            //批量查询字段值详情
            Map<String, String> fieldValueBizIdToNameMap = new HashMap<>();
            Map<String, String> fieldValueBizIdToCodeMap = new HashMap<>();
            if (!CollectionUtils.isEmpty(allFieldValueBizIds)) {
                // 去重
                List<String> distinctFieldValueBizIds = allFieldValueBizIds.stream()
                        .distinct()
                        .collect(Collectors.toList());

                ApiFieldValueListRequest request = new ApiFieldValueListRequest();
                request.setFieldValueBizIdList(distinctFieldValueBizIds);

                try {
                    Result<List<ApiFieldValueDetailResponse>> result = apiFieldValueFeignClient.list(request);
                    if (result != null && result.getCode() == 200 && !CollectionUtils.isEmpty(result.getData())) {
                        log.info("查询到字段值详情数量: {}", result.getData().size());

                        // 构建映射关系：字段值业务ID -> 字段值名称 和 字段值业务ID -> 字段值编码
                        for (ApiFieldValueDetailResponse response : result.getData()) {
                            log.debug("字段值详情 - fieldValueBizId: {}, fieldValueCode: {}, value: {}",
                                    response.getFieldValueBizId(), response.getFieldValueCode(), response.getValue());

                            fieldValueBizIdToNameMap.put(response.getFieldValueBizId(), response.getValue());

                            // 如果fieldValueCode不为空，则放入映射表
                            if (StringUtils.isNotBlank(response.getFieldValueCode())) {
                                fieldValueBizIdToCodeMap.put(response.getFieldValueBizId(), response.getFieldValueCode());
                            }
                        }

                        log.info("字段值编码映射表: {}", fieldValueBizIdToCodeMap);
                    } else {
                        log.warn("查询字段值详情失败，fieldValueBizIdList: {}", distinctFieldValueBizIds);
                    }
                } catch (Exception e) {
                    log.error("调用字段值服务异常", e);
                }
            }

            //构建返回结果
            for (int i = 0; i < attributeSettingList.size(); i++) {
                AttributeSetting dto = attributeSettingList.get(i);
                ApiAttributeSettingDto settingDto = new ApiAttributeSettingDto();
                BeanUtils.copyProperties(dto, settingDto);

                List<String> valueBizIds = settingDtoToValueBizIdsMap.get(String.valueOf(i));
                if (!CollectionUtils.isEmpty(valueBizIds)) {
                    // 构建字段值名称（多个值用分号分隔）
                    Map<String, String> finalFieldValueBizIdToNameMap = fieldValueBizIdToNameMap;
                    List<String> valueNames = valueBizIds.stream()
                            .map(fieldValueBizId -> {
                                String name = finalFieldValueBizIdToNameMap.get(fieldValueBizId);
                                // 如果查询不到名称，使用业务ID本身
                                return StringUtils.isNotBlank(name) ? name : fieldValueBizId;
                            })
                            .collect(Collectors.toList());
                    settingDto.setValueNames(StringUtils.join(valueNames, ";"));

                    // 构建字段值编码（多个值用分号分隔）
                    Map<String, String> finalFieldValueBizIdToCodeMap = fieldValueBizIdToCodeMap;
                    List<String> valueCodes = valueBizIds.stream()
                            .map(fieldValueBizId -> {
                                String code = finalFieldValueBizIdToCodeMap.get(fieldValueBizId);
                                // 如果查询不到字段值编码，就塞空字符串
                                return StringUtils.isNotBlank(code) ? code : "";
                            })
                            .collect(Collectors.toList());
                    settingDto.setValueCodes(StringUtils.join(valueCodes, ";"));
                } else {
                    // 如果没有字段值业务ID，则使用原始值
                    settingDto.setValueNames(dto.getValue());
                    settingDto.setValueCodes("");
                }

                apiAttributeSettingDtoList.add(settingDto);
            }
        }
        return apiAttributeSettingDtoList;
    }

    /**
     * 从value字段解析可能的业务ID
     * 这个方法尝试识别value字段中是否包含业务ID
     */
    private List<String> parsePossibleBizIdsFromValue(String value) {
        List<String> bizIds = new ArrayList<>();

        if (StringUtils.isBlank(value)) {
            return bizIds;
        }

        // 如果value包含分号，按分号分割
        if (value.contains(";")) {
            String[] parts = value.split(";");
            for (String part : parts) {
                String trimmedPart = part.trim();
                // 如果部分看起来像是业务ID（包含下划线且长度较长）
                if (isLikelyBizId(trimmedPart)) {
                    bizIds.add(trimmedPart);
                }
            }
        }
        // 如果不包含分号，检查整个value
        else if (isLikelyBizId(value)) {
            bizIds.add(value);
        }

        return bizIds;
    }

    /**
     * 判断一个字符串是否可能是业务ID
     * 这是一个保守的判断，只检查基本特征
     */
    private boolean isLikelyBizId(String str) {
        if (StringUtils.isBlank(str)) {
            return false;
        }

        // 业务ID通常包含下划线，且长度较长
        // 这是一个简单的判断，可以根据实际情况调整
        return str.contains("_") && str.length() > 10;
    }
}