package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.constant.CodeGeneratorConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.CodeGenerator;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiProductService;
import com.yd.product.feign.request.product.ApiProductAddRequest;
import com.yd.product.feign.request.product.ApiProductEditRequest;
import com.yd.product.feign.request.product.ApiProductPageRequest;
import com.yd.product.feign.response.product.ApiProductDetailResponse;
import com.yd.product.feign.response.product.ApiProductPageResponse;
import com.yd.product.service.dto.ProductDto;
import com.yd.product.service.model.Product;
import com.yd.product.service.model.RelProjectProduct;
import com.yd.product.service.model.RelTenantProduct;
import com.yd.product.service.service.IProductService;
import com.yd.product.service.service.IRelProjectProductService;
import com.yd.product.service.service.IRelTenantProductService;
import com.yd.user.feign.client.reltenantproject.ApiRelTenantProjectFeignClient;
import com.yd.user.feign.request.reltenantproject.QueryRelTenantProjectListRequest;
import com.yd.user.feign.response.reltenantproject.QueryRelTenantProjectListResponse;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiProductServiceImpl implements ApiProductService {

    @Autowired
    private IProductService iProductService;

    @Autowired
    private IRelTenantProductService iRelTenantProductService;

    @Autowired
    private IRelProjectProductService iRelProjectProductService;

    @Autowired
    private ApiRelTenantProjectFeignClient apiRelTenantProjectFeignClient;

    /**
     * 分页列表查询-产品基础信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiProductPageResponse>> page(ApiProductPageRequest request) {
        Page<ApiProductPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiProductPageResponse> iPage = iProductService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-产品基础信息
     * @param request
     * @return
     */
    @Override
    public Result<String> add(ApiProductAddRequest request) {
        //校验产品名称的唯一性
        List<Product> productList = iProductService.queryList(ProductDto.builder()
                .productName(request.getProductName())
                .build());
        if (!CollectionUtils.isEmpty(productList)) {
            throw new BusinessException("产品名称已存在");
        }
        Product product = new Product();
        BeanUtils.copyProperties(request,product);
        product.setProductBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PRODUCT.getCode()));
        product.setProductCode(CodeGenerator.generate10Code(CodeGeneratorConstants.PREFIX_PRODUCT_CODE));
        iProductService.saveOrUpdate(product);
        return Result.success(product.getProductBizId());
    }

    /**
     * 编辑-产品基础信息
     * @param request
     * @return
     */
    @Override
    public Result<String> edit(ApiProductEditRequest request) {
        //校验产品信息是否存在
        Result<Product> productResult = checkProductIsExist(request.getProductBizId());
        Product product = productResult.getData();

        //校验产品名称的唯一性
        List<Product> productList = iProductService.queryList(ProductDto.builder()
                .productName(request.getProductName())
                .productBizId(request.getProductBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(productList)) {
            throw new BusinessException("产品名称已存在");
        }
        BeanUtils.copyProperties(request,product);
        iProductService.saveOrUpdate(product);

        return Result.success(product.getProductBizId());
    }

    /**
     * 详情-产品基础信息
     * @param productBizId 产品基础信息表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiProductDetailResponse> detail(String productBizId) {
        //校验产品信息是否存在
        Result<Product> productResult = checkProductIsExist(productBizId);
        Product product = productResult.getData();

        ApiProductDetailResponse response = new ApiProductDetailResponse();
        BeanUtils.copyProperties(product,response);
        return Result.success(response);
    }

    /**
     * 删除-产品基础信息
     * @param productBizId 产品基础信息表唯一业务ID
     * @return
     */
    @Override
    public Result del(String productBizId) {
        return null;
    }

    /**
     * 根据项目业务ID列表查询租户项目关系列表
     * @return
     */
    public Result<List<QueryRelTenantProjectListResponse>> queryRelTenantProjectList(List<String> projectBizIdList) {
        QueryRelTenantProjectListRequest request = new QueryRelTenantProjectListRequest();
        request.setProjectBizIdList(projectBizIdList);
        return apiRelTenantProjectFeignClient.queryRelTenantProjectList(request);
    }

    /**
     * 保存租户产品关系列表
     * @return
     */
    public Result saveRelTenantProductList(List<QueryRelTenantProjectListResponse> responses,String productBizId) {
        //先删后新增
        //根据产品业务ID删除租户产品关系
        iRelTenantProductService.delByProductBizId(productBizId);

        //新增租户产品关系
        if (!CollectionUtils.isEmpty(responses)) {
            List<RelTenantProduct> saveList = responses.stream().map(dto -> {
                RelTenantProduct relTenantProduct = new RelTenantProduct();
                relTenantProduct.setProductBizId(productBizId);
                relTenantProduct.setTenantBizId(dto.getTenantBizId());
                return relTenantProduct;
            }).collect(Collectors.toList());
            iRelTenantProductService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 保存项目产品关系列表
     * @return
     */
    public Result saveRelProjectProductList(List<QueryRelTenantProjectListResponse> responses,String productBizId) {
        //先删后新增
        //根据产品业务ID删除项目产品关系
        iRelProjectProductService.delByProductBizId(productBizId);

        //新增项目产品关系
        if (!CollectionUtils.isEmpty(responses)) {
            List<RelProjectProduct> saveList = responses.stream().map(dto -> {
                RelProjectProduct relProjectProduct = new RelProjectProduct();
                relProjectProduct.setProductBizId(productBizId);
                relProjectProduct.setTenantBizId(dto.getTenantBizId());
                relProjectProduct.setProjectBizId(dto.getProjectBizId());
                return relProjectProduct;
            }).collect(Collectors.toList());
            iRelProjectProductService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 校验产品信息是否存在
     * @param productBizId
     * @return
     */
    @Override
    public Result<Product> checkProductIsExist(String productBizId) {
        Product product = iProductService.queryOne(productBizId);
        if (Objects.isNull(product)) {
            //数据不存在
            throw new BusinessException("产品信息数据不存在");
        }
        return Result.success(product);
    }
}
