package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.base.feign.client.field.ApiFieldFeignClient;
import com.yd.base.feign.client.relobjectcategory.ApiRelObjectCategoryFeignClient;
import com.yd.base.feign.request.field.ApiFieldListRequest;
import com.yd.base.feign.request.relobjectcategory.ApiRelObjectCategoryListAddRequest;
import com.yd.base.feign.request.relobjectcategory.ApiRelObjectCategoryQueryRequest;
import com.yd.base.feign.response.field.ApiFieldDetailResponse;
import com.yd.base.feign.response.relobjectcategory.ApiRelObjectCategoryQueryResponse;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.NoYesEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.*;
import com.yd.product.feign.dto.ApiProductLaunchDto;
import com.yd.product.feign.dto.ApiProjectDto;
import com.yd.product.feign.dto.ApiSpeciesPriceDto;
import com.yd.product.feign.dto.ApiSpeciesTypeDto;
import com.yd.product.feign.enums.ProductCommonEnum;
import com.yd.product.feign.request.product.ApiProductAddRequest;
import com.yd.product.feign.request.product.ApiProductEditRequest;
import com.yd.product.feign.request.productlaunch.*;
import com.yd.product.feign.response.productlaunch.ApiAttributeSettingDto;
import com.yd.product.feign.response.productlaunch.ApiProductLaunchCategoryAddResponse;
import com.yd.product.feign.response.productlaunch.ApiProductLaunchDetailResponse;
import com.yd.product.feign.response.productlaunch.ApiProductLaunchPageResponse;
import com.yd.product.service.model.*;
import com.yd.product.service.service.*;
import com.yd.user.feign.client.reltenantproject.ApiRelTenantProjectFeignClient;
import com.yd.user.feign.client.sysproject.ApiSysProjectFeignClient;
import com.yd.user.feign.request.reltenantproject.QueryRelTenantProjectListRequest;
import com.yd.user.feign.request.sysproject.ApiSysProjectListRequest;
import com.yd.user.feign.response.reltenantproject.QueryRelTenantProjectListResponse;
import com.yd.user.feign.response.sysproject.ApiSysProjectDetailResponse;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiProductLaunchServiceImpl implements ApiProductLaunchService {

    @Autowired
    private IProductLaunchService iProductLaunchService;

    @Autowired
    private ApiProductService apiProductService;

    @Autowired
    private IProductService iProductService;

    @Autowired
    private ApiAttributeSettingService apiAttributeSettingService;

    @Autowired
    private ApiSpeciesSettingService apiSpeciesSettingService;

    @Autowired
    private ApiSpeciesPriceService apiSpeciesPriceService;

    @Autowired
    private IRelTenantProductLaunchService iRelTenantProductLaunchService;

    @Autowired
    private IRelProjectProductLaunchService iRelProjectProductLaunchService;

    @Autowired
    private ApiRelObjectCategoryFeignClient apiRelObjectCategoryFeignClient;

    @Autowired
    private ApiRelTenantProjectFeignClient apiRelTenantProjectFeignClient;

    @Autowired
    private ApiFieldFeignClient apiFieldFeignClient;

    @Autowired
    private ApiSysProjectFeignClient apiSysProjectFeignClient;

    @Override
    public Result<IPage<ApiProductLaunchPageResponse>> page(ApiProductLaunchPageRequest request) {
        Page<ApiProductLaunchPageResponse> page = new Page<ApiProductLaunchPageResponse>(request.getPageNo(), request.getPageSize());
        IPage<ApiProductLaunchPageResponse> iPage = iProductLaunchService.page(page, request);
        return Result.success(iPage);
    }
    /**
     * 添加-产品上架类目信息
     * @param request
     * @return
     */
    @Override
    public Result<ApiProductLaunchCategoryAddResponse> addCategory(ApiProductLaunchCategoryAddRequest request) {
        ApiProductLaunchCategoryAddResponse response = new ApiProductLaunchCategoryAddResponse();
        if (StringUtils.isNotBlank(request.getProductBizId())) {
            //产品唯一业务ID不为空，校验产品信息是否存在
            Product product = iProductService.queryOne(request.getProductBizId());
            if (Objects.isNull(product)) {
                //数据不存在
                throw new BusinessException("产品信息数据不存在");
            }
        }
        String productBizId = "";
        //保存产品基础信息
        if (StringUtils.isBlank(request.getProductBizId())) {
            //新增产品基础信息
            Result<String> result = apiProductService.add(ApiProductAddRequest.builder()
                    .productName(request.getProductName())
                    .build());
            productBizId = result.getData();
        }else {
            //编辑产品基础信息
            Result<String> result = apiProductService.edit(ApiProductEditRequest.builder()
                    .productName(request.getProductName())
                    .productBizId(request.getProductBizId())
                    .build());
            productBizId = result.getData();
        }

        //添加产品上架信息
        ProductLaunch productLaunch = new ProductLaunch();
        productLaunch.setTitle(request.getProductName());
        productLaunch.setMainUrls(String.join(";",request.getMainUrlsList()));
        productLaunch.setProductBizId(productBizId);
        //产品上架信息表唯一业务ID
        productLaunch.setProductLaunchBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PRODUCT_LAUNCH.getCode()));
        //待审核
        productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_DSH.getItemValue());
        iProductLaunchService.saveOrUpdate(productLaunch);
        //添加产品上架对象分类关系信息
        ApiRelObjectCategoryListAddRequest addRequest = new ApiRelObjectCategoryListAddRequest();
        addRequest.setCategoryBizIdList(request.getCategoryBizIdList());
        addRequest.setObjectBizId(productLaunch.getProductLaunchBizId());
        addRequest.setObjectName(productLaunch.getTitle());
        addRequest.setObjectTableName(CommonEnum.UID_TYPE_PRODUCT_LAUNCH.getCode());
        addRequest.setObjectType(CommonEnum.UID_TYPE_PRODUCT_LAUNCH.getCode());
        apiRelObjectCategoryFeignClient.addRelObjectCategoryList(addRequest);

        response.setId(productLaunch.getId());
        response.setProductBizId(productBizId);
        response.setProductLaunchBizId(productLaunch.getProductLaunchBizId());
        return Result.success(response);
    }

    /**
     * 详情-产品上架信息
     * @param productLaunchBizId 产品上架信息表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiProductLaunchDetailResponse> detail(String productLaunchBizId) {
        ApiProductLaunchDetailResponse response = new ApiProductLaunchDetailResponse();
        ProductLaunch productLaunch = iProductLaunchService.queryOne(productLaunchBizId);
        if (Objects.isNull(productLaunch)) {
            throw new BusinessException("产品上架信息不存在");
        }
        //产品上架主信息
        ApiProductLaunchDto apiProductLaunchDto = new ApiProductLaunchDto();
        BeanUtils.copyProperties(productLaunch,apiProductLaunchDto);
        //详情图列表
        apiProductLaunchDto.setDetailUrlsList(StringUtils.isNotBlank(productLaunch.getDetailUrls()) ? Arrays.asList(productLaunch.getDetailUrls().split(";").clone()) : new ArrayList<>());
        //主图url列表
        apiProductLaunchDto.setMainUrlsList(StringUtils.isNotBlank(productLaunch.getMainUrls()) ? Arrays.asList(productLaunch.getMainUrls().split(";").clone()) : new ArrayList<>());

        //获取绑定的分类ID列表
        ApiRelObjectCategoryQueryRequest request = new ApiRelObjectCategoryQueryRequest();
        List<String> objectBizIdList = new ArrayList<>();
        objectBizIdList.add(productLaunchBizId);
        request.setObjectBizIdList(objectBizIdList);
        Result<List<ApiRelObjectCategoryQueryResponse>> listResult = apiRelObjectCategoryFeignClient.query(request);
        if (!CollectionUtils.isEmpty(listResult.getData())) {
            List<String> categoryBizIdList = listResult.getData().stream().map(ApiRelObjectCategoryQueryResponse::getCategoryBizId).collect(Collectors.toList());
            apiProductLaunchDto.setCategoryBizIdList(categoryBizIdList);
        }
        apiProductLaunchDto.setApiProjectDtoList(apiProjectDtoList(productLaunch.getProjectBizIdList()));
        apiProductLaunchDto.setProjectBizIdList(StringUtils.isNotBlank(productLaunch.getProjectBizIdList()) ? Arrays.asList(productLaunch.getProjectBizIdList().split(";").clone()) : new ArrayList<>());

        if (StringUtils.isNotBlank(productLaunch.getProductBizId())) {
            Product product = iProductService.queryOne(productLaunch.getProductBizId());
            apiProductLaunchDto.setTitle(!Objects.isNull(product) ? product.getProductName() : "");
        }
        response.setApiProductLaunchDto(apiProductLaunchDto);

        //设置的产品上架信息的参数列表
        List<ApiAttributeSettingDto> apiAttributeSettingDtoList = apiAttributeSettingService.queryAttributeSettingDtoList(productLaunchBizId);
        if (!CollectionUtils.isEmpty(apiAttributeSettingDtoList)) {
            List<String> fieldBizIdList = apiAttributeSettingDtoList.stream()
                    .map(ApiAttributeSettingDto::getFieldBizId).collect(Collectors.toList());
            ApiFieldListRequest apiFieldListRequest = new ApiFieldListRequest();
            apiFieldListRequest.setFieldBizIdList(fieldBizIdList);
            Result<List<ApiFieldDetailResponse>> result = apiFieldFeignClient.list(apiFieldListRequest);
            List<ApiFieldDetailResponse> detailResponses = result.getData();
            apiAttributeSettingDtoList.stream().map(dto -> {
                if (CollectionUtils.isEmpty(detailResponses)) {
                    dto.setTextBoxType("");
                }else {
                    List<ApiFieldDetailResponse> responses = detailResponses.stream().filter(d -> d.getFieldBizId().equals(dto.getFieldBizId())).collect(Collectors.toList());
                    if (!CollectionUtils.isEmpty(responses)) {
                        ApiFieldDetailResponse response1 = responses.get(0);
                        dto.setTextBoxType(response1.getTextBoxType());
                    }
                }
                return dto;
            }).collect(Collectors.toList());
        }
        response.setApiAttributeSettingDtoList(apiAttributeSettingDtoList);

        //规格信息配置列表
        // 规格信息配置列表 - 按规格类型分组
        List<ApiSpeciesTypeDto> apiSpeciesTypeDtoList = apiSpeciesSettingService.querySpeciesTypeDtoList(productLaunchBizId);
        response.setApiSpeciesTypeDtoList(apiSpeciesTypeDtoList);

        //规格价格配置列表
        List<ApiSpeciesPriceDto> apiSpeciesPriceDtoList = apiSpeciesPriceService.querySpeciesPriceDtoList(productLaunchBizId);
        response.setApiSpeciesPriceDtoList(apiSpeciesPriceDtoList);


        return Result.success(response);
    }

    /**
     * 根据项目业务id返回封装结构
     * @param projectBizIds
     * @return
     */
    public List<ApiProjectDto> apiProjectDtoList(String projectBizIds) {
        List<String> projectBizIdList = StringUtils.isNotBlank(projectBizIds) ? Arrays.asList(projectBizIds.split(";").clone()) : new ArrayList<>();
        List<ApiProjectDto> list = new ArrayList<>();
        ApiSysProjectListRequest request = new ApiSysProjectListRequest();
        request.setProjectBizIdList(projectBizIdList);
        Result<List<ApiSysProjectDetailResponse>> result = apiSysProjectFeignClient.list(request);
        if (!CollectionUtils.isEmpty(result.getData())) {
            list = result.getData().stream().map(d -> {
                ApiProjectDto dto = new ApiProjectDto();
                BeanUtils.copyProperties(d,dto);
                return dto;
            }).collect(Collectors.toList());
        }
        return list;
    }

    /**
     * 保存-产品上架信息
     * @param request
     * @return
     */
    @Override
    public Result save(ApiProductLaunchSaveRequest request) {
        String productLaunchBizId = "";
        if (!Objects.isNull(request.getSource()) && request.getSource() == 1) {
            //复制来源
            //添加-产品上架类目信息
            ApiProductLaunchCategoryAddRequest addRequest = new ApiProductLaunchCategoryAddRequest();
            addRequest.setCategoryBizIdList(request.getApiProductLaunchDto().getCategoryBizIdList());
            addRequest.setMainUrlsList(request.getApiProductLaunchDto().getMainUrlsList());
            addRequest.setProductBizId(request.getApiProductLaunchDto().getProductBizId());
            addRequest.setProductName(request.getApiProductLaunchDto().getTitle());
            Result<ApiProductLaunchCategoryAddResponse> result = addCategory(addRequest);
            ApiProductLaunchCategoryAddResponse addResponse = result.getData();

            ApiProductLaunchDto apiProductLaunchDto = request.getApiProductLaunchDto();
            ProductLaunch productLaunch = new ProductLaunch();
            //更新产品上架主信息
            BeanUtils.copyProperties(apiProductLaunchDto,productLaunch);
            productLaunch.setId(addResponse.getId());
            productLaunch.setProductLaunchBizId(addResponse.getProductLaunchBizId());
            productLaunch.setMainUrls(!CollectionUtils.isEmpty(apiProductLaunchDto.getMainUrlsList()) ? String.join(";",apiProductLaunchDto.getMainUrlsList()) : "");
            productLaunch.setDetailUrls(!CollectionUtils.isEmpty(apiProductLaunchDto.getDetailUrlsList()) ? String.join(";",apiProductLaunchDto.getDetailUrlsList()) : "");
            productLaunch.setProjectBizIdList(!CollectionUtils.isEmpty(apiProductLaunchDto.getProjectBizIdList()) ? String.join(";",apiProductLaunchDto.getProjectBizIdList()) : "");
            productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_DSH.getItemValue());
            iProductLaunchService.saveOrUpdate(productLaunch);

            //设置产品上架信息参数列表
            apiAttributeSettingService.save(request.getApiAttributeSettingDtoList(),productLaunch.getProductLaunchBizId());
            //保存产品上架规格配置列表
            apiSpeciesSettingService.save(request.getApiSpeciesSettingDtoList(),productLaunch.getProductLaunchBizId());
            //保存规格价格配置列表
            apiSpeciesPriceService.save(request.getApiSpeciesPriceDtoList(),productLaunch.getProductLaunchBizId());

            productLaunchBizId = productLaunch.getProductLaunchBizId();
        }else {
            //其他来源
            ApiProductLaunchDto apiProductLaunchDto = request.getApiProductLaunchDto();
            //校验产品上架信息是否存在
            Result<ProductLaunch> result = checkProductLaunchIsExist(apiProductLaunchDto.getProductLaunchBizId());
            ProductLaunch productLaunch = result.getData();
            //更新产品上架主信息
            BeanUtils.copyProperties(apiProductLaunchDto,productLaunch);
            productLaunch.setMainUrls(!CollectionUtils.isEmpty(apiProductLaunchDto.getMainUrlsList()) ? String.join(";",apiProductLaunchDto.getMainUrlsList()) : "");
            productLaunch.setDetailUrls(!CollectionUtils.isEmpty(apiProductLaunchDto.getDetailUrlsList()) ? String.join(";",apiProductLaunchDto.getDetailUrlsList()) : "");
            productLaunch.setProjectBizIdList(!CollectionUtils.isEmpty(apiProductLaunchDto.getProjectBizIdList()) ? String.join(";",apiProductLaunchDto.getProjectBizIdList()) : "");
            iProductLaunchService.saveOrUpdate(productLaunch);

            //设置产品上架信息参数列表
            apiAttributeSettingService.save(request.getApiAttributeSettingDtoList(),productLaunch.getProductLaunchBizId());
            //保存产品上架规格配置列表
            apiSpeciesSettingService.save(request.getApiSpeciesSettingDtoList(),productLaunch.getProductLaunchBizId());
            //保存规格价格配置列表
            apiSpeciesPriceService.save(request.getApiSpeciesPriceDtoList(),productLaunch.getProductLaunchBizId());

            productLaunchBizId = productLaunch.getProductLaunchBizId();
        }
        return Result.success(productLaunchBizId);
    }

    /**
     * 审核-产品上架信息
     * @param request
     * @return
     */
    @Override
    public Result approval(ApiProductLaunchApprovalRequest request) {
        //校验产品上架信息是否存在
        Result<ProductLaunch> result = checkProductLaunchIsExist(request.getProductLaunchBizId());
        ProductLaunch productLaunch = result.getData();

        //审核备注
        productLaunch.setApprovalRemark(request.getApprovalRemark());
        if (ProductCommonEnum.APPROVAL_STATUS_TG.getItemValue().equals(request.getApprovalStatus())){
            //审核通过 -> 更新上架状态和更新租户/项目产品上架关系数据
            //定时开始是否开启，开启-待开售状态 关闭-在售状态
            if (NoYesEnum.YES.getItemValue().equals(productLaunch.getIsTiming())) {
                //定时开始是否开启，开启-预售状态
                productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_YS.getItemValue());
            }else if (NoYesEnum.NO.getItemValue().equals(productLaunch.getIsTiming())){
                //关闭-在售状态
                productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_ZS.getItemValue());
            }
            iProductLaunchService.saveOrUpdate(productLaunch);

            //更新租户/项目产品上架关系数据
            List<String> projectBizIdList = StringUtils.isNotBlank(productLaunch.getProjectBizIdList()) ? Arrays.asList(productLaunch.getProjectBizIdList().split(";").clone()) : new ArrayList<>();
            if (!CollectionUtils.isEmpty(projectBizIdList)) {
                //根据项目业务ID列表查询租户项目关系列表
                Result<List<QueryRelTenantProjectListResponse>> listResult = queryRelTenantProjectList(projectBizIdList);
                List<QueryRelTenantProjectListResponse> list = listResult.getData();
                //保存项目产品上架信息关系列表
                saveRelProjectProductLaunchList(list,productLaunch.getProductLaunchBizId());
                //保存租户产品上架信息关系列表
                saveRelTenantProductLaunchList(list,productLaunch.getProductLaunchBizId());
            }
        }else if (ProductCommonEnum.APPROVAL_STATUS_WTG.getItemValue().equals(request.getApprovalStatus())) {
            //审核未通过
            //更新上架状态为审核失败状态
            productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_SB.getItemValue());
            iProductLaunchService.saveOrUpdate(productLaunch);

            //审核未通过要清空租户/项目产品上架关系数据
            //根据产品业务ID删除项目产品关系
            iRelProjectProductLaunchService.delByProductLaunchBizId(productLaunch.getProductLaunchBizId());
            //根据产品业务ID删除租户产品关系
            iRelTenantProductLaunchService.delByProductLaunchBizId(productLaunch.getProductLaunchBizId());
        }

        return Result.success();
    }

    /**
     * 编辑-产品上架状态
     * @param request
     * @return
     */
    @Override
    public Result editStatus(ApiProductLaunchEditStatusRequest request) {
        //校验产品上架信息是否存在
        Result<ProductLaunch> result = checkProductLaunchIsExist(request.getProductLaunchBizId());
        ProductLaunch productLaunch = result.getData();
        if (ProductCommonEnum.PRODUCT_LAUNCH_STATUS_SJ.getItemValue().equals(request.getStatus())) {
            //上架 -> 待审核
            productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_DSH.getItemValue());
        }else if (ProductCommonEnum.PRODUCT_LAUNCH_STATUS_XJ.getItemValue().equals(request.getStatus())) {
            //下架 -> 下架
            productLaunch.setStatus(ProductCommonEnum.PRODUCT_LAUNCH_STATUS_XJ.getItemValue());
            //下架要清空租户/项目产品上架关系数据
            //根据产品业务ID删除项目产品关系
            iRelProjectProductLaunchService.delByProductLaunchBizId(productLaunch.getProductLaunchBizId());
            //根据产品业务ID删除租户产品关系
            iRelTenantProductLaunchService.delByProductLaunchBizId(productLaunch.getProductLaunchBizId());
        }
        iProductLaunchService.saveOrUpdate(productLaunch);
        return Result.success();
    }

    /**
     * 根据项目业务ID列表查询租户项目关系列表
     * @return
     */
    public Result<List<QueryRelTenantProjectListResponse>> queryRelTenantProjectList(List<String> projectBizIdList) {
        QueryRelTenantProjectListRequest request = new QueryRelTenantProjectListRequest();
        request.setProjectBizIdList(projectBizIdList);
        return apiRelTenantProjectFeignClient.queryRelTenantProjectList(request);
    }

    /**
     * 保存租户产品上架信息关系列表
     * @param responses
     * @param productLaunchBizId
     * @return
     */
    public Result saveRelTenantProductLaunchList(List<QueryRelTenantProjectListResponse> responses,String productLaunchBizId) {
        //先删后新增
        //根据产品业务ID删除租户产品关系
        iRelTenantProductLaunchService.delByProductLaunchBizId(productLaunchBizId);

        //新增租户产品关系
        if (!CollectionUtils.isEmpty(responses)) {
            List<RelTenantProductLaunch> saveList = responses.stream().map(dto -> {
                RelTenantProductLaunch relTenantProductLaunch = new RelTenantProductLaunch();
                relTenantProductLaunch.setProductLaunchBizId(productLaunchBizId);
                relTenantProductLaunch.setTenantBizId(dto.getTenantBizId());
                return relTenantProductLaunch;
            }).collect(Collectors.toList());
            iRelTenantProductLaunchService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 保存项目产品上架信息关系列表
     * @return
     */
    public Result saveRelProjectProductLaunchList(List<QueryRelTenantProjectListResponse> responses,String productLaunchBizId) {
        //先删后新增
        //根据产品业务ID删除项目产品关系
        iRelProjectProductLaunchService.delByProductLaunchBizId(productLaunchBizId);

        //新增项目产品关系
        if (!CollectionUtils.isEmpty(responses)) {
            List<RelProjectProductLaunch> saveList = responses.stream().map(dto -> {
                RelProjectProductLaunch relProjectProductLaunch = new RelProjectProductLaunch();
                relProjectProductLaunch.setProductLaunchBizId(productLaunchBizId);
                relProjectProductLaunch.setTenantBizId(dto.getTenantBizId());
                relProjectProductLaunch.setProjectBizId(dto.getProjectBizId());
                return relProjectProductLaunch;
            }).collect(Collectors.toList());
            iRelProjectProductLaunchService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 校验产品上架信息是否存在
     * @param productLaunchBizId
     * @return
     */
    @Override
    public Result<ProductLaunch> checkProductLaunchIsExist(String productLaunchBizId) {
        ProductLaunch productLaunch = iProductLaunchService.queryOne(productLaunchBizId);
        if (Objects.isNull(productLaunch)) {
            //数据不存在
            throw new BusinessException("产品上架信息数据不存在");
        }
        return Result.success(productLaunch);
    }
}
