package com.yd.product.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.product.api.service.ApiExpectedCommissionRatioService;
import com.yd.product.api.service.ApiExpectedSpeciesService;
import com.yd.product.feign.dto.ApiExpectedCommissionRatioBatchSaveDto;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioAddRequest;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioBatchSaveRequest;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioEditRequest;
import com.yd.product.feign.request.expectedcommissionratio.ApiExpectedCommissionRatioPageRequest;
import com.yd.product.feign.response.expectedcommissionratio.ApiExpectedCommissionRatioDetailResponse;
import com.yd.product.feign.response.expectedcommissionratio.ApiExpectedCommissionRatioPageResponse;
import com.yd.product.service.model.AnnouncementCommissionRatio;
import com.yd.product.service.model.ExpectedCommissionRatio;
import com.yd.product.service.service.IExpectedCommissionRatioService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiExpectedCommissionRatioServiceImpl implements ApiExpectedCommissionRatioService {

    @Autowired
    private IExpectedCommissionRatioService iExpectedCommissionRatioService;

    @Autowired
    private ApiExpectedSpeciesService apiExpectedSpeciesService;

    /**
     * 分页列表查询-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiExpectedCommissionRatioPageResponse>> page(ApiExpectedCommissionRatioPageRequest request) {
        Page<ApiExpectedCommissionRatioPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiExpectedCommissionRatioPageResponse> iPage = iExpectedCommissionRatioService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 批量保存-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result batchSave(ApiExpectedCommissionRatioBatchSaveRequest request) {
        List<String> expectedSpeciesBizIdList = new ArrayList<>();
        expectedSpeciesBizIdList.add(request.getExpectedSpeciesBizId());
        //先删后新增
        iExpectedCommissionRatioService.del(expectedSpeciesBizIdList);

        //新增
        if (!CollectionUtils.isEmpty(request.getRatioBatchSaveDtoList())) {
            List<ExpectedCommissionRatio> saveList = request.getRatioBatchSaveDtoList()
                    .stream()
                    .map(dto -> {
                        ExpectedCommissionRatio ratio = new ExpectedCommissionRatio();
                        BeanUtils.copyProperties(dto,ratio);
//                        ratio.setExpectedSpeciesBizId(request.getAnnouncementSpeciesBizId());
                        ratio.setExpectedCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_COMMISSION_RATIO.getCode()));
                        return ratio;
                    }).collect(Collectors.toList());
            iExpectedCommissionRatioService.saveOrUpdateBatch(saveList);
        }
        return Result.success();
    }

    /**
     * 添加-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiExpectedCommissionRatioAddRequest request) {
        //校验来佣比率规格信息是否存在
        apiExpectedSpeciesService.checkExpectedSpeciesIsExist(request.getExpectedSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //新增
        ExpectedCommissionRatio ratio = new ExpectedCommissionRatio();
        BeanUtils.copyProperties(request,ratio);
        ratio.setExpectedCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_COMMISSION_RATIO.getCode()));
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 编辑-来佣比率规格明细信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiExpectedCommissionRatioEditRequest request) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(request.getExpectedCommissionRatioBizId());
        ExpectedCommissionRatio ratio = result.getData();
        //校验来佣比率规格信息是否存在
        apiExpectedSpeciesService.checkExpectedSpeciesIsExist(request.getExpectedSpeciesBizId());
        //入参校验,佣金年限校验不同条件下的唯一性 TODO
        //编辑
        BeanUtils.copyProperties(request,ratio);
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 详情-来佣比率规格明细信息
     * @param expectedSpeciesBizId 来佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiExpectedCommissionRatioDetailResponse> detail(String expectedSpeciesBizId) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedSpeciesBizId);
        ExpectedCommissionRatio ratio = result.getData();
        ApiExpectedCommissionRatioDetailResponse response = new ApiExpectedCommissionRatioDetailResponse();
        BeanUtils.copyProperties(ratio,response);
        return Result.success(response);
    }

    /**
     * 删除-来佣比率规格明细信息
     * @param expectedSpeciesBizId 来佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result del(String expectedSpeciesBizId) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedSpeciesBizId);
        ExpectedCommissionRatio ratio = result.getData();
        iExpectedCommissionRatioService.removeById(ratio.getId());
        return Result.success();
    }

    /**
     * 复制-来佣比率规格明细信息
     * @param expectedSpeciesBizId 来佣比率规格明细表唯一业务ID
     * @return
     */
    @Override
    public Result copy(String expectedSpeciesBizId) {
        //校验来佣比率规格明细信息是否存在
        Result<ExpectedCommissionRatio> result = checkExpectedCommissionRatioIsExist(expectedSpeciesBizId);
        ExpectedCommissionRatio ratio = result.getData();
        ratio.setId(null);
        ratio.setExpectedCommissionRatioBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_COMMISSION_RATIO.getCode()));
        iExpectedCommissionRatioService.saveOrUpdate(ratio);
        return Result.success();
    }

    /**
     * 校验来佣比率规格明细信息是否存在
     * @param expectedCommissionRatioBizId
     * @return
     */
    @Override
    public Result<ExpectedCommissionRatio> checkExpectedCommissionRatioIsExist(String expectedCommissionRatioBizId) {
        ExpectedCommissionRatio expectedCommissionRatio = iExpectedCommissionRatioService.queryOne(expectedCommissionRatioBizId);
        if (Objects.isNull(expectedCommissionRatio)) {
            //数据不存在
            throw new BusinessException("来佣比率规格明细信息不存在");
        }
        return Result.success(expectedCommissionRatio);
    }
}
