package com.yd.oss.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.oss.api.service.ApiOssFileService;
import com.yd.oss.api.service.ApiRelObjectMaterialService;
import com.yd.oss.feign.request.ApiRelObjectMaterialListAddRequest;
import com.yd.oss.feign.request.ApiRelObjectMaterialPageRequest;
import com.yd.oss.feign.request.ApiRelObjectMaterialUploadSubmitRequest;
import com.yd.oss.feign.response.ApiRelObjectMaterialPageResponse;
import com.yd.oss.service.model.RelObjectMaterial;
import com.yd.oss.service.service.IRelObjectMaterialService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Service
@Slf4j
public class ApiRelObjectMaterialServiceImpl implements ApiRelObjectMaterialService {

    @Autowired
    private IRelObjectMaterialService iRelObjectMaterialService;

    @Autowired
    private ApiOssFileService apiOssFileService;

    /**
     * 分页列表查询-对象材料关系表信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiRelObjectMaterialPageResponse>> page(ApiRelObjectMaterialPageRequest request) {
        Page<ApiRelObjectMaterialPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiRelObjectMaterialPageResponse> iPage = iRelObjectMaterialService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 删除-对象材料关系表信息
     * @param relObjectMaterialBizId
     * @return
     */
    @Override
    public Result del(String relObjectMaterialBizId) {
        //校验对象材料关系表信息是否存在
        Result<RelObjectMaterial> result = checkRelObjectMaterialIsExist(relObjectMaterialBizId);
        RelObjectMaterial relObjectMaterial = result.getData();
        iRelObjectMaterialService.removeById(relObjectMaterial.getId());
        return Result.success();
    }

    /**
     * 添加-单个对象和材料列表关系信息
     * @param request
     * @return
     */
    @Override
    public Result addRelObjectMaterialList(ApiRelObjectMaterialListAddRequest request) {
        //先删后新增
        iRelObjectMaterialService.delByObjectBizId(request.getObjectBizId());

        if (!CollectionUtils.isEmpty(request.getMaterialDtoList())) {
            List<RelObjectMaterial> saveList = request.getMaterialDtoList().stream().map(dto -> {
                RelObjectMaterial relObjectMaterial = new RelObjectMaterial();
                BeanUtils.copyProperties(request,relObjectMaterial);
                relObjectMaterial.setMaterialBizId(dto.getMaterialBizId());
                relObjectMaterial.setRelObjectMaterialBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_REL_OBJECT_MATERIAL.getCode()));
                if (!CollectionUtils.isEmpty(dto.getFileBizIdList())) {
                    //附件不为空, 已上传
                    relObjectMaterial.setStatus("YSC");
                }else {
                    //附件为空, 未上传
                    relObjectMaterial.setStatus("WSC");
                }
                return relObjectMaterial;
            }).collect(Collectors.toList());
            iRelObjectMaterialService.saveOrUpdateBatch(saveList);

            //更新保存文件元数据列表
            apiOssFileService.updateByMaterial(saveList,request.getMaterialDtoList());
        }
        return Result.success();
    }

    /**
     * 上传-提交
     * @param request
     * @return
     */
    @Override
    public Result submit(ApiRelObjectMaterialUploadSubmitRequest request) {
        Result<RelObjectMaterial> result = checkRelObjectMaterialIsExist(request.getRelObjectMaterialBizId());
        RelObjectMaterial relObjectMaterial = result.getData();
        if (!CollectionUtils.isEmpty(request.getFileBizIdList())) {
            relObjectMaterial.setStatus("YSC");
            iRelObjectMaterialService.saveOrUpdate(relObjectMaterial);
        }
        //更新保存文件元数据列表
        apiOssFileService.updateByMaterial(relObjectMaterial.getRelObjectMaterialBizId(),request.getFileBizIdList());
        return Result.success();
    }

    /**
     * 校验对象材料关系表信息是否存在
     * @param relObjectMaterialBizId
     * @return
     */
    public Result<RelObjectMaterial> checkRelObjectMaterialIsExist(String relObjectMaterialBizId) {
        RelObjectMaterial relObjectMaterial = iRelObjectMaterialService.queryOne(relObjectMaterialBizId);
        if (Objects.isNull(relObjectMaterial)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(relObjectMaterial);
    }

}
