package com.yd.oss.service.utils;

import com.documents4j.api.DocumentType;
import com.documents4j.api.IConverter;
import com.documents4j.job.LocalConverter;
import lombok.extern.slf4j.Slf4j;
import java.io.*;

@Slf4j
public class PdfUtil {

    /**
     * windows系统word转pdf
     * @param pdfFile 转换后的pdf文件
     * @param wordFile word源文件
     */
    public static void winWordToPdf(File pdfFile, File wordFile) {
        try {
            IConverter converter = LocalConverter.builder().build();
            converter.convert(new FileInputStream(wordFile))
                    .as(DocumentType.DOCX)
                    .to(new FileOutputStream(pdfFile))
                    .as(DocumentType.PDF).execute();
        } catch (FileNotFoundException e) {
            log.error("word转换pdf失败", e);
        }
    }

    /**
     * linux系统word转pdf　　　* 使用LibreOffice转换。系统需安装LibreOffice　　　* 转换命令 libreoffice --invisible --convert-to pdf --outdir output_dir source_path　　　* 转换后的pdf文件名使用的是源文件的名称，所以如果要指定输出文件名称，就需把源文件名称改成想要输出的名称
     * @param pdfFile 转换后的pdf文件
     * @param wordFile word源文件
     */
    public static void linuxWordToPdf(File pdfFile, File wordFile) {
        String sourcePath = wordFile.getAbsolutePath();
        String outDir = pdfFile.getAbsolutePath().substring(0, pdfFile.getAbsolutePath().lastIndexOf(File.separator));

        // 使用绝对路径尝试不同的命令
        String[] commands = {
                "/usr/bin/libreoffice --invisible --convert-to pdf --outdir " + outDir + " " + sourcePath,
                "/usr/bin/soffice --invisible --convert-to pdf --outdir " + outDir + " " + sourcePath,
                "/usr/lib64/libreoffice/program/soffice --invisible --convert-to pdf --outdir " + outDir + " " + sourcePath
        };

        boolean success = false;
        Exception lastException = null;

        for (String command : commands) {
            try {
                log.info("尝试命令: " + command);
                executeLinuxCmd(command);

                // 检查PDF文件是否生成
                if (pdfFile.exists() && pdfFile.length() > 0) {
                    log.info("PDF转换成功，使用命令: " + command);
                    success = true;
                    break;
                } else {
                    log.warn("命令执行成功但未生成PDF文件: " + command);
                }
            } catch (Exception e) {
                lastException = e;
                log.warn("命令执行失败: " + command + ", 错误: " + e.getMessage());
            }
        }

        if (!success) {
            throw new RuntimeException("所有LibreOffice转换命令都失败了", lastException);
        }
    }

    /**
     * 执行命令行
     *
     * @param cmd 命令行
     * @return
     */
    private static boolean executeLinuxCmd(String cmd)  {
        try {
            Process process = Runtime.getRuntime().exec(cmd);
            process.waitFor();
        } catch (InterruptedException e) {
            log.error("executeLinuxCmd 执行Linux命令异常：", e);
            Thread.currentThread().interrupt();
            return false;
        } catch (IOException e) {
            log.error("获取系统命令执行环境异常", e);
        }
        return true;
    }

    public static void main(String[] args) {
        // 待转换的word文件
        File wordFile = new File("D:\\wjzh\\xcd.docx");
        // 转换后的pdf文件
        File pdfFile = new File("D:\\wjzh\\zhxcd.pdf");
        // 获取当前系统名称
        String osName = System.getProperty("os.name").toLowerCase();
        // 根据系统选择执行方法
        if (osName.contains("win")) {
            PdfUtil.winWordToPdf(pdfFile, wordFile);
        } else if (osName.contains("nux") || osName.contains("nix")) {
            PdfUtil.linuxWordToPdf(pdfFile, wordFile);
        }
    }
}
