package com.yd.insurance.base.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.insurance.base.api.service.ApiInsuranceProductPlanService;
import com.yd.insurance.base.feign.request.ApiInsuranceProductPlanAddRequest;
import com.yd.insurance.base.feign.request.ApiInsuranceProductPlanEditRequest;
import com.yd.insurance.base.feign.request.ApiInsuranceProductPlanPageRequest;
import com.yd.insurance.base.feign.response.ApiInsuranceProductDetailResponse;
import com.yd.insurance.base.feign.response.ApiInsuranceProductPlanDetailResponse;
import com.yd.insurance.base.feign.response.ApiInsuranceProductPlanPageResponse;
import com.yd.insurance.base.service.dto.InsuranceProductDto;
import com.yd.insurance.base.service.dto.InsuranceProductPlanDto;
import com.yd.insurance.base.service.enums.ProductDeptEnum;
import com.yd.insurance.base.service.model.InsuranceProduct;
import com.yd.insurance.base.service.model.InsuranceProductPlan;
import com.yd.insurance.base.service.service.IInsuranceProductPlanService;
import com.yd.user.feign.response.reldeptuser.ApiRelTenantDeptUserImportPageResponse;
import com.yd.user.service.model.RelProjectMenu;
import com.yd.user.service.model.SysProject;
import com.yd.user.service.model.SysTenant;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Objects;

@Service
public class ApiInsuranceProductPlanServiceImpl implements ApiInsuranceProductPlanService {

    @Autowired
    private IInsuranceProductPlanService iInsuranceProductPlanService;

    /**
     * 保险产品计划分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiInsuranceProductPlanPageResponse>> page(ApiInsuranceProductPlanPageRequest request) {
        Page<ApiInsuranceProductPlanPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiInsuranceProductPlanPageResponse> iPage = iInsuranceProductPlanService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 获取保险产品计划详情
     * @param planBizId 保险产品计划唯一标识(业务ID)
     * @return
     */
    @Override
    public Result<ApiInsuranceProductPlanDetailResponse> detail(String planBizId) {
        Result<InsuranceProductPlan> result = checkInsuranceProductPlan(planBizId);
        InsuranceProductPlan insuranceProductPlan = result.getData();

        ApiInsuranceProductPlanDetailResponse response = new ApiInsuranceProductPlanDetailResponse();
        BeanUtils.copyProperties(insuranceProductPlan,response);

        return Result.success(response);
    }

    /**
     * 添加保险产品计划
     * @param request
     * @return
     */
    @Override
    public Result add(ApiInsuranceProductPlanAddRequest request) {

        //校验产品计划名称的唯一性
        List<InsuranceProductPlan> list = iInsuranceProductPlanService.queryList(InsuranceProductPlanDto.builder().planName(request.getPlanName()).build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.PLAN_NAME_EXISTS.getCode(),ResultCode.PLAN_NAME_EXISTS.getMessage());
        }
        //新增保险产品
        InsuranceProductPlan insuranceProductPlan = new InsuranceProductPlan();
        BeanUtils.copyProperties(request,insuranceProductPlan);
        insuranceProductPlan.setPlanBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_INSURANCE_PRODUCT_PLAN.getCode()));
        iInsuranceProductPlanService.saveOrUpdate(insuranceProductPlan);

        return Result.success();
    }

    /**
     * 编辑保险产品计划
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiInsuranceProductPlanEditRequest request) {

        //校验保险产品计划是否存在
        Result<InsuranceProductPlan> result = checkInsuranceProductPlan(request.getPlanBizId());
        InsuranceProductPlan insuranceProductPlan = result.getData();

        //校验保险产品计划名称的唯一性
        List<InsuranceProductPlan> list = iInsuranceProductPlanService.queryList(InsuranceProductPlanDto.builder()
                .planName(request.getPlanName())
                .planBizId(request.getPlanBizId())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.PRODUCT_NAME_EXISTS.getCode(),ResultCode.PRODUCT_NAME_EXISTS.getMessage());
        }

        BeanUtils.copyProperties(request,insuranceProductPlan);
        iInsuranceProductPlanService.saveOrUpdate(insuranceProductPlan);

        return Result.success();
    }

    /**
     * 删除保险产品计划（逻辑删除）
     * @param planBizId 保险产品计划唯一标识(业务ID)
     * @return
     */
    @Override
    public Result del(String planBizId) {
        InsuranceProductPlan insuranceProductPlan = iInsuranceProductPlanService.queryOne(planBizId);
        if (Objects.isNull(insuranceProductPlan)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        iInsuranceProductPlanService.removeByPlanBizId(planBizId);
        return Result.success();
    }

    /**
     * 编辑保险产品计划状态
     * @param planBizId 保险产品计划唯一标识(业务ID)
     * @param status 状态(0:禁用 1:启用)
     * @return
     */
    @Override
    public Result editStatus(String planBizId, Integer status) {
        //校验保险产品计划是否存在
        Result<InsuranceProductPlan> result = checkInsuranceProductPlan(planBizId);
        InsuranceProductPlan insuranceProductPlan = result.getData();

        insuranceProductPlan.setStatus(status);
        iInsuranceProductPlanService.saveOrUpdate(insuranceProductPlan);

        return Result.success();
    }

    /**
     * 校验保险产品计划是否存在
     * @param planBizId
     * @return
     */
    public Result<InsuranceProductPlan> checkInsuranceProductPlan(String planBizId) {
        InsuranceProductPlan insuranceProductPlan = iInsuranceProductPlanService.queryOne(planBizId);
        if (Objects.isNull(insuranceProductPlan)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(insuranceProductPlan);
    }
}
