package com.yd.insurance.base.api.service.impl;

import com.yd.common.result.Result;
import com.yd.insurance.base.api.service.ApiRelInsuranceProductDeptService;
import com.yd.insurance.base.feign.dto.CdDeptDto;
import com.yd.insurance.base.feign.dto.SsDeptDto;
import com.yd.insurance.base.service.dto.RelInsuranceProductDeptDto;
import com.yd.insurance.base.service.enums.ProductDeptEnum;
import com.yd.insurance.base.service.model.RelInsuranceProductDept;
import com.yd.insurance.base.service.service.IRelInsuranceProductDeptService;
import com.yd.user.service.dto.SysDeptDto;
import com.yd.user.service.model.SysDept;
import com.yd.user.service.service.ISysDeptService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

@Service
public class ApiRelInsuranceProductDeptServiceImpl implements ApiRelInsuranceProductDeptService {

    @Autowired
    private IRelInsuranceProductDeptService iRelInsuranceProductDeptService;

    @Autowired
    private ISysDeptService iSysDeptService;

    /**
     * 更新保险产品和部门关系数据（先删后增加）
     * @param type 更新类型(1-所属保险公司 2-出单公司)
     * @param productBizId 保险产品ID（业务id）
     * @param deptBizIdList 部门id列表（业务id）
     * @return
     */
    @Override
    public Result updateRelInsuranceProductDeptList(Integer type,
                                                    String productBizId,
                                                    List<String> deptBizIdList) {
        //先删后增加
        iRelInsuranceProductDeptService.removeByProductBizIdAndType(type,productBizId);

        List<RelInsuranceProductDept> list = new ArrayList<>();
        if (!CollectionUtils.isEmpty(deptBizIdList)) {
            //部门唯一标识(业务ID) 列表（选中列表）不为空
            list = deptBizIdList.stream().map(bizId -> {
                RelInsuranceProductDept relInsuranceProductDept = new RelInsuranceProductDept();
                relInsuranceProductDept.setType(type);
                relInsuranceProductDept.setProductBizId(productBizId);
                relInsuranceProductDept.setDeptBizId(bizId);
                return relInsuranceProductDept;
            }).collect(Collectors.toList());

            iRelInsuranceProductDeptService.saveOrUpdateBatch(list);
        }
        return Result.success();
    }

    /**
     * 根据产品id查询所属公司对象集合
     * @param productBizId
     * @return
     */
    @Override
    public Result<List<SsDeptDto>> getSsDeptDtoList(String productBizId) {
        List<SsDeptDto> ssDeptDtoList = new ArrayList<>();

        List<RelInsuranceProductDept> list = iRelInsuranceProductDeptService.queryList(RelInsuranceProductDeptDto.builder()
                .productBizId(productBizId)
                .type(ProductDeptEnum.SS.getValue())
                .build());

        if (!CollectionUtils.isEmpty(list)) {
            List<String> deptBizIdList = list.stream().map(RelInsuranceProductDept::getDeptBizId).collect(Collectors.toList());
            List<SysDept> sysDeptList = iSysDeptService.queryList(SysDeptDto.builder()
                    .deptBizIdList(deptBizIdList)
                    .build());
            if (!CollectionUtils.isEmpty(sysDeptList)) {
                ssDeptDtoList = sysDeptList.stream().map(dto -> {
                    SsDeptDto ssDeptDto = new SsDeptDto();
                    BeanUtils.copyProperties(dto,ssDeptDto);
                    return ssDeptDto;
                }).collect(Collectors.toList());
            }
        }
        return Result.success(ssDeptDtoList);
    }

    /**
     * 根据产品id查询出单公司对象集合
     * @param productBizId
     * @return
     */
    @Override
    public Result<List<CdDeptDto>> getCdDeptDtoList(String productBizId) {
        List<CdDeptDto> cdDeptDtoList = new ArrayList<>();

        List<RelInsuranceProductDept> list = iRelInsuranceProductDeptService.queryList(RelInsuranceProductDeptDto.builder()
                .productBizId(productBizId)
                .type(ProductDeptEnum.CD.getValue())
                .build());

        if (!CollectionUtils.isEmpty(list)) {
            List<String> deptBizIdList = list.stream().map(RelInsuranceProductDept::getDeptBizId).collect(Collectors.toList());
            List<SysDept> sysDeptList = iSysDeptService.queryList(SysDeptDto.builder()
                    .deptBizIdList(deptBizIdList)
                    .build());
            if (!CollectionUtils.isEmpty(sysDeptList)) {
                cdDeptDtoList = sysDeptList.stream().map(dto -> {
                    CdDeptDto cdDeptDto = new CdDeptDto();
                    BeanUtils.copyProperties(dto,cdDeptDto);
                    return cdDeptDto;
                }).collect(Collectors.toList());
            }
        }
        return Result.success(cdDeptDtoList);
    }

}
