package com.yd.email.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.ApiEmailProviderConfigService;
import com.yd.email.api.service.ApiEmailSenderConfigService;
import com.yd.email.feign.dto.ApiEmailSenderConfigDto;
import com.yd.email.feign.request.ApiEmailSenderConfigAddRequest;
import com.yd.email.feign.request.ApiEmailSenderConfigEditRequest;
import com.yd.email.feign.request.ApiEmailSenderConfigPageRequest;
import com.yd.email.feign.response.ApiEmailSenderConfigDetailResponse;
import com.yd.email.feign.response.ApiEmailSenderConfigPageResponse;
import com.yd.email.service.dto.EmailSenderConfigDto;
import com.yd.email.service.model.EmailProviderConfig;
import com.yd.email.service.model.EmailSenderConfig;
import com.yd.email.service.service.IEmailSenderConfigService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

/**
 * 邮件发件人实现类
 */
@Slf4j
@Service
public class ApiEmailSenderConfigServiceImpl implements ApiEmailSenderConfigService {

    @Autowired
    private IEmailSenderConfigService iEmailSenderConfigService;

    @Autowired
    private ApiEmailProviderConfigService apiEmailProviderConfigService;

    /**
     * 分页查询-发件人
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiEmailSenderConfigPageResponse>> page(ApiEmailSenderConfigPageRequest request) {
        Page<ApiEmailSenderConfigPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiEmailSenderConfigPageResponse> iPage = iEmailSenderConfigService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-发件人配置
     * @param request
     * @return
     */
    @Override
    public Result add(ApiEmailSenderConfigAddRequest request) {
        //校验邮箱服务商是否存在
        apiEmailProviderConfigService.checkEmailProviderConfigIsExist(request.getProviderBizId());
        //校验邮箱名称唯一性
        List<EmailSenderConfig> list = iEmailSenderConfigService.queryList(EmailSenderConfigDto.builder()
                .email(request.getEmail())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.EMAIL_EXISTS.getCode(),ResultCode.EMAIL_EXISTS.getMessage());
        }

        EmailSenderConfig emailSenderConfig = new EmailSenderConfig();
        BeanUtils.copyProperties(request,emailSenderConfig);
        //设置发件人业务唯一ID
        emailSenderConfig.setSenderBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_SENDER_CONFIG.getCode()));
        //邮箱密码或授权码加密存入 TODO
        iEmailSenderConfigService.saveOrUpdate(emailSenderConfig);
        return Result.success();
    }

    /**
     * 编辑-发件人配置
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiEmailSenderConfigEditRequest request) {
        //校验邮件发件人是否存在
        Result<EmailSenderConfig> result = checkEmailSenderConfigIsExist(request.getSenderBizId());
        EmailSenderConfig emailSenderConfig = result.getData();
        Long id = emailSenderConfig.getId();
        //校验邮箱服务商是否存在
        apiEmailProviderConfigService.checkEmailProviderConfigIsExist(request.getProviderBizId());
        //校验邮箱名称唯一性
        List<EmailSenderConfig> list = iEmailSenderConfigService.queryList(EmailSenderConfigDto.builder()
                .email(request.getEmail())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.EMAIL_EXISTS.getCode(),ResultCode.EMAIL_EXISTS.getMessage());
        }

        BeanUtils.copyProperties(request,emailSenderConfig);
        emailSenderConfig.setId(id);
        //邮箱密码或授权码加密存入 TODO
        iEmailSenderConfigService.saveOrUpdate(emailSenderConfig);
        return Result.success();
    }

    /**
     * 详情-发件人
     * @param senderBizId 发件人唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailSenderConfigDetailResponse> detail(String senderBizId) {
        ApiEmailSenderConfigDetailResponse response = new ApiEmailSenderConfigDetailResponse();
        //校验邮件发件人是否存在
        Result<EmailSenderConfig> result = checkEmailSenderConfigIsExist(senderBizId);
        EmailSenderConfig emailSenderConfig = result.getData();
        //校验邮箱服务商是否存在
        Result<EmailProviderConfig> result1 = apiEmailProviderConfigService.checkEmailProviderConfigIsExist(emailSenderConfig.getProviderBizId());
        EmailProviderConfig emailProviderConfig = result1.getData();
        BeanUtils.copyProperties(emailSenderConfig,response);
        response.setProviderName(emailProviderConfig.getProviderName());
        return Result.success(response);
    }

    /**
     * 删除-发件人
     * @param senderBizId 发件人唯一业务ID
     * @return
     */
    @Override
    public Result del(String senderBizId) {
        //校验邮件发件人是否存在
        Result<EmailSenderConfig> result = checkEmailSenderConfigIsExist(senderBizId);
        EmailSenderConfig emailSenderConfig = result.getData();
        iEmailSenderConfigService.removeById(emailSenderConfig.getId());
        return Result.success();
    }

    /**
     * 根据发件人唯一业务ID获取发件人信息和发件人绑定的服务商信息
     * @param senderBizId 发件人唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailSenderConfigDto> getApiEmailSenderConfigDto(String senderBizId) {
        ApiEmailSenderConfigDto response = new ApiEmailSenderConfigDto();
        //校验邮件发件人是否存在
        Result<EmailSenderConfig> result = checkEmailSenderConfigIsExist(senderBizId);
        EmailSenderConfig emailSenderConfig = result.getData();
        //校验邮箱服务商是否存在
        Result<EmailProviderConfig> result1 = apiEmailProviderConfigService.checkEmailProviderConfigIsExist(emailSenderConfig.getProviderBizId());
        EmailProviderConfig emailProviderConfig = result1.getData();
        //拷贝发件人信息
        BeanUtils.copyProperties(emailSenderConfig,response);
        //拷贝邮件服务商信息
        BeanUtils.copyProperties(emailProviderConfig,response);
        response.setSendEmail(emailSenderConfig.getEmail());
        //发件人邮箱-是否设置SMTP需要身份验证 默认true
        response.setMailSmtpAuth("true");
        response.setStarttlsEnable("true");
        return Result.success(response);
    }

    /**
     * 校验邮件发件人是否存在
     * @param senderBizId
     * @return
     */
    public Result<EmailSenderConfig> checkEmailSenderConfigIsExist(String senderBizId) {
        EmailSenderConfig emailSenderConfig = iEmailSenderConfigService.queryOne(senderBizId);
        if (Objects.isNull(emailSenderConfig)) {
            //数据不存在
            throw new BusinessException("邮件发件人信息不存在！");
        }
        return Result.success(emailSenderConfig);
    }

}
