package com.yd.email.api.service.impl;

import com.yd.email.api.service.XxlJobService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;
import java.util.Calendar;
import java.util.Date;

@Service
@Slf4j
public class XxlJobServiceImpl implements XxlJobService {

    // 从配置文件中注入XXL-Job管理端的地址
    @Value("${xxl.job.admin.addresses}")
    private String adminAddresses;

    // 从配置文件中注入执行器的应用名称
    @Value("${xxl.job.executor.appname}")
    private String appName;

    // 自动注入Spring的RestTemplate用于HTTP请求
    @Autowired
    private RestTemplate restTemplate;

    /**
     * 添加定时任务
     * @param taskBizId 邮件任务ID
     * @param scheduleTime 计划执行时间
     * @return 返回创建的jobId
     */
    @Override
    public String addScheduleJob(String taskBizId, Date scheduleTime) {
        // 使用try-catch捕获可能出现的异常
        try {
            // 将Date时间转换为Cron表达式（在指定时间执行一次）
            String cronExpression = convertDateToCron(scheduleTime);

            // 创建MultiValueMap用于存储HTTP请求参数
            MultiValueMap<String, String> params = new LinkedMultiValueMap<>();
            // 设置执行器组ID，默认为1
            params.add("jobGroup", "1");
            // 设置任务描述，包含任务ID用于标识
            params.add("jobDesc", "邮件发送任务-" + taskBizId);
            // 设置任务作者
            params.add("author", "system");
            // 设置调度类型为CRON表达式
            params.add("scheduleType", "CRON");
            // 设置CRON表达式
            params.add("scheduleConf", cronExpression);
            // 设置任务模式为BEAN模式
            params.add("glueType", "BEAN");
            // 设置执行器处理器名称（对应@XxlJob注解的值）
            params.add("executorHandler", "mailSendJobHandler");
            // 设置任务参数（传递邮件任务ID）
            params.add("executorParam", taskBizId);
            // 设置路由策略为第一个
            params.add("executorRouteStrategy", "FIRST");
            // 设置调度过期策略为忽略
            params.add("misfireStrategy", "DO_NOTHING");
            // 设置阻塞处理策略为串行执行
            params.add("executorBlockStrategy", "SERIAL_EXECUTION");

            // 构建完整的API请求URL
            String url = adminAddresses + "/jobinfo/add";
            // 发送POST请求到XXL-Job管理端创建任务
            ResponseEntity<String> response = restTemplate.postForEntity(url, params, String.class);

            // 检查HTTP响应状态码是否为2xx成功
            if (response.getStatusCode().is2xxSuccessful()) {
                // 记录成功日志
                log.info("创建XXL-Job任务成功, taskId: {}", taskBizId);
                // 从响应体中提取jobId
                return extractJobId(response.getBody());
            } else {
                // 如果HTTP请求失败，抛出运行时异常
                throw new RuntimeException("XXL-Job API调用失败: " + response.getStatusCode());
            }

        } catch (Exception e) {
            // 捕获所有异常并记录错误日志
            log.error("创建XXL-Job任务失败", e);
            // 抛出包装后的运行时异常
            throw new RuntimeException("创建定时任务失败: " + e.getMessage());
        }
    }

    /**
     * 将Date转换为Cron表达式（只执行一次）
     * @param date 需要转换的日期时间
     * @return 返回对应的Cron表达式
     */
    private String convertDateToCron(Date date) {
        // 创建Calendar实例用于时间解析
        Calendar calendar = Calendar.getInstance();
        // 设置Calendar的时间为传入的Date
        calendar.setTime(date);

        // 获取秒数（0-59）
        int second = calendar.get(Calendar.SECOND);
        // 获取分钟数（0-59）
        int minute = calendar.get(Calendar.MINUTE);
        // 获取小时数（24小时制，0-23）
        int hour = calendar.get(Calendar.HOUR_OF_DAY);
        // 获取日期（1-31）
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        // 获取月份（Calendar的月份从0开始，所以需要+1）
        int month = calendar.get(Calendar.MONTH) + 1;
        // 获取年份
        int year = calendar.get(Calendar.YEAR);

        // 格式化Cron表达式：秒 分 时 日 月 周 年
        // 注意：Cron表达式通常不包含年份，但这里为了单次执行包含了年份
        return String.format("%d %d %d %d %d ? %d", second, minute, hour, day, month, year);
    }

    /**
     * 从XXL-Job的响应体中提取jobId
     * @param responseBody HTTP响应体内容
     * @return 返回提取的jobId
     */
    private String extractJobId(String responseBody) {
        // 注意：这里需要根据XXL-Job API的实际返回格式进行解析
        // 目前使用时间戳生成临时jobId，实际应该解析JSON响应
        // XXL-Job通常返回JSON格式：{"code":200,"msg":null,"content":123}
        // 其中content就是jobId

        // 临时方案：生成基于时间戳的jobId
        return "job_" + System.currentTimeMillis();
    }
}