package com.yd.email.service.handler;

import com.xxl.job.core.biz.model.ReturnT;
import com.xxl.job.core.handler.IJobHandler;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

@Component
@Slf4j
public class MailSendJobHandler extends IJobHandler {
    
    @Autowired
    private MailService mailService;
    
    @Autowired
    private MailTaskMapper mailTaskMapper;
    
    @Autowired
    private MailRecipientMapper mailRecipientMapper;
    
    @Override
    public ReturnT<String> execute(String param) throws Exception {
        log.info("开始执行邮件发送任务，参数: {}", param);
        
        Long taskId = Long.parseLong(param);
        MailTask mailTask = mailTaskMapper.selectById(taskId);
        
        if (mailTask == null) {
            return new ReturnT<>(ReturnT.FAIL_CODE, "邮件任务不存在");
        }
        
        // 更新任务状态为发送中
        mailTask.setStatus(1);
        mailTaskMapper.updateById(mailTask);
        
        try {
            // 查询所有收件人
            List<MailRecipient> recipients = mailRecipientMapper.selectByTaskId(taskId);
            
            int successCount = 0;
            int failCount = 0;
            
            for (MailRecipient recipient : recipients) {
                try {
                    List<String> ccList = recipient.getCcAddresses() != null ? 
                        Arrays.asList(recipient.getCcAddresses().split(",")) : 
                        new ArrayList<>();
                    
                    mailService.sendMail(
                        mailTask.getFromAddress(),
                        recipient.getToAddress(),
                        ccList,
                        mailTask.getSubject(),
                        mailTask.getContent(),
                        mailTask.getAttachmentPath()
                    );
                    
                    // 更新发送状态
                    recipient.setSendStatus(1);
                    recipient.setSendTime(new Date());
                    mailRecipientMapper.updateById(recipient);
                    successCount++;
                    
                } catch (Exception e) {
                    log.error("发送邮件失败: {}", recipient.getToAddress(), e);
                    recipient.setSendStatus(2);
                    recipient.setErrorMsg(e.getMessage());
                    mailRecipientMapper.updateById(recipient);
                    failCount++;
                }
            }
            
            // 更新任务状态
            mailTask.setStatus(failCount == 0 ? 2 : 3); // 全部成功为2，有失败为3
            mailTaskMapper.updateById(mailTask);
            
            log.info("邮件发送任务完成: 成功{}个, 失败{}个", successCount, failCount);
            return new ReturnT<>("发送完成: 成功" + successCount + "个, 失败" + failCount + "个");
            
        } catch (Exception e) {
            log.error("邮件发送任务执行异常", e);
            mailTask.setStatus(3);
            mailTaskMapper.updateById(mailTask);
            return new ReturnT<>(ReturnT.FAIL_CODE, "任务执行异常: " + e.getMessage());
        }
    }
}