package com.yd.email.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.ApiEmailVariableService;
import com.yd.email.feign.request.ApiEmailVariableAddRequest;
import com.yd.email.feign.request.ApiEmailVariableEditRequest;
import com.yd.email.feign.request.ApiEmailVariablePageRequest;
import com.yd.email.feign.response.ApiEmailVariableDetailResponse;
import com.yd.email.feign.response.ApiEmailVariablePageResponse;
import com.yd.email.service.dto.EmailVariableDto;
import com.yd.email.service.model.EmailVariable;
import com.yd.email.service.service.IEmailVariableService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

/**
 * 变量实现类
 */
@Slf4j
@Service
public class ApiEmailVariableServiceImpl implements ApiEmailVariableService {

    @Autowired
    private IEmailVariableService iEmailVariableService;

    /**
     * 分页查询-变量
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiEmailVariablePageResponse>> page(ApiEmailVariablePageRequest request) {
        Page<ApiEmailVariablePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiEmailVariablePageResponse> iPage = iEmailVariableService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-变量
     * @param request
     * @return
     */
    @Override
    public Result add(ApiEmailVariableAddRequest request) {
        //校验变量字段名称中文名的唯一性
        List<EmailVariable> list = iEmailVariableService.queryList(EmailVariableDto.builder()
                .variableNameCn(request.getVariableNameCn())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("变量字段名称中文名已存在！");
        }
        //校验变量字段名称英文名的唯一性
        List<EmailVariable> list1 = iEmailVariableService.queryList(EmailVariableDto.builder()
                .variableNameEn(request.getVariableNameEn())
                .build());
        if (!CollectionUtils.isEmpty(list1)) {
            throw new BusinessException("变量字段名称英文名已存在！");
        }

        EmailVariable emailVariable = new EmailVariable();
        BeanUtils.copyProperties(request,emailVariable);
        //设置变量业务唯一ID
        emailVariable.setVariableBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_VARIABLE.getCode()));
        iEmailVariableService.saveOrUpdate(emailVariable);
        return Result.success();
    }

    /**
     * 编辑-变量
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiEmailVariableEditRequest request) {
        //校验变量信息是否存在
        Result<EmailVariable> result = checkEmailVariableIsExist(request.getVariableBizId());
        EmailVariable emailVariable = result.getData();
        Long id = emailVariable.getId();
        //校验变量字段名称中文名的唯一性
        List<EmailVariable> list = iEmailVariableService.queryList(EmailVariableDto.builder()
                .variableNameCn(request.getVariableNameCn())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("变量字段名称中文名已存在！");
        }
        //校验变量字段名称英文名的唯一性
        List<EmailVariable> list1 = iEmailVariableService.queryList(EmailVariableDto.builder()
                .variableNameEn(request.getVariableNameEn())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list1)) {
            throw new BusinessException("变量字段名称英文名已存在！");
        }

        BeanUtils.copyProperties(request,emailVariable);
        emailVariable.setId(id);
        iEmailVariableService.saveOrUpdate(emailVariable);
        return Result.success();
    }

    /**
     * 详情-变量
     * @param variableBizId 变量唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailVariableDetailResponse> detail(String variableBizId) {
        //校验变量信息是否存在
        Result<EmailVariable> result = checkEmailVariableIsExist(variableBizId);
        EmailVariable emailVariable = result.getData();
        ApiEmailVariableDetailResponse response = new ApiEmailVariableDetailResponse();
        BeanUtils.copyProperties(emailVariable,response);
        return Result.success(response);
    }

    /**
     * 删除-变量
     * @param variableBizId 变量唯一业务ID
     * @return
     */
    @Override
    public Result del(String variableBizId) {
        //校验变量信息是否存在
        Result<EmailVariable> result = checkEmailVariableIsExist(variableBizId);
        EmailVariable emailVariable = result.getData();
        iEmailVariableService.removeById(emailVariable.getId());
        return Result.success();
    }

    /**
     * 校验变量信息是否存在
     * @param variableBizId
     * @return
     */
    public Result<EmailVariable> checkEmailVariableIsExist(String variableBizId) {
        EmailVariable emailVariable = iEmailVariableService.queryOne(variableBizId);
        if (Objects.isNull(emailVariable)) {
            //数据不存在
            throw new BusinessException("变量信息不存在！");
        }
        return Result.success(emailVariable);
    }
}
