package com.yd.email.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.ApiEmailProviderConfigService;
import com.yd.email.feign.request.ApiEmailProviderConfigAddRequest;
import com.yd.email.feign.request.ApiEmailProviderConfigEditRequest;
import com.yd.email.feign.request.ApiEmailProviderConfigPageRequest;
import com.yd.email.feign.response.ApiEmailProviderConfigDetailResponse;
import com.yd.email.feign.response.ApiEmailProviderConfigPageResponse;
import com.yd.email.service.dto.EmailProviderConfigDto;
import com.yd.email.service.model.EmailProviderConfig;
import com.yd.email.service.service.IEmailProviderConfigService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

/**
 * 邮箱服务商实现类
 */
@Slf4j
@Service
public class ApiEmailProviderConfigServiceImpl implements ApiEmailProviderConfigService {

    @Autowired
    private IEmailProviderConfigService iEmailProviderConfigService;

    /**
     * 分页查询-邮箱服务商
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiEmailProviderConfigPageResponse>> page(ApiEmailProviderConfigPageRequest request) {
        Page<ApiEmailProviderConfigPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiEmailProviderConfigPageResponse> iPage = iEmailProviderConfigService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-邮箱服务商
     * @param request
     * @return
     */
    @Override
    public Result add(ApiEmailProviderConfigAddRequest request) {
        //校验邮箱服务商名称唯一性
        List<EmailProviderConfig> list = iEmailProviderConfigService.queryList(EmailProviderConfigDto.builder()
                .providerName(request.getProviderName())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("邮箱服务商名称已存在！");
        }
        //校验SMTP服务器地址唯一性
        List<EmailProviderConfig> list1 = iEmailProviderConfigService.queryList(EmailProviderConfigDto.builder()
                .smtpHost(request.getSmtpHost())
                .build());
        if (!CollectionUtils.isEmpty(list1)) {
            throw new BusinessException("SMTP服务器地址已存在！");
        }

        EmailProviderConfig emailProviderConfig = new EmailProviderConfig();
        BeanUtils.copyProperties(request,emailProviderConfig);
        //设置邮箱服务商业务唯一ID
        emailProviderConfig.setProviderBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_PROVIDER_CONFIG.getCode()));
        iEmailProviderConfigService.saveOrUpdate(emailProviderConfig);
        return Result.success();
    }

    /**
     * 编辑-邮箱服务商
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiEmailProviderConfigEditRequest request) {
        //校验邮箱服务商是否存在
        Result<EmailProviderConfig> result = checkEmailProviderConfigIsExist(request.getProviderBizId());
        EmailProviderConfig emailProviderConfig = result.getData();
        //校验邮箱服务商名称唯一性
        List<EmailProviderConfig> list = iEmailProviderConfigService.queryList(EmailProviderConfigDto.builder()
                .providerName(request.getProviderName())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("邮箱服务商名称已存在！");
        }
        //校验SMTP服务器地址唯一性
        List<EmailProviderConfig> list1 = iEmailProviderConfigService.queryList(EmailProviderConfigDto.builder()
                .smtpHost(request.getSmtpHost())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list1)) {
            throw new BusinessException("SMTP服务器地址已存在！");
        }

        BeanUtils.copyProperties(request,emailProviderConfig);
        iEmailProviderConfigService.saveOrUpdate(emailProviderConfig);
        return Result.success();
    }

    /**
     * 详情-邮箱服务商
     * @param providerBizId 邮箱服务商唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailProviderConfigDetailResponse> detail(String providerBizId) {
        //校验邮箱服务商是否存在
        Result<EmailProviderConfig> result = checkEmailProviderConfigIsExist(providerBizId);
        EmailProviderConfig emailProviderConfig = result.getData();
        ApiEmailProviderConfigDetailResponse response = new ApiEmailProviderConfigDetailResponse();
        BeanUtils.copyProperties(emailProviderConfig,response);
        return Result.success(response);
    }

    /**
     * 删除-邮箱服务商
     * @param providerBizId 邮箱服务商唯一业务ID
     * @return
     */
    @Override
    public Result del(String providerBizId) {
        //校验邮箱服务商是否存在
        Result<EmailProviderConfig> result = checkEmailProviderConfigIsExist(providerBizId);
        EmailProviderConfig emailProviderConfig = result.getData();
        iEmailProviderConfigService.removeById(emailProviderConfig.getId());
        return Result.success();
    }

    /**
     * 校验邮箱服务商是否存在
     * @param providerBizId
     * @return
     */
    @Override
    public Result<EmailProviderConfig> checkEmailProviderConfigIsExist(String providerBizId) {
        EmailProviderConfig EmailProviderConfig = iEmailProviderConfigService.queryOne(providerBizId);
        if (Objects.isNull(EmailProviderConfig)) {
            //数据不存在
            throw new BusinessException("邮箱服务商不存在！");
        }
        return Result.success(EmailProviderConfig);
    }
}
