package com.yd.email.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.common.utils.UUIDUtil;
import com.yd.email.api.service.ApiEmailContactImportService;
import com.yd.email.feign.request.ApiEmailContactImportEditRequest;
import com.yd.email.feign.request.ApiEmailContactImportPageRequest;
import com.yd.email.feign.request.ApiEmailContactImportSelectAddRequest;
import com.yd.email.feign.response.ApiEmailContactImportPageResponse;
import com.yd.email.feign.response.ApiEmailContactImportResponse;
import com.yd.email.feign.response.ApiEmailContactImportSelectAddResponse;
import com.yd.email.service.dto.EmailContactImportDto;
import com.yd.email.service.model.EmailContactImport;
import com.yd.email.service.service.IEmailContactImportService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 邮件联系人导入临时表实现类
 */
@Slf4j
@Service
public class ApiEmailContactImportServiceImpl implements ApiEmailContactImportService {

    @Autowired
    private IEmailContactImportService iEmailContactImportService;

    /**
     * 分页查询-邮件联系人导入的列表
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiEmailContactImportPageResponse>> page(ApiEmailContactImportPageRequest request) {
        Page<ApiEmailContactImportPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiEmailContactImportPageResponse> iPage = iEmailContactImportService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-选择联系人（写邮件选择联系人的时候保存的接口）
     * @param request
     * @return
     */
    @Override
    public Result<ApiEmailContactImportSelectAddResponse> add(ApiEmailContactImportSelectAddRequest request) {
        ApiEmailContactImportSelectAddResponse response = new ApiEmailContactImportSelectAddResponse();
        String sessionId = "";

        if (StringUtils.isBlank(request.getSessionId())) {
            //当前会话id（前端缓存有值就传，没有值后端生成返回给前端存入前端缓存）
            //无值生成不带连字符的随机UUID
            sessionId = UUIDUtil.randomUUIDWithoutHyphen();
        }
        if (!CollectionUtils.isEmpty(request.getApiEmailContactDtoList())) {
            //需要导入临时表的联系人列表
            String finalSessionId = sessionId;
            List<EmailContactImport> addList = request.getApiEmailContactDtoList()
                    .stream().map(dto -> {
                        EmailContactImport emailContactImport = new EmailContactImport();
                        emailContactImport.setImportBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_CONTACT_IMPORT.getCode()));
                        emailContactImport.setContactBizId(dto.getContactBizId());
                        emailContactImport.setCcEmail(dto.getCcEmails());
                        emailContactImport.setReceiveEmail(dto.getContactEmail());
                        emailContactImport.setSessionId(finalSessionId);
                        emailContactImport.setSource(0);
                        return emailContactImport;
                    }).collect(Collectors.toList());
            iEmailContactImportService.saveOrUpdateBatch(addList);
        }
        response.setSessionId(sessionId);
        return Result.success(response);
    }

    /**
     * 编辑-邮件联系人导入信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiEmailContactImportEditRequest request) {
        //校验邮件联系人导入信息是否存在
        Result<EmailContactImport> result = checkEmailContactImportIsExist(request.getImportBizId());
        EmailContactImport emailContactImport = result.getData();
        Long id = emailContactImport.getId();
        //校验当前会话id下的收件人邮箱是否唯一
        List<EmailContactImport> list = iEmailContactImportService.queryList(EmailContactImportDto.builder()
                .sessionId(emailContactImport.getSessionId())
                .receiveEmail(request.getReceiveEmail())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("收件人邮箱已存在！");
        }
        if (Objects.equals(request.getReceiveEmail(),emailContactImport.getReceiveEmail())) {
            //判断入参收件人邮箱和库里收件人邮箱是否一致，如果不一致说明修改过，清空收件人唯一业务ID字段
            emailContactImport.setContactBizId("");
        }
        emailContactImport.setId(id);
        emailContactImport.setReceiveEmail(request.getReceiveEmail());
        emailContactImport.setCcEmail(!CollectionUtils.isEmpty(request.getCcEmailList()) ? String.join(";",request.getCcEmailList()) : "");
        iEmailContactImportService.saveOrUpdate(emailContactImport);
        return Result.success();
    }

    /**
     * 详情-邮件联系人导入信息
     * @param importBizId 导入临时表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailContactImportResponse> detail(String importBizId) {
        //校验邮件联系人导入信息是否存在
        Result<EmailContactImport> result = checkEmailContactImportIsExist(importBizId);
        EmailContactImport emailContactImport = result.getData();
        ApiEmailContactImportResponse response = new ApiEmailContactImportResponse();
        BeanUtils.copyProperties(emailContactImport,response);
        response.setCcEmailList(StringUtils.isNotBlank(emailContactImport.getCcEmail()) ?
                Arrays.stream(emailContactImport.getCcEmail().split(";"))
                .map(String::trim)
                .filter(s -> !s.isEmpty())
                .collect(Collectors.toList()): new ArrayList<>());
        return Result.success(response);
    }


    /**
     * 删除-邮件联系人导入信息
     * @param importBizId 导入临时表唯一业务ID
     * @return
     */
    @Override
    public Result del(String importBizId) {
        //校验邮件联系人导入信息是否存在
        Result<EmailContactImport> result = checkEmailContactImportIsExist(importBizId);
        EmailContactImport emailContactImport = result.getData();

        iEmailContactImportService.removeById(emailContactImport.getId());
        return Result.success();
    }

    /**
     * 校验邮件联系人导入信息是否存在
     * @param importBizId
     * @return
     */
    public Result<EmailContactImport> checkEmailContactImportIsExist(String importBizId) {
        EmailContactImport emailContactImport = iEmailContactImportService.queryOne(importBizId);
        if (Objects.isNull(emailContactImport)) {
            //数据不存在
            throw new BusinessException("邮件联系人导入信息不存在！");
        }
        return Result.success(emailContactImport);
    }
}
