package com.yd.email.api.service.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.service.ApiEmailContactCcService;
import com.yd.email.api.service.ApiEmailContactService;
import com.yd.email.feign.dto.ApiEmailContactCcDto;
import com.yd.email.feign.enums.RecipientTypeEnum;
import com.yd.email.feign.request.ApiEmailContactAddRequest;
import com.yd.email.feign.request.ApiEmailContactEditRequest;
import com.yd.email.feign.request.ApiEmailContactPageRequest;
import com.yd.email.feign.response.ApiEmailContactDetailResponse;
import com.yd.email.feign.response.ApiEmailContactPageResponse;
import com.yd.email.service.dto.EmailContactDto;
import com.yd.email.service.model.EmailContact;
import com.yd.email.service.service.IEmailContactService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

/**
 * 邮箱联系人（收件人）实现类
 */
@Slf4j
@Service
public class ApiEmailContactServiceImpl implements ApiEmailContactService {

    @Autowired
    private IEmailContactService iEmailContactService;

    @Autowired
    private ApiEmailContactCcService apiEmailContactCcService;

    /**
     * 分页查询-联系人（收件人）
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiEmailContactPageResponse>> page(ApiEmailContactPageRequest request) {
        Page<ApiEmailContactPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiEmailContactPageResponse> iPage = iEmailContactService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-联系人（收件人）配置
     * @param request
     * @return
     */
    @Override
    public Result add(ApiEmailContactAddRequest request) {
        //校验联系人邮箱唯一性
        List<EmailContact> list = iEmailContactService.queryList(EmailContactDto.builder()
                .email(request.getEmail())
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.EMAIL_EXISTS.getCode(),ResultCode.EMAIL_EXISTS.getMessage());
        }

        EmailContact emailContact = new EmailContact();
        BeanUtils.copyProperties(request,emailContact);
        //设置联系人（收件人）业务唯一ID
        emailContact.setContactBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_CONTACT.getCode()));
        iEmailContactService.saveOrUpdate(emailContact);

        //保存关联抄送邮箱的列表
        apiEmailContactCcService.saveEmailContactCcList(buildCcEmailList(request.getCcEmailList(),emailContact.getContactBizId()),emailContact.getContactBizId());
        return Result.success();
    }

    /**
     * 编辑-联系人（收件人）配置
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiEmailContactEditRequest request) {
        //校验邮箱联系人是否存在
        Result<EmailContact> result = checkEmailContactIsExist(request.getContactBizId());
        EmailContact emailContact = result.getData();
        Long id = emailContact.getId();
        //校验联系人邮箱唯一性
        List<EmailContact> list = iEmailContactService.queryList(EmailContactDto.builder()
                .email(request.getEmail())
                .isExcludeMy(true)
                .build());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException(ResultCode.EMAIL_EXISTS.getCode(),ResultCode.EMAIL_EXISTS.getMessage());
        }

        BeanUtils.copyProperties(request,emailContact);
        emailContact.setId(id);
        iEmailContactService.saveOrUpdate(emailContact);

        //保存关联抄送邮箱的列表
        apiEmailContactCcService.saveEmailContactCcList(buildCcEmailList(request.getCcEmailList(),emailContact.getContactBizId()),emailContact.getContactBizId());
        return Result.success();
    }

    /**
     * 详情-联系人（收件人）
     * @param contactBizId 联系人（收件人）唯一业务ID
     * @return
     */
    @Override
    public Result<ApiEmailContactDetailResponse> detail(String contactBizId) {
        //校验邮箱联系人是否存在
        Result<EmailContact> result = checkEmailContactIsExist(contactBizId);
        EmailContact emailContact = result.getData();
        ApiEmailContactDetailResponse response = new ApiEmailContactDetailResponse();
        BeanUtils.copyProperties(emailContact,response);

        //获取抄送人邮箱列表
        Result<List<String>> result1 = apiEmailContactCcService.getEmailContactCcList(contactBizId);
        response.setCcEmailList(result1.getData());
        return Result.success(response);
    }

    /**
     * 删除-联系人（收件人）
     * @param contactBizId 联系人（收件人）唯一业务ID
     * @return
     */
    @Override
    public Result del(String contactBizId) {
        //校验邮箱联系人是否存在
        Result<EmailContact> result = checkEmailContactIsExist(contactBizId);
        EmailContact emailContact = result.getData();
        iEmailContactService.removeById(emailContact.getId());
        return Result.success();
    }

    /**
     * 构建邮件抄送人列表
     * @param ccEmailList
     * @param contactBizId
     * @return
     */
    public List<ApiEmailContactCcDto> buildCcEmailList(List<String> ccEmailList, String contactBizId) {
        List<ApiEmailContactCcDto> contactCcDtoList = new ArrayList<>();
        if (CollectionUtils.isEmpty(contactCcDtoList)) {
            contactCcDtoList = ccEmailList.stream().map(email -> {
                ApiEmailContactCcDto dto = new ApiEmailContactCcDto();
                dto.setContactBizId(contactBizId);
                dto.setEmail(email);
                //抄送
                dto.setType(RecipientTypeEnum.TCC.getItemValue());
                return dto;
            }).collect(Collectors.toList());
        }
        return contactCcDtoList;
    }

    /**
     * 校验邮箱联系人是否存在
     * @param contactBizId
     * @return
     */
    public Result<EmailContact> checkEmailContactIsExist(String contactBizId) {
        EmailContact emailContact = iEmailContactService.queryOne(contactBizId);
        if (Objects.isNull(emailContact)) {
            //数据不存在
            throw new BusinessException("邮箱联系人信息不存在！");
        }
        return Result.success(emailContact);
    }

    /**
     * 通用变量模板-替换邮件模板内容的变量占位符
     * @param content 替换的占位符内容
     * @param contactBizId 收件人业务ID
     * @return
     */
    @Override
    public String replacePlaceholders(String content, String contactBizId) {
        if (StringUtils.isBlank(content)) return "";

        String result = content;
        EmailContact emailContact = iEmailContactService.queryOne(contactBizId);

        if (!Objects.isNull(emailContact)) {
            // 使用 Optional 或三目运算符处理 null
            String name = Optional.ofNullable(emailContact.getName())
                    .filter(s -> !s.trim().isEmpty())
                    .orElse("");
            String appellation = Optional.ofNullable(emailContact.getAppellation())
                    .orElse("");
            String companyName = Optional.ofNullable(emailContact.getCompanyName())
                    .orElse("");

            result = result.replace("{{name}}", name)
                    .replace("{{appellation}}", appellation)
                    .replace("{{compantName}}", companyName);
        }

        return result;
    }
}
