package com.yd.email.api.handler;

import com.alibaba.fastjson.JSON;
import com.xxl.job.core.context.XxlJobHelper;
import com.xxl.job.core.handler.annotation.XxlJob;
import com.yd.email.api.service.ApiEmailService;
import com.yd.email.feign.enums.EmailTaskStatusEnum;
import com.yd.email.service.model.EmailTask;
import com.yd.email.service.model.EmailTaskRecipients;
import com.yd.email.service.service.IEmailTaskRecipientsService;
import com.yd.email.service.service.IEmailTaskService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * 邮件发送任务处理器 - XXL-Job定时任务执行器
 * 使用@XxlJob注解方式
 */
@Component
@Slf4j
public class MailSendJobHandler {

    @Autowired
    private IEmailTaskService iEmailTaskService;

    @Autowired
    private IEmailTaskRecipientsService iEmailTaskRecipientsService;

    @Autowired
    private ApiEmailService apiEmailService;

    /**
     * XXL-Job任务执行入口方法
     */
    @XxlJob("mailSendJobHandler")
    public void execute() throws Exception {
        // 从XXL-Job参数中获取任务ID
        String param = XxlJobHelper.getJobParam();
        // 记录任务开始日志
        log.info("开始执行邮件发送任务，参数: {}", param);

        // 解析任务参数（参数为邮件任务业务ID）
        String taskBizId = param;
        // 根据邮件任务业务ID查询邮件任务信息
        EmailTask emailTask = iEmailTaskService.queryOne(taskBizId);
        log.info("根据邮件任务业务ID查询邮件任务信息: {}: " + JSON.toJSONString(emailTask));

        // 检查邮件任务是否存在
        if (Objects.isNull(emailTask)) {
            // 记录错误并返回
            XxlJobHelper.log("邮件任务不存在: " + taskBizId);
            XxlJobHelper.handleFail("邮件任务不存在");
            return;
        }

        // 更新任务状态为发送中
        emailTask.setStatus(EmailTaskStatusEnum.SENDING.getItemValue());
        iEmailTaskService.saveOrUpdate(emailTask);

        try {
            // 查询该任务下的所有收件人信息
            List<EmailTaskRecipients> recipients = iEmailTaskRecipientsService.queryList(taskBizId);
            log.info("查询该任务下的所有收件人信息: {}: " + JSON.toJSONString(recipients));

            // 初始化成功和失败计数器
            int successCount = 0;
            int failCount = 0;

            // 遍历所有收件人，逐个发送邮件
            for (EmailTaskRecipients recipient : recipients) {
                try {
                    // 处理抄送人列表：将数据库中的逗号分隔字符串转换为List
                    List<String> ccList = StringUtils.isNotBlank(recipient.getCcEmail()) ?
                            Arrays.asList(recipient.getCcEmail().split(",")) :
                            new ArrayList<>();
                    log.info("处理抄送人列表：将数据库中的逗号分隔字符串转换为List: {}: " + JSON.toJSONString(ccList));
                    // 调用邮件服务发送邮件
                    apiEmailService.sendMail(
                            emailTask.getSendEmail(),    // 发件人地址
                            recipient.getReceiveEmail(),     // 收件人地址
                            ccList,                       // 抄送人列表
                            emailTask.getSubject(),        // 邮件主题
                            emailTask.getContent(),        // 邮件内容
                            emailTask.getAttachmentPath()  // 附件路径
                    );

                    // 发送成功：更新收件人状态为成功
                    recipient.setStatus(EmailTaskStatusEnum.SUCCESSFUL.getItemValue());
                    recipient.setSendTime(LocalDateTime.now());  // 记录实际发送时间
                    iEmailTaskRecipientsService.saveOrUpdate(recipient);
                    successCount++;  // 成功计数加1

                } catch (Exception e) {
                    // 发送失败：记录错误日志
                    log.error("发送邮件失败: {}", recipient.getReceiveEmail(), e);
                    // 更新收件人状态为失败
                    recipient.setStatus(EmailTaskStatusEnum.FAILED.getItemValue());
                    recipient.setErrorMsg(e.getMessage());  // 保存错误信息
                    iEmailTaskRecipientsService.saveOrUpdate(recipient);
                    failCount++;  // 失败计数加1
                }
            }

            // 根据发送结果更新邮件任务状态
            emailTask.setStatus(failCount == 0 ?
                    EmailTaskStatusEnum.ALL_SUCCESSFUL.getItemValue() :
                    EmailTaskStatusEnum.PARTIAL_FAILURE.getItemValue());
            iEmailTaskService.saveOrUpdate(emailTask);

            // 记录任务完成日志
            log.info("邮件发送任务完成: 成功{}个, 失败{}个", successCount, failCount);

            // 设置任务执行结果
            if (failCount == 0) {
                XxlJobHelper.handleSuccess("发送完成: 成功" + successCount + "个");
            } else {
                XxlJobHelper.handleSuccess("发送完成: 成功" + successCount + "个, 失败" + failCount + "个");
            }

        } catch (Exception e) {
            // 任务执行过程中发生异常
            log.error("邮件发送任务执行异常", e);
            // 更新任务状态为全部发送失败
            emailTask.setStatus(EmailTaskStatusEnum.ALL_FAILED.getItemValue());
            iEmailTaskService.saveOrUpdate(emailTask);
            // 返回任务执行异常信息
            XxlJobHelper.handleFail("任务执行异常: " + e.getMessage());
        }
    }
}