package com.yd.email.api.service.impl;

import com.alibaba.fastjson.JSON;
import com.yd.email.api.service.ApiEmailService;
import com.yd.email.feign.dto.ApiSendMailDto;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.activation.DataHandler;
import javax.activation.DataSource;
import javax.activation.FileDataSource;
import javax.mail.*;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import java.io.File;
import java.util.Date;
import java.util.List;
import java.util.Properties;

/**
 * 邮件服务实现类
 */
@Service
@Slf4j
public class ApiEmailServiceImpl implements ApiEmailService {

    @Value("${spring.mail.host}") // 从配置文件中注入邮件服务器主机地址
    private String host;

    @Value("${spring.mail.username}") // 从配置文件中注入发件人用户名
    private String username;

    @Value("${spring.mail.password}") // 从配置文件中注入发件人密码
    private String password;

    /**
     * 发送邮件
     * @param dto
     * @throws Exception
     */
    @Override
    public void sendMail(ApiSendMailDto dto) throws Exception {
        log.info("发送邮件的主要方法参数: {}: "
                + dto.getSendEmail() + "；"
                + dto.getReceiveEmail() + "；"
                + JSON.toJSONString(dto.getCcEmailList())
                + "；" + dto.getSubject() + "；" + dto.getContent() );

        // 创建邮件配置属性对象
        Properties props = new Properties();
        // 设置SMTP需要身份验证
        props.put("mail.smtp.auth", dto.getMailSmtpAuth());
        // 启用TLS加密传输
        props.put("mail.smtp.starttls.enable", dto.getStarttlsEnable());
        // 设置邮件服务器主机名（发件人邮箱-SMTP服务器地址）
        props.put("mail.smtp.host", dto.getSmtpHost());
        log.info("mail.smtp.host：{}",dto.getSmtpHost());
        //发件人邮箱（单个）
        log.info("spring.mail.username：{}",dto.getSendEmail());
        //发件人邮箱密码或授权码
        log.info("spring.mail.password：{}",dto.getPassword());
        // 设置邮件服务器端口号（587是TLS标准端口）
        //发件人邮箱-SMTP服务器端口
        props.put("mail.smtp.port", dto.getSmtpPort());

        // 创建邮件会话对象，传入配置和认证器
        Session session = Session.getInstance(props, new Authenticator() {
            // 重写获取密码认证的方法
            protected PasswordAuthentication getPasswordAuthentication() {
                // 返回用户名密码认证对象
                return new PasswordAuthentication(dto.getSendEmail(), dto.getPassword());
            }
        });

        // 使用try-catch块处理邮件发送异常
        try {
            // 创建MIME类型邮件消息对象
            Message message = new MimeMessage(session);
            // 设置发件人地址
            message.setFrom(new InternetAddress(dto.getSendEmail()));
            // 设置收件人地址（支持多个收件人，用逗号分隔）
            message.setRecipients(Message.RecipientType.TO, InternetAddress.parse(dto.getReceiveEmail()));

            // 判断是否有抄送人
            if (!CollectionUtils.isEmpty(dto.getCcEmailList())) {
                // 将抄送人列表转换为逗号分隔的字符串
                String ccAddresses = String.join(",", dto.getCcEmailList());
                // 设置抄送人地址
                message.setRecipients(Message.RecipientType.CC, InternetAddress.parse(ccAddresses));
            }

            // 设置邮件主题
            message.setSubject(dto.getSubject());
            // 设置邮件发送时间（当前时间）
            message.setSentDate(new Date());

            // 创建邮件正文部分
            MimeBodyPart messageBodyPart = new MimeBodyPart();
            // 设置正文内容，指定HTML格式和UTF-8编码
            messageBodyPart.setContent(dto.getContent(), "text/html;charset=utf-8");

            // 创建多部分内容容器（用于组合正文和附件）
            Multipart multipart = new MimeMultipart();
            // 将正文部分添加到多部分容器中
            multipart.addBodyPart(messageBodyPart);

            // 判断是否有附件路径
            if (StringUtils.isNotBlank(dto.getAttachmentPath())) {
                // 按分号分割多个附件路径
                String[] attachmentPaths = dto.getAttachmentPath().split(";");
                for (String attachmentPath : attachmentPaths) {
                    // 去除路径两端的空格
                    attachmentPath = attachmentPath.trim();
                    // 检查路径是否非空
                    if (StringUtils.isNotBlank(attachmentPath)) {
                        try {
                            // 创建附件部分
                            MimeBodyPart attachmentPart = new MimeBodyPart();
                            // 创建文件数据源
                            DataSource source = new FileDataSource(attachmentPath);
                            // 设置附件的数据处理器
                            attachmentPart.setDataHandler(new DataHandler(source));
                            // 设置附件文件名（使用原文件名）
                            attachmentPart.setFileName(new File(attachmentPath).getName());
                            // 将附件部分添加到多部分容器中
                            multipart.addBodyPart(attachmentPart);
                            log.info("添加附件: {}", attachmentPath);
                        } catch (Exception e) {
                            log.error("添加附件失败: {}", attachmentPath, e);
                            // 可以选择继续处理其他附件，或者抛出异常
                            throw new RuntimeException("添加附件失败: " + attachmentPath, e);
                        }
                    }
                }
            }

            // 将多部分内容设置为邮件的完整内容
            message.setContent(multipart);

            // 发送邮件
            Transport.send(message);
            // 记录成功日志
            log.info("邮件发送成功: {} -> {}", dto.getSendEmail(), dto.getReceiveEmail());
//            log.info("发送邮件Transport.send(message) {}", JSON.toJSONString(message));

        } catch (MessagingException e) {
            // 记录失败日志，包含异常信息
            log.error("邮件发送失败: {} -> {}", dto.getSendEmail(), dto.getReceiveEmail(), e);
            // 抛出运行时异常
            throw new RuntimeException("邮件发送失败: " + e.getMessage());
        }
    }

}