package com.yd.email.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.DateUtil;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.email.api.async.ApiEmailSendAsyncService;
import com.yd.email.api.service.*;
import com.yd.email.feign.dto.ApiEmailSenderConfigDto;
import com.yd.email.feign.dto.ApiSendMailDto;
import com.yd.email.feign.enums.EmailTaskStatusEnum;
import com.yd.email.feign.request.ApiSendEmailRequest;
import com.yd.email.feign.response.ApiSendEmailResponse;
import com.yd.email.feign.utils.StringUtil;
import com.yd.email.service.model.EmailTask;
import com.yd.email.service.model.EmailTaskRecipients;
import com.yd.email.service.service.IEmailTaskService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import java.util.List;
import java.util.Objects;

/**
 * 邮件发送实现类
 */
@Service
@Slf4j
public class ApiEmailSendServiceImpl implements ApiEmailSendService {


    @Autowired
    private IEmailTaskService iEmailTaskService;

    @Autowired
    private ApiEmailSenderConfigService apiEmailSenderConfigService;

    @Autowired
    private ApiEmailService apiEmailService;

    @Autowired
    private ApiEmailTaskRecipientsService apiEmailTaskRecipientsService;

    @Autowired
    private ApiEmailSendAsyncService apiEmailSendAsyncService;

    @Autowired
    private ApiEmailContactService apiEmailContactService;

    /**
     * 发送邮件
     * @param request
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result<ApiSendEmailResponse> sendEmail(ApiSendEmailRequest request) {
        ApiSendEmailResponse response = new ApiSendEmailResponse();
        try {
            EmailTask mailTask = buildEmailTask(request);
            iEmailTaskService.saveOrUpdate(mailTask);

            //邮件任务唯一业务ID
            String taskBizId = mailTask.getTaskBizId();
            //变量分组唯一业务ID不为空，来源是1-excel导入联系人 为空是0-选择联系人导入
            Integer source = StringUtils.isNotBlank(request.getVariableGroupBizId()) ? 1 : 0;

            // 保存收件人信息
            //根据当前当前会话id和数据来源查询导入数据，保存到邮件任务收件人关联表中，供后续使用
            Result<List<EmailTaskRecipients>> result = apiEmailTaskRecipientsService
                    .saveEmailTaskRecipientsBySessionId(taskBizId,mailTask.getStatus(),request.getSessionId(),source);
            List<EmailTaskRecipients> recipientsList = result.getData();

            //计划发送时间（为空表示立即发送，不为空表示定时发送）
            String jobId = "";
            if (!Objects.isNull(request.getScheduleTime())) {
                //不为空表示定时发送：创建XXL-Job定时任务-异步方法
                apiEmailSendAsyncService.addScheduleJob(taskBizId, DateUtil.convertDateByLocalDateTime(request.getScheduleTime()));
            }else {
                //为空表示立即发送邮件-异步方法
                apiEmailSendAsyncService.sendImmediatelyEmail(mailTask, recipientsList);
            }
            response.setJobId(jobId);
            response.setScheduleTime(request.getScheduleTime());
            response.setTaskBizId(taskBizId);
            return Result.success(response);
        } catch (Exception e) {
            log.error("创建邮件发送任务失败", e);
            throw new BusinessException("发送邮件异常");
        }
    }



    /**
     * 测试发送邮件
     * @param request
     * @return
     */
    @Override
    public Result<ApiSendEmailResponse> testSendEmail(ApiSendEmailRequest request) {
        ApiSendEmailResponse response = new ApiSendEmailResponse();
        EmailTask mailTask = buildEmailTask(request);
        //邮件任务唯一业务ID
        String taskBizId = mailTask.getTaskBizId();
        //TODO 变量
        //立即发送->发送给自己。发件人和收件人都是自己
        //调用邮件服务发送邮件入参DTO
        ApiSendMailDto apiSendMailDto = new ApiSendMailDto();
        BeanUtils.copyProperties(mailTask,apiSendMailDto);
        //根据发件人唯一业务ID获取发件人信息和发件人绑定的服务商信息
        Result<ApiEmailSenderConfigDto> result = apiEmailSenderConfigService.getApiEmailSenderConfigDto(mailTask.getSenderBizId());
        ApiEmailSenderConfigDto apiEmailSenderConfigDto = result.getData();
        BeanUtils.copyProperties(apiEmailSenderConfigDto,apiSendMailDto);
        //收件人邮箱（单个）
        apiSendMailDto.setReceiveEmail(mailTask.getSendEmail());
        //抄送人邮箱列表
//        apiSendMailDto.setCcEmailList(ccEmailList);
        //变量分组唯一业务ID不为空，来源是1-excel导入联系人 为空是0-选择联系人导入
        Integer source = StringUtils.isNotBlank(request.getVariableGroupBizId()) ? 1 : 0;

        // 保存收件人信息
        //根据当前当前会话id和数据来源查询导入数据，保存到邮件任务收件人关联表中，供后续使用
        Result<List<EmailTaskRecipients>> result1 = apiEmailTaskRecipientsService
                .queryEmailTaskRecipientsBySessionId(taskBizId,mailTask.getStatus(),request.getSessionId(),source);
        List<EmailTaskRecipients> recipientsList = result1.getData();
        EmailTaskRecipients recipient = recipientsList.get(0);

        //邮件内容
        if (StringUtil.hasPlaceholdersSimple(mailTask.getContent())
                && StringUtils.isNotBlank(mailTask.getVariableGroupBizId())) {
            //检测内容是否有占位符，有占位符并且变量分组唯一业务ID不为空就替换具体内容,替换具体内容：recipient.getVariables()
            //变量替换内容（固定+非固定变量）
            String content = StringUtil.replacePlaceholders(mailTask.getContent(),
                    recipient.getVariables());
            apiSendMailDto.setContent(content);
        }else if (StringUtil.hasPlaceholdersSimple(mailTask.getContent())
                && StringUtils.isBlank(mailTask.getVariableGroupBizId())) {
            //检测内容是否有占位符，有占位符并且变量分组唯一业务ID为空就替换具体内容。替换具体内容：通用变量字段：收件人姓名、收件人称谓、公司，这三个字段内容从收件人联系人获取
            //检测内容是否含有通用变量
            if (StringUtil.hasTyVariables(mailTask.getContent())) {
                String content = apiEmailContactService.replacePlaceholders(mailTask.getContent(),recipient.getContactBizId(),recipient.getCcEmail());
                apiSendMailDto.setContent(content);
            }
        }

        // 调用邮件服务发送邮件
        try {
            apiEmailService.sendMail(apiSendMailDto);
        } catch (Exception e) {
            e.printStackTrace();
        }
        response.setScheduleTime(request.getScheduleTime());
        response.setTaskBizId(taskBizId);
        return Result.success(response);
    }


    /**
     * 构造邮箱任务对象
     * @param request
     * @return
     */
    public EmailTask buildEmailTask(ApiSendEmailRequest request){
        //保存邮件任务到数据库
        EmailTask mailTask = new EmailTask();
        //邮件任务唯一业务ID
        mailTask.setTaskBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_TASK.getCode()));
        //任务名称：邮件主题 + 邮件发送任务
        mailTask.setTaskName(request.getSubject() + "邮件发送任务");
        //关联发件人唯一业务ID
        mailTask.setSenderBizId(request.getSenderBizId());
        //发件人邮箱
        mailTask.setSendEmail(request.getSendEmail());
        //邮件主题
        mailTask.setSubject(request.getSubject());
        //邮件内容
        mailTask.setContent(request.getContent());
        //发送邮件的附件路径（多个用分号分隔）
        mailTask.setAttachmentPath(request.getAttachmentPath());
        //计划发送时间（为空表示立即发送，不为空表示定时发送）
        mailTask.setScheduleTime(request.getScheduleTime());
        //任务状态：计划发送时间（为空表示立即发送状态为发送中，不为空表示发送状态为定时发送）
        String taskStatus = !Objects.isNull(request.getScheduleTime()) ? EmailTaskStatusEnum.SCHEDULED.getItemValue() : EmailTaskStatusEnum.SENDING.getItemValue();
        mailTask.setStatus(taskStatus);
        //变量分组唯一业务ID
        mailTask.setVariableGroupBizId(request.getVariableGroupBizId());
        //邮件内容的填充的变量字段（数组分号分隔，英文字段名）
//            mailTask.setFillVariable(!CollectionUtils.isEmpty(request.getFillVariableList()) ? String.join(";") : "");
        //导入临时表的当前会话id（前端使用，用于查询。存于前端缓存）（冗余字段，选择联系人和导入联系人excel都有值）
        mailTask.setSessionId(request.getSessionId());
        return mailTask;
    }

}
