package com.yd.email.api.service.impl;

import com.alibaba.fastjson.JSON;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.common.utils.UUIDUtil;
import com.yd.email.api.service.ApiEmailContactCcService;
import com.yd.email.api.service.ApiEmailContactService;
import com.yd.email.api.service.ApiEmailFileService;
import com.yd.email.api.service.ApiEmailVariableGroupService;
import com.yd.email.feign.dto.ApiEmailVariableDto;
import com.yd.email.feign.enums.GeneralVariableEnum;
import com.yd.email.feign.enums.excel.ContactExcelEnum;
import com.yd.email.feign.request.ApiExportVariableExcelRequest;
import com.yd.email.feign.request.ApiImportVariableExcelRequest;
import com.yd.email.feign.response.ApiEmailVariableGroupDetailResponse;
import com.yd.email.feign.response.ApiExportVariableExcelResponse;
import com.yd.email.feign.response.ApiImportContactExcelResponse;
import com.yd.email.feign.response.ApiImportVariableExcelResponse;
import com.yd.email.service.dto.EmailContactDto;
import com.yd.email.service.dto.EmailVariableDto;
import com.yd.email.service.model.EmailContact;
import com.yd.email.service.model.EmailContactImport;
import com.yd.email.service.model.EmailVariable;
import com.yd.email.service.service.IEmailContactImportService;
import com.yd.email.service.service.IEmailContactService;
import com.yd.email.service.service.IEmailVariableService;
import com.yd.oss.feign.client.ApiExcelFeignClient;
import com.yd.oss.feign.dto.ExportParam;
import com.yd.oss.feign.dto.ExportResult;
import com.yd.oss.feign.request.ApiExportRequest;
import com.yd.oss.feign.result.ImportResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import org.springframework.web.multipart.MultipartFile;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiEmailFileServiceImpl implements ApiEmailFileService {

    @Autowired
    private ApiExcelFeignClient apiExcelFeignClient;

    @Autowired
    private ApiEmailVariableGroupService apiEmailVariableGroupService;

    @Autowired
    private IEmailVariableService iEmailVariableService;

    @Autowired
    private IEmailContactImportService iEmailContactImportService;

    @Autowired
    private IEmailContactService iEmailContactService;

    @Autowired
    private ApiEmailContactCcService apiEmailContactCcService;

    @Autowired
    private ApiEmailContactService apiEmailContactService;

    // 固定表头字段-通用变量
    private static final Set<String> FIXED_HEADERS = new HashSet<>(Arrays.asList(
            GeneralVariableEnum.GENERAL_CONTACT.getItemLabel(),
            GeneralVariableEnum.GENERAL_EMAIL.getItemLabel(),
            GeneralVariableEnum.GENERAL_CC_EMAIL.getItemLabel(),
            GeneralVariableEnum.GENERAL_APPELLATION.getItemLabel(),
            GeneralVariableEnum.GENERAL_COMPANTNAME.getItemLabel()));

    /**
     * 导出-变量excel
     * @param request
     * @return
     */
    @Override
    public Result<ApiExportVariableExcelResponse> exportVariableExcel(ApiExportVariableExcelRequest request) {
        Result<ApiEmailVariableGroupDetailResponse> result = apiEmailVariableGroupService.detail(request.getVariableGroupBizId());
        ApiEmailVariableGroupDetailResponse response = result.getData();
        //分组绑定的变量列表
        List<ApiEmailVariableDto> emailVariableDtoList = response.getEmailVariableDtoList();
        List<String> FieldNameList = new ArrayList<>(FIXED_HEADERS);
        if (!CollectionUtils.isEmpty(emailVariableDtoList)) {
            FieldNameList.addAll(emailVariableDtoList.stream().map(ApiEmailVariableDto::getVariableNameCn).collect(Collectors.toList()));
        }

        //构建导出参数
        ExportParam exportParam = new ExportParam();
        exportParam.setFileName("变量"); // 基础文件名
        exportParam.setFieldNames(FieldNameList); // 要导出的字段
        exportParam.setUploadToOss(true); // 设置为true表示上传到OSS
        ApiExportRequest exportRequest = new ApiExportRequest();
        exportRequest.setExportParam(exportParam);
        //远程调用OSS的Feign，通用-Excel导出参数（如果传dataList附加数据，不传单纯导出表头参数）
        Result<ExportResult> resultResult = apiExcelFeignClient.export(exportRequest);
        ApiExportVariableExcelResponse response1 = new ApiExportVariableExcelResponse();
        if (!Objects.isNull(resultResult.getData())) {
            response1.setUrl(resultResult.getData().getOssUrl());
        }
        return Result.success(response1);
    }



    /**
     * 导入-变量excel
     * @param file
     * @param request
     * @return
     */
    @Override
    public Result<ApiImportVariableExcelResponse> importVariableExcel(MultipartFile file,
                                                                      ApiImportVariableExcelRequest request) {
        ApiImportVariableExcelResponse response = new ApiImportVariableExcelResponse();
        String sessionId = request.getSessionId();
        if (StringUtils.isBlank(request.getSessionId())) {
            sessionId = UUIDUtil.randomUUIDWithoutHyphen();
        }
        response.setSessionId(sessionId);

        try {
            // 远程调用Feign，通用Excel导入接口
            Result<ImportResult> result = apiExcelFeignClient.importExcel(file, 0, 0, null);
            ImportResult importResult = result.getData();
            if (Objects.isNull(result.getData()) || (!Objects.isNull(result.getData()) && !importResult.isSuccess())) {
                return Result.fail("Excel导入失败: " + (importResult.getMessage() != null ? importResult.getMessage() : "未知错误"));
            }

            if (!importResult.isSuccess()) {
                return Result.fail("Excel解析失败: " + importResult.getMessage());
            }

            log.info("表头信息: {}", importResult.getHeaders());
            log.info("导入数据条数: {}", importResult.getTotalCount());

            //导入-变量excel-校验excel字段
            processImportDataCheck(importResult.getData());

            //根据表头字段（变量字段）中文名查询对应的英文名，返回集合（固定+非固定）
            List<EmailVariable> emailVariableList = getEmailVariableList(importResult.getHeaders());

            // 处理每一行数据
            List<EmailContactImport> contactImports = processImportData(importResult.getData(),
                    importResult.getHeaders(),
                    sessionId,
                    emailVariableList);

            // 批量保存到数据库
            if (!CollectionUtils.isEmpty(contactImports)) {
                boolean saveResult = iEmailContactImportService.saveOrUpdateBatch(contactImports);
                if (saveResult) {
                    response.setImportCount(contactImports.size());
                    log.info("成功导入 {} 条联系人数据", contactImports.size());
                } else {
                    return Result.fail("数据保存失败");
                }
            } else {
                response.setImportCount(0);
                log.warn("Excel中未找到有效数据");
            }

            return Result.success(response);

        } catch (Exception e) {
            log.error("导入变量Excel异常", e);
            return Result.fail("导入失败: " + e.getMessage());
        }
    }

    /**
     * 获取变量字段（排除固定字段）
     * @param allHeaders
     * @return
     */
    private List<String> getVariableHeaders(List<String> allHeaders) {
        List<String> variableHeaders = new ArrayList<>();
        if (CollectionUtils.isEmpty(allHeaders)) {
            return variableHeaders;
        }

        for (String header : allHeaders) {
            if (!FIXED_HEADERS.contains(header)) {
                variableHeaders.add(header);
            }
        }
        return variableHeaders;
    }

    /**
     * 根据表头字段（变量字段）中文名查询对应的英文名，返回集合
     * @param variableHeaders
     * @return
     */
    public List<EmailVariable> getEmailVariableList(List<String> variableHeaders) {
        List<EmailVariable> emailVariableList = iEmailVariableService.queryList(EmailVariableDto
                .builder().variableNameCnList(variableHeaders).build());
        return emailVariableList;
    }

    /**
     * 导入-变量excel-处理导入数据
     * @param data
     * @param headers
     * @param sessionId
     * @param emailVariableList
     * @return
     */
    private List<EmailContactImport> processImportData(List<Map<String, Object>> data,
                                                       List<String> headers,
                                                       String sessionId,
                                                       List<EmailVariable> emailVariableList) {
        List<EmailContactImport> contactImports = new ArrayList<>();

        if (CollectionUtils.isEmpty(data)) {
            return contactImports;
        }

        for (Map<String, Object> row : data) {
            EmailContactImport contactImport = new EmailContactImport();

            // 设置固定字段
            contactImport.setImportBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_CONTACT_IMPORT.getCode()));
            contactImport.setSessionId(sessionId);
            contactImport.setReceiveName(getStringValue(row, GeneralVariableEnum.GENERAL_CONTACT.getItemLabel()));
            contactImport.setReceiveEmail(getStringValue(row, GeneralVariableEnum.GENERAL_EMAIL.getItemLabel()));
            contactImport.setCcEmail(getStringValue(row, GeneralVariableEnum.GENERAL_CC_EMAIL.getItemLabel()));

            // 处理变量数据，转换为JSON格式
            String variablesJson = processVariableData(row,headers,emailVariableList);
            contactImport.setVariables(variablesJson);

            // 设置其他字段
            //1-excel导入联系人
            contactImport.setSource(1);
            contactImports.add(contactImport);
        }

        return contactImports;
    }

    /**
     * 导入-变量excel-校验excel字段
     * @param data
     * @return
     */
    public Result processImportDataCheck(List<Map<String, Object>> data) {
        Set<String> emailSet = new HashSet<>();
        int i = 1;
        for (Map<String, Object> row : data) {
            String email = getStringValue(row, GeneralVariableEnum.GENERAL_EMAIL.getItemLabel());

//            if (StringUtils.isBlank(getStringValue(row, GeneralVariableEnum.GENERAL_CONTACT.getItemLabel()))) {
//                throw new BusinessException("第" + i + "行的" + GeneralVariableEnum.GENERAL_CONTACT.getItemLabel() + "不能为空！");
//            }
            if (StringUtils.isBlank(getStringValue(row, GeneralVariableEnum.GENERAL_EMAIL.getItemLabel()))) {
                throw new BusinessException("第" + i + "行的" + GeneralVariableEnum.GENERAL_EMAIL.getItemLabel() + "不能为空！");
            }
//            if (StringUtils.isBlank(getStringValue(row, GeneralVariableEnum.GENERAL_CC_EMAIL.getItemLabel()))) {
//                throw new BusinessException("第" + i + "行的" + GeneralVariableEnum.GENERAL_CC_EMAIL.getItemLabel() + "不能为空！");
//            }
//            if (StringUtils.isBlank(getStringValue(row, GeneralVariableEnum.GENERAL_APPELLATION.getItemLabel()))) {
//                throw new BusinessException("第" + i + "行的" + GeneralVariableEnum.GENERAL_APPELLATION.getItemLabel() + "不能为空！");
//            }
//            if (StringUtils.isBlank(getStringValue(row, GeneralVariableEnum.GENERAL_COMPANTNAME.getItemLabel()))) {
//                throw new BusinessException("第" + i + "行的" + GeneralVariableEnum.GENERAL_COMPANTNAME.getItemLabel() + "不能为空！");
//            }
            // 检查收件人邮箱是否重复
            if (emailSet.contains(email)) {
                throw new BusinessException("第" + i + "行的" + GeneralVariableEnum.GENERAL_EMAIL.getItemLabel() + "重复：" + email);
            }
            emailSet.add(email);
            i++;
        }
        return Result.success();
    }

    /**
     * 处理变量数据，转换为JSON格式
     * @param row
     * @param headers
     * @param emailVariableList
     * @return
     */
    private String processVariableData(Map<String, Object> row,
                                       List<String> headers,
                                       List<EmailVariable> emailVariableList) {
        List<Map<String, String>> variableList = new ArrayList<>();

        if (CollectionUtils.isEmpty(emailVariableList)) {
            return "";
        }

        for (String header : headers) {
            String value = getStringValue(row, header);
            Map<String, String> variableMap = new HashMap<>();
            // 中文变量字段名
            variableMap.put("cn_key", header);
            // 英文变量字段名（需要转换，根据中文名查变量表转换）
            variableMap.put("en_key", convertToEnKey(header,emailVariableList));
            // 占位符内容
            variableMap.put("content", value);
            variableList.add(variableMap);
        }

        return CollectionUtils.isEmpty(variableList) ? null : JSON.toJSONString(variableList);
    }

    /**
     * 将中文字段名转换为英文键名（根据表返回的数据对应转换）
     * @param cnKey 中文变量字段名
     * @param emailVariableList 变量列表
     * @return
     */
    public String convertToEnKey(String cnKey, List<EmailVariable> emailVariableList) {
        if (StringUtils.isBlank(cnKey) || CollectionUtils.isEmpty(emailVariableList)) {
            return "";
        }

        return emailVariableList.stream()
                .filter(dto -> StringUtils.equals(dto.getVariableNameCn(), cnKey))
                .findFirst()
                .map(EmailVariable::getVariableNameEn) // 假设EmailVariable类中有getVariableNameEn方法
                .orElseGet(() -> {
                    // 如果找不到对应的英文键名，可以返回一个默认值或进行转换
                    log.warn("未找到中文变量名 '{}' 对应的英文键名", cnKey);
                    throw new BusinessException("变量表未找到中文变量名："+cnKey+"，请检查！");
                });
    }

    /**
     * 安全获取字符串值
     * @param row
     * @param key
     * @return
     */
    private String getStringValue(Map<String, Object> row, String key) {
        if (row == null || !row.containsKey(key)) {
            return null;
        }

        Object value = row.get(key);
        return value != null ? value.toString().trim() : null;
    }

    /**
     * 导入-联系人excel
     * @param file
     * @return
     */
    @Override
    public Result importContactExcel(MultipartFile file) {
        try {
            // 远程调用Feign，通用Excel导入接口
            Result<ImportResult> result = apiExcelFeignClient.importExcel(file, 0, 0, null);
            ImportResult importResult = result.getData();
            if (Objects.isNull(result.getData()) || (!Objects.isNull(result.getData()) && !importResult.isSuccess())) {
                return Result.fail("Excel导入失败: " + (importResult.getMessage() != null ? importResult.getMessage() : "未知错误"));
            }

            if (!importResult.isSuccess()) {
                return Result.fail("Excel解析失败: " + importResult.getMessage());
            }

            log.info("表头信息: {}", importResult.getHeaders());
            log.info("导入数据条数: {}", importResult.getTotalCount());

            //导入-联系人excel-处理导入数据（处理每一行数据（包括校验））
            processImportContactExcelData(importResult.getData());

            return Result.success();

        } catch (Exception e) {
            log.error("导入变量Excel异常", e);
            return Result.fail("导入失败: " + e.getMessage());
        }
    }

    /**
     * 导入-联系人excel-处理导入数据（处理每一行数据（包括校验））
     * @param data
     * @return
     */
    public Result processImportContactExcelData(List<Map<String, Object>> data) {
        //获取Security上下文当前用户的登录信息
        AuthUserDto authUserDto = SecurityUtil.getCurrentLoginUser();
        //导入-联系人excel-处理导入数据（处理每一行数据（包括校验））-校验excel字段
        processImportContactExcelDataCheck(data);

        for (Map<String, Object> row : data) {
            EmailContact emailContact = new EmailContact();
            String ccEmail = getStringValue(row, ContactExcelEnum.CC_EMAIL.getItemLabel());
            List<String> ccEmailList = StringUtils.isNotBlank(ccEmail) ? Arrays.asList(ccEmail.split(";").clone()) : new ArrayList<>();
            //设置联系人（收件人）业务唯一ID
            emailContact.setContactBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_CONTACT.getCode()));
            emailContact.setName(getStringValue(row, ContactExcelEnum.NAME.getItemLabel()));
            emailContact.setAppellation(getStringValue(row, ContactExcelEnum.APPELLATION.getItemLabel()));
            emailContact.setCompanyName(getStringValue(row, ContactExcelEnum.COMPANY_NAME.getItemLabel()));
            emailContact.setEmail(getStringValue(row, ContactExcelEnum.EMAIL.getItemLabel()));
            emailContact.setType(getStringValue(row, ContactExcelEnum.TYPE.getItemLabel()));
            emailContact.setCreatorId(authUserDto.getUserBizId());

            //设置联系人（收件人）业务唯一ID
            emailContact.setContactBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EMAIL_CONTACT.getCode()));
            iEmailContactService.saveOrUpdate(emailContact);

            //保存关联抄送邮箱的列表
            apiEmailContactCcService.saveEmailContactCcList(apiEmailContactService.buildCcEmailList(ccEmailList,emailContact.getContactBizId()),emailContact.getContactBizId());
        }

        return Result.success();
    }

    /**
     * 导入-联系人excel-处理导入数据（处理每一行数据（包括校验））-校验excel字段
     * @param data
     * @return
     */
    public Result processImportContactExcelDataCheck(List<Map<String, Object>> data) {
        Set<String> emailSet = new HashSet<>();
        int i = 2;
        for (Map<String, Object> row : data) {
            String email = getStringValue(row, ContactExcelEnum.EMAIL.getItemLabel());

            if (StringUtils.isBlank(getStringValue(row, ContactExcelEnum.NAME.getItemLabel()))) {
                throw new BusinessException("第" + i + "行的" + ContactExcelEnum.NAME.getItemLabel() + "不能为空！");
            }
            if (StringUtils.isBlank(getStringValue(row, ContactExcelEnum.APPELLATION.getItemLabel()))) {
                throw new BusinessException("第" + i + "行的" + ContactExcelEnum.APPELLATION.getItemLabel() + "不能为空！");
            }
            if (StringUtils.isBlank(getStringValue(row, ContactExcelEnum.COMPANY_NAME.getItemLabel()))) {
                throw new BusinessException("第" + i + "行的" + ContactExcelEnum.COMPANY_NAME.getItemLabel() + "不能为空！");
            }
            if (StringUtils.isBlank(getStringValue(row, ContactExcelEnum.EMAIL.getItemLabel()))) {
                throw new BusinessException("第" + i + "行的" + ContactExcelEnum.EMAIL.getItemLabel() + "不能为空！");
            }
            if (StringUtils.isBlank(getStringValue(row, ContactExcelEnum.CC_EMAIL.getItemLabel()))) {
                throw new BusinessException("第" + i + "行的" + ContactExcelEnum.CC_EMAIL.getItemLabel() + "不能为空！");
            }
            // 检查收件人邮箱是否重复
            if (emailSet.contains(email)) {
                throw new BusinessException("第" + i + "行的" + ContactExcelEnum.EMAIL.getItemLabel() + "重复：" + email);
            }
            emailSet.add(email);
            i++;
        }
        //根据收件人邮箱查询库里有没有重复数据
        List<EmailContact> emailContactList = iEmailContactService.queryList(EmailContactDto.builder().emailList(new ArrayList<>(emailSet)).build());
        if (!CollectionUtils.isEmpty(emailContactList)) {
            //以下邮箱在库里重复
            throw new BusinessException("以下收件人邮箱在库里重复：" + String.join(",",emailContactList.stream().map(EmailContact::getEmail).collect(Collectors.toList())));
        }
        return Result.success();
    }
}
