package com.yd.email.api.async;

import com.alibaba.fastjson.JSON;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.DateUtil;
import com.yd.email.api.service.*;
import com.yd.email.feign.dto.ApiEmailSenderConfigDto;
import com.yd.email.feign.dto.ApiSendMailDto;
import com.yd.email.feign.enums.EmailTaskStatusEnum;
import com.yd.email.feign.utils.StringUtil;
import com.yd.email.service.model.EmailTask;
import com.yd.email.service.model.EmailTaskRecipients;
import com.yd.email.service.service.IEmailTaskRecipientsService;
import com.yd.email.service.service.IEmailTaskService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Component;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

@Slf4j
@Component
public class ApiEmailSendAsyncService {

    @Autowired
    private IEmailTaskService iEmailTaskService;

    @Autowired
    private IEmailTaskRecipientsService iEmailTaskRecipientsService;

    @Autowired
    private ApiEmailSenderConfigService apiEmailSenderConfigService;

    @Autowired
    private ApiEmailService apiEmailService;

    @Autowired
    private ApiEmailContactService apiEmailContactService;

    @Autowired
    private XxlJobService xxlJobService;

    /**
     * 创建XXL-Job定时任务-异步方法
     * @param taskBizId
     * @param scheduleTime
     */
    @Async("emailTaskExecutor")
    public void addScheduleJob(String taskBizId, Date scheduleTime) {
        String jobId = xxlJobService.addScheduleJob(taskBizId, scheduleTime);
        log.info("创建XXL-Job定时任务成功-->jobId：{}",jobId);
    }

    /**
     * 立即发送邮件-异步方法
     * @param emailTask
     * @param recipients
     * @return
     */
    @Async("emailTaskExecutor")
    public void sendImmediatelyEmail(EmailTask emailTask, List<EmailTaskRecipients> recipients) {
        //主任务发送时间
        LocalDateTime taskSendTime = LocalDateTime.now();
        try {
            //调用邮件服务发送邮件入参DTO
            ApiSendMailDto apiSendMailDto = new ApiSendMailDto();
            BeanUtils.copyProperties(emailTask,apiSendMailDto);
            //根据发件人唯一业务ID获取发件人信息和发件人绑定的服务商信息
            Result<ApiEmailSenderConfigDto> result = apiEmailSenderConfigService.getApiEmailSenderConfigDto(emailTask.getSenderBizId());
            ApiEmailSenderConfigDto apiEmailSenderConfigDto = result.getData();
            BeanUtils.copyProperties(apiEmailSenderConfigDto,apiSendMailDto);

            //该任务下的所有收件人信息
            log.info("查询该任务下的所有收件人信息: {}: " + JSON.toJSONString(recipients));

            // 初始化成功和失败计数器
            int successCount = 0;
            int failCount = 0;

            // 遍历所有收件人，逐个发送邮件
            for (EmailTaskRecipients recipient : recipients) {
                try {
                    log.info("遍历所有收件人，逐个发送邮件->recipient单个对象：{}",JSON.toJSONString(recipient));
                    // 添加发送间隔，避免频率过高
                    if (successCount > 0) {
                        Thread.sleep(1000); // 1秒间隔
                    }

                    // 处理抄送人列表：将数据库中的逗号分隔字符串转换为List
                    List<String> ccEmailList = StringUtils.isNotBlank(recipient.getCcEmail()) ?
                            Arrays.asList(recipient.getCcEmail().split(";")) :
                            new ArrayList<>();
                    log.info("处理抄送人列表：将数据库中的逗号分隔字符串转换为List: {}: " + JSON.toJSONString(ccEmailList));
                    //收件人邮箱（单个）
                    apiSendMailDto.setReceiveEmail(recipient.getReceiveEmail());
                    //抄送人邮箱列表
                    apiSendMailDto.setCcEmailList(ccEmailList);

                    //邮件内容
                    if (StringUtil.hasPlaceholdersSimple(emailTask.getContent())
                            && StringUtils.isNotBlank(emailTask.getVariableGroupBizId())) {
                        //检测内容是否有占位符，有占位符并且变量分组唯一业务ID不为空就替换具体内容,替换具体内容：recipient.getVariables()
                        //非通用变量替换内容
                        String content = StringUtil.replacePlaceholders(emailTask.getContent(),
                                recipient.getVariables());
//                        //通用变量替换内容
//                        //检测内容是否含有通用变量
//                        if (StringUtil.hasTyVariables(emailTask.getContent())) {
//                            content = apiEmailContactService.replacePlaceholders(content,recipient.getContactBizId());
//                        }
                        apiSendMailDto.setContent(content);
                    }else if (StringUtil.hasPlaceholdersSimple(emailTask.getContent())
                            && StringUtils.isBlank(emailTask.getVariableGroupBizId())) {
                        //检测内容是否有占位符，有占位符并且变量分组唯一业务ID为空就替换具体内容。替换具体内容：通用变量字段：收件人姓名、收件人称谓、公司，这三个字段内容从收件人联系人获取
                        //检测内容是否含有通用变量
                        if (StringUtil.hasTyVariables(emailTask.getContent())) {
                            String content = apiEmailContactService.replacePlaceholders(emailTask.getContent(),recipient.getContactBizId(),recipient.getCcEmail());
                            apiSendMailDto.setContent(content);
                        }
                    }

                    // 调用邮件服务发送邮件
                    apiEmailService.sendMail(apiSendMailDto);

                    // 发送成功：更新收件人状态为成功
                    recipient.setStatus(EmailTaskStatusEnum.SUCCESSFUL.getItemValue());
                    recipient.setSendTime(LocalDateTime.now());  // 记录实际发送时间
                    iEmailTaskRecipientsService.saveOrUpdate(recipient);
                    successCount++;  // 成功计数加1

                } catch (Exception e) {
                    // 发送失败：记录错误日志
                    log.error("发送邮件失败: {}", recipient.getReceiveEmail(), e);
                    // 更新收件人状态为失败
                    recipient.setStatus(EmailTaskStatusEnum.FAILED.getItemValue());
                    recipient.setErrorMsg(e.getMessage());  // 保存错误信息
                    iEmailTaskRecipientsService.saveOrUpdate(recipient);
                    failCount++;  // 失败计数加1
                }
            }

            // 根据发送结果更新邮件任务状态
            emailTask.setStatus(failCount == 0 ?
                    EmailTaskStatusEnum.ALL_SUCCESSFUL.getItemValue() :
                    EmailTaskStatusEnum.PARTIAL_FAILURE.getItemValue());
            //主任务发送时间
            emailTask.setSendTime(taskSendTime);
            iEmailTaskService.saveOrUpdate(emailTask);

            // 记录任务完成日志
            log.info("邮件发送任务完成: 成功{}个, 失败{}个", successCount, failCount);

            // 设置任务执行结果
            if (failCount == 0) {
                log.info("发送完成: 成功" + successCount + "个");
            } else {
                log.info("发送完成: 成功" + successCount + "个, 失败" + failCount + "个");
            }

        } catch (Exception e) {
            // 任务执行过程中发生异常
            log.error("邮件发送任务执行异常", e);
            // 更新任务状态为全部发送失败
            emailTask.setStatus(EmailTaskStatusEnum.ALL_FAILED.getItemValue());
            iEmailTaskService.saveOrUpdate(emailTask);
            // 返回任务执行异常信息
            log.error("任务执行异常: " + e.getMessage());
            throw new BusinessException("立即发送邮件异常");
        }
    }
}
