package com.yd.csf.service.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.*;
import com.yd.csf.service.enums.PolicyFollowStatusEnum;
import com.yd.csf.service.model.Policy;
import com.yd.csf.service.model.PolicyBroker;
import com.yd.csf.service.model.PolicyFollow;
import com.yd.csf.service.model.PolicyFollowRecord;
import com.yd.csf.service.service.PolicyBrokerService;
import com.yd.csf.service.service.PolicyFollowRecordService;
import com.yd.csf.service.service.PolicyFollowService;
import com.yd.csf.service.dao.PolicyFollowMapper;
import com.yd.csf.service.service.PolicyService;
import com.yd.csf.service.vo.PolicyFollowVO;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

/**
* @author Zhang Jianan
* @description 针对表【policy_follow(新单跟进表)】的数据库操作Service实现
* @createDate 2025-09-16 18:18:06
*/
@Service
public class PolicyFollowServiceImpl extends ServiceImpl<PolicyFollowMapper, PolicyFollow>
    implements PolicyFollowService{

    @Resource
    private PolicyBrokerService policyBrokerService;
    @Resource
    private PolicyService policyService;
    @Resource
    private PolicyFollowService policyFollowService;
    @Resource
    private PolicyFollowRecordService policyFollowRecordService;


    @Override
    public PolicyFollow getByPolicyBizId(String policyBizId) {
        return this.getOne(new QueryWrapper<PolicyFollow>().eq("policy_biz_id", policyBizId));
    }

    @Override
    public QueryWrapper<PolicyFollow> getQueryWrapper(PolicyFollowQueryRequest policyFollowQueryRequest) {
        QueryWrapper<PolicyFollow> queryWrapper = new QueryWrapper<>();
        if (policyFollowQueryRequest == null) {
            return queryWrapper;
        }
        String policyNo = policyFollowQueryRequest.getPolicyNo();
        String customerName = policyFollowQueryRequest.getCustomerName();
        String customerBizId = policyFollowQueryRequest.getCustomerBizId();
        String status = policyFollowQueryRequest.getStatus();
        String insurer = policyFollowQueryRequest.getInsurer();
        String productName = policyFollowQueryRequest.getProductName();
        Date signDateStart = policyFollowQueryRequest.getSignDateStart();
        Date signDateEnd = policyFollowQueryRequest.getSignDateEnd();

        queryWrapper.eq(ObjectUtils.isNotEmpty(policyNo), "policy_no", policyNo);
        queryWrapper.eq(ObjectUtils.isNotEmpty(customerName), "customer_name", customerName);
        queryWrapper.eq(ObjectUtils.isNotEmpty(customerBizId), "customer_biz_id", customerBizId);
        queryWrapper.eq(ObjectUtils.isNotEmpty(status), "status", status);
        queryWrapper.eq(ObjectUtils.isNotEmpty(insurer), "insurance", insurer);
        queryWrapper.eq(ObjectUtils.isNotEmpty(productName), "product_code", productName);
        queryWrapper.ge(ObjectUtils.isNotEmpty(signDateStart), "sign_date", signDateStart);
        queryWrapper.le(ObjectUtils.isNotEmpty(signDateEnd), "sign_date", signDateEnd);

        // 排序
        queryWrapper.orderByDesc("id");

        return queryWrapper;
    }

    @Override
    public Page<PolicyFollowVO> getPolicyFollowVOPage(Page<PolicyFollow> policyFollowPage) {
        List<PolicyFollow> policyFollowList = policyFollowPage.getRecords();
        Page<PolicyFollowVO> policyFollowVOPage = new Page<>(policyFollowPage.getCurrent(), policyFollowPage.getSize(), policyFollowPage.getTotal());
        if (CollUtil.isEmpty(policyFollowList)) {
            return policyFollowVOPage;
        }
        // 对象列表 => 封装对象列表
        List<PolicyFollowVO> policyFollowVOList = policyFollowList.stream().map(PolicyFollowVO::objToVo).collect(Collectors.toList());

        // 关联查询转介人
        List<String> policyBizIdList = policyFollowVOList.stream().map(PolicyFollowVO::getPolicyBizId).collect(Collectors.toList());
        if (CollUtil.isNotEmpty(policyBizIdList)) {
            // 转介人业务id列表 => 转介人列表
            List<PolicyBroker> policyBrokerList = policyBrokerService.list(new QueryWrapper<PolicyBroker>().in("policy_biz_id", policyBizIdList));
            // 转介人列表 => 转介人业务id列表
            Map<String, List<PolicyBroker>> brokerMap = policyBrokerList.stream().collect(Collectors.groupingBy(PolicyBroker::getPolicyBizId));
            // 转介人列表 => 封装转介人列表
            policyFollowVOList.forEach(policyFollowVO -> {
                List<PolicyBroker> brokerList = brokerMap.get(policyFollowVO.getPolicyBizId());
                policyFollowVO.setBrokerList(brokerList);
            });
        }

        return policyFollowVOPage.setRecords(policyFollowVOList);
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updatePolicyFollow(PolicyFollowUpdateRequest policyFollowUpdateRequest) {
        if (policyFollowUpdateRequest == null || ObjectUtils.isEmpty(policyFollowUpdateRequest.getPolicyBizId())) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        String policyBizId = policyFollowUpdateRequest.getPolicyBizId();
        PolicyFollow policyFollow = getByPolicyBizId(policyBizId);
        if (policyFollow == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }
        BeanUtils.copyProperties(policyFollowUpdateRequest, policyFollow, "id", "policyBizId");

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();
        policyFollow.setUpdaterId(loginUserId);
        policyFollow.setUpdateTime(new Date());

        // 先删除原有的转介人
        policyBrokerService.remove(new QueryWrapper<PolicyBroker>().eq("policy_biz_id", policyBizId));
        // 保存转介人
        List<PolicyBrokerDto> brokerList = policyFollowUpdateRequest.getBrokerList();
        List<PolicyBroker> policyBrokerList = brokerList.stream().map(policyBrokerDto -> {
            PolicyBroker policyBroker = new PolicyBroker();
            BeanUtils.copyProperties(policyBrokerDto, policyBroker);
            policyBroker.setPolicyBizId(policyBizId);
            policyBroker.setCreatorId(loginUserId);
            policyBroker.setCreateTime(new Date());
            policyBroker.setUpdaterId(loginUserId);
            policyBroker.setUpdateTime(new Date());
            return policyBroker;
        }).collect(Collectors.toList());

        if (CollUtil.isNotEmpty(policyBrokerList)) {
            policyBrokerService.saveBatch(policyBrokerList);
        }
        return updateById(policyFollow);
    }

    @Override
    public PolicyFollowVO getPolicyFollowVO(PolicyFollow policyFollow) {
        if (policyFollow == null) {
            return null;
        }
        PolicyFollowVO policyFollowVO = PolicyFollowVO.objToVo(policyFollow);
        // 关联查询转介人
        QueryWrapper<PolicyBroker> queryWrapper = new QueryWrapper<>();
        queryWrapper.eq("policy_biz_id", policyFollow.getPolicyBizId());
        List<PolicyBroker> policyBrokerList = policyBrokerService.list(queryWrapper);
        policyFollowVO.setBrokerList(policyBrokerList);

        return policyFollowVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Map<String, Object> addPolicyFollow(PolicyFollowAddRequest policyFollowAddRequest) {
        PolicyFollow policyFollow = new PolicyFollow();
        BeanUtils.copyProperties(policyFollowAddRequest, policyFollow);
        // 生成新单编号
        String policyBizId = RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_POLICY.getCode());
        policyFollow.setPolicyBizId(policyBizId);
        // 创建 Policy对象
        Policy policy = new Policy();
        BeanUtils.copyProperties(policyFollowAddRequest, policy);
        policy.setPolicyBizId(policyBizId);
        // 保存转介人
        List<PolicyBroker> policyBrokerList = policyFollowAddRequest.getBrokerList();
        if (CollUtil.isNotEmpty(policyBrokerList)) {
            policyBrokerList.forEach(policyBroker -> {
                policyBroker.setPolicyBizId(policyBizId);
            });
        }

        this.save(policyFollow);
        policyService.save(policy);
        policyBrokerService.saveBatch(policyBrokerList);

        // 返回新单跟进唯一业务ID
        return Collections.singletonMap("policyBizId", policyBizId);
    }

    @Override
    public Boolean addToPolicy(AddToPolicyRequest addToPolicyRequest) {
        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        List<String> policyNoList = addToPolicyRequest.getPolicyNoList();
        // 查询所有跟进记录
        List<PolicyFollow> policyFollowList = list(new QueryWrapper<PolicyFollow>().in("policy_no", policyNoList));
        for (PolicyFollow policyFollow : policyFollowList) {
            // 查询保单是否存在
            String policyBizId = policyFollow.getPolicyBizId();
            Policy policy = policyService.getOne(new QueryWrapper<Policy>().eq("policy_biz_id", policyBizId));
            if (policy != null) {
                // 更新
                BeanUtils.copyProperties(policyFollow, policy, "id");
                policy.setUpdaterId(loginUserId);
                policy.setUpdateTime(new Date());

                policyService.updateById(policy);
            } else {
                // 不存在则创建
                policy = new Policy();
                BeanUtils.copyProperties(policyFollow, policy, "id");
                policy.setId(null);
                policy.setPolicyBizId(policyBizId);
                policy.setCreatorId(loginUserId);
                policy.setCreateTime(new Date());
                policy.setUpdaterId(loginUserId);
                policy.setUpdateTime(new Date());

                policyService.save(policy);
            }
        }
        return true;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean changePolicyFollowStatus(ChangePolicyFollowStatusRequest changePolicyFollowStatusRequest) {

        String policyBizId = changePolicyFollowStatusRequest.getPolicyBizId();
        PolicyFollow policyFollow = getByPolicyBizId(policyBizId);
        if (policyFollow == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }
        // 校验状态是否存在
        PolicyFollowStatusEnum policyFollowStatusEnum = PolicyFollowStatusEnum.getEnumByValue(changePolicyFollowStatusRequest.getStatus());
        if (policyFollowStatusEnum == null) {
            throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "跟进状态不存在");
        }
        // 获取当前跟进状态
        PolicyFollowStatusEnum currentStatusEnum = PolicyFollowStatusEnum.getEnumByValue(policyFollow.getStatus());
        // 校验状态
        validPolicyFollowStatus(currentStatusEnum, policyFollowStatusEnum);

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        policyFollow.setStatus(changePolicyFollowStatusRequest.getStatus());
        policyFollow.setUpdaterId(loginUserId);
        policyFollowService.updateById(policyFollow);

        // 新增新单状态记录
        PolicyFollowRecord policyFollowRecord = new PolicyFollowRecord();
        policyFollowRecord.setId(null);
        policyFollowRecord.setPolicyBizId(policyBizId);
        policyFollowRecord.setTitle(Objects.requireNonNull(policyFollowStatusEnum.getItemLabel()));
        policyFollowRecord.setMessage(changePolicyFollowStatusRequest.getMessage());
        policyFollowRecord.setCreatorId(loginUserId);
        policyFollowRecord.setCreateTime(new Date());
        policyFollowRecordService.save(policyFollowRecord);

        return true;
    }

    @Override
    public Boolean uploadAttachment(AttachmentUploadRequest attachmentUploadRequest) {
        String policyBizId = attachmentUploadRequest.getPolicyBizId();
        PolicyFollow policyFollow = getByPolicyBizId(policyBizId);
        if (policyFollow == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }
        policyFollow.setAttachments(String.join(",", attachmentUploadRequest.getAttachmentList()));

        PolicyFollow updateObj = new PolicyFollow();
        updateObj.setId(policyFollow.getId());
        updateObj.setAttachments(policyFollow.getAttachments());

        return updateById(updateObj);
    }

    private static void validPolicyFollowStatus(PolicyFollowStatusEnum currentStatusEnum, PolicyFollowStatusEnum policyFollowStatusEnum) {
        if (PolicyFollowStatusEnum.FOLLOW_UP.equals(currentStatusEnum) && !PolicyFollowStatusEnum.CHECKING.equals(policyFollowStatusEnum)) {
            throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "待跟进状态只能改为文件检查中");
        }
        if (PolicyFollowStatusEnum.CHECKING.equals(currentStatusEnum) && !PolicyFollowStatusEnum.UNDERWRITING.equals(policyFollowStatusEnum)) {
            throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "文件检查中状态只能改为承保中");
        }
        if (PolicyFollowStatusEnum.UNDERWRITING.equals(currentStatusEnum)) {
            if (PolicyFollowStatusEnum.FINISHED.equals(policyFollowStatusEnum) || PolicyFollowStatusEnum.DECLINED.equals(policyFollowStatusEnum) || PolicyFollowStatusEnum.CANCELLED.equals(policyFollowStatusEnum)) {
                throw new BusinessException(ResultCode.PARAM_CHECK_ERROR.getCode(), "承保中状态只能改为已完成");
            }
        }
    }

}




