package com.yd.csf.api.service.impl;

import cn.hutool.core.bean.BeanUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.constant.RedisConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ProjectEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.enums.TenantEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.common.utils.RedisUtil;
import com.yd.csf.api.dto.*;
import com.yd.csf.api.service.ApiBasicLawCalculateService;
import com.yd.csf.api.service.ApiExpectedFortuneLogService;
import com.yd.csf.api.service.ApiExpectedFortuneService;
import com.yd.csf.feign.request.expectedfortune.*;
import com.yd.csf.feign.response.expectedfortune.ApiExpectedFortunePageResponse;
import com.yd.csf.feign.response.expectedfortune.ApiGenerateExpectedFortuneResponse;
import com.yd.csf.service.dto.CommissionRuleBindingDto;
import com.yd.csf.service.dto.FortuneStatisticsVO;
import com.yd.csf.service.dto.QueryPolicyAndBrokerDto;
import com.yd.csf.service.dto.QueryPolicyBrokerDto;
import com.yd.csf.service.enums.FortuneStatusEnum;
import com.yd.csf.service.model.CommissionRuleBinding;
import com.yd.csf.service.model.ExpectedFortune;
import com.yd.csf.service.model.Fortune;
import com.yd.csf.service.model.PolicyBroker;
import com.yd.csf.service.service.ICommissionRuleBindingService;
import com.yd.csf.service.service.IExpectedFortuneService;
import com.yd.csf.service.service.PolicyBrokerService;
import com.yd.csf.service.service.PolicyService;
import com.yd.csf.service.vo.ExpectedFortuneStatisticsVO;
import com.yd.csf.service.vo.ExpectedFortuneStatisticsVO;
import com.yd.user.feign.client.clientuser.ApiClientUserFeignClient;
import jodd.util.StringUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.support.TransactionTemplate;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiExpectedFortuneServiceImpl implements ApiExpectedFortuneService {

    @Autowired
    private PolicyService policyService;

    @Autowired
    private ApiClientUserFeignClient apiClientUserFeignClient;

    @Autowired
    private ICommissionRuleBindingService iCommissionRuleBindingService;

    @Autowired
    private ApiBasicLawCalculateService apiBasicLawCalculateService;

    @Autowired
    private IExpectedFortuneService iExpectedFortuneService;

    @Autowired
    private ApplicationContext applicationContext;

    private ApiExpectedFortuneService getSelf() {
        return applicationContext.getBean(ApiExpectedFortuneService.class);
    }

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private RedisUtil redisUtil;

    @Autowired
    private ApiExpectedFortuneLogService apiExpectedFortuneLogService;

    @Autowired
    private PolicyBrokerService policyBrokerService;

    /**
     * 生成预计发佣
     * @param request
     * @return
     */
    @Override
    public Result<ApiGenerateExpectedFortuneResponse> generate(ApiGenerateExpectedFortuneRequest request) {
        //查询当前保单号是否正在执行预计发佣的缓存，有值就说明正在执行，无值说明执行完毕或者没有执行。
        String value = redisUtil.getCacheObject(RedisConstants.EXPECTED_FORTUNE + request.getPolicyNo());
        if (StringUtil.isNotBlank(value)) {
            //有值正在执行，提示
            return Result.fail("生成预计发佣正在执行中，无需再次执行");
        }else {
            //无值设置值，表示正在执行
            redisUtil.setCacheObject(RedisConstants.EXPECTED_FORTUNE + request.getPolicyNo(),request.getPolicyNo());
        }
        List<ExpectedFortune> list = iExpectedFortuneService.queryList(request.getPolicyNo());
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("当前保单已经生成过预计发佣数据，无需再次生成");
        }
        //根据保单号查询保单和转介人列表信息
        List<QueryPolicyAndBrokerDto> queryPolicyAndBrokerDtoList =  policyService.queryPolicyBrokerList(request.getPolicyNo());
        if (CollectionUtils.isEmpty(queryPolicyAndBrokerDtoList)) {
            throw new BusinessException("保单和转介人信息不存在");
        }

        //查询所有绑定基本法的业务员(客户端用户表用户), 计算业务员绑定的所有基本法对应的计算值, 新增积分明细和发佣记录
        List<CommissionRuleBinding> commissionRuleBindingList = new ArrayList<>();
        //根据租户ID和项目ID查询客户端用户ID列表
        Result<List<String>> result = apiClientUserFeignClient.clientUserBizIdList(TenantEnum.YD.getTenantBizId(), ProjectEnum.CSF_MINI_PROGRAM.getProjectBizId());
        if (!CollectionUtils.isEmpty(result.getData())) {
            commissionRuleBindingList = iCommissionRuleBindingService.queryList(CommissionRuleBindingDto.builder()
                    .targetIdList(result.getData())
                    .build());
        }
        if (CollectionUtils.isEmpty(commissionRuleBindingList)) {
            throw new BusinessException("绑定基本法数据不存在");
        }

        //遍历保单转介人列表信息 -> 调用基本法算出预计发佣列表
        // 通过代理对象调用
        getSelf().execute(queryPolicyAndBrokerDtoList, commissionRuleBindingList,request.getPolicyNo());

        return Result.success(null,"生成预计发佣正在处理....，稍后查看预计发佣列表");
    }

    /**
     * 异步处理-> 遍历保单转介人列表信息 -> 调用基本法算出预计发佣列表
     * @param queryPolicyAndBrokerDtoList
     * @param commissionRuleBindingList
     * @return
     */
    @Override
    @Async("commonAsyncExecutor")
    public Result execute(List<QueryPolicyAndBrokerDto> queryPolicyAndBrokerDtoList,
                          List<CommissionRuleBinding> commissionRuleBindingList,
                          String policyNo) {

        // 使用编程式事务，确保异步方法内的事务一致性
        return transactionTemplate.execute(status -> {
            try {
                for (QueryPolicyAndBrokerDto brokerDto : queryPolicyAndBrokerDtoList) {
                    Integer paymentTerm = brokerDto.getPaymentTerm();
                    if (Objects.isNull(paymentTerm)) {
                        throw new BusinessException("保单的供款年期不存在");
                    }

                    for (int i = 1; i <= paymentTerm; i++) {
                        executeBilling(ExecuteBillingDto.builder()
                                .name(brokerDto.getBrokerName())
                                .policyAndBrokerDto(brokerDto)
                                .issueNumber(i)
                                .build());

                        for (CommissionRuleBinding binding : commissionRuleBindingList) {
                            executeReward(ExecuteBillingDto.builder()
                                    .clientUserBizId(binding.getTargetId())
                                    .name(binding.getTargetName())
                                    .policyAndBrokerDto(brokerDto)
                                    .issueNumber(i)
                                    .build());
                        }
                    }
                }
                //执行成功完毕，也要销毁redis缓存
                redisUtil.deleteObject(RedisConstants.EXPECTED_FORTUNE + policyNo);

                //保存预计发佣日志记录
                //成功日志
                apiExpectedFortuneLogService.saveExpectedFortuneLog(ApiExpectedFortuneLogDto.builder()
                        .errorMsg("")
                        .policyNo(policyNo)
                        .status(0)
                        .build());
                return Result.success();
            } catch (Exception e) {
                //抛出异常，销毁redis缓存
                redisUtil.deleteObject(RedisConstants.EXPECTED_FORTUNE + policyNo);
                status.setRollbackOnly(); // 标记回滚
                log.error("异步执行预计发佣失败，已回滚所有操作", e);
                //保存预计发佣日志记录
                apiExpectedFortuneLogService.saveExpectedFortuneLog(ApiExpectedFortuneLogDto.builder()
                        .errorMsg(e.getMessage())
                        .policyNo(policyNo)
                        .status(1)
                        .build());
                throw new BusinessException("异步处理失败: " + e.getMessage());
            }
        });
    }

    /**
     * 执行 - 奖励（非销售佣金基本法）
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public Result executeReward(ExecuteBillingDto dto) {
        QueryPolicyAndBrokerDto brokerDto = dto.getPolicyAndBrokerDto();
        //构造销售佣金基本法项目的顺序下标值执行
        List<Integer> executionOrderList = new ArrayList<>();
        executionOrderList.add(1);

        //执行获得积分用户的非销售佣金基本法项目
        Result<List<AlgorithmResDto>> result = apiBasicLawCalculateService.policyBrokerAlgorithm(AlgorithmDto.builder()
                .brokerBizId(brokerDto.getBrokerBizId())
                .clientUserBizId(dto.getClientUserBizId())
                .sqlTemplateParamDto(SqlTemplateParamDto.builder()
                        .brokerBizId(brokerDto.getBrokerBizId())
                        .clientUserBizId(dto.getClientUserBizId())
                        .policyNo(brokerDto.getPolicyNo())
                        .productBizId(brokerDto.getPlanBizId())
                        .productCode(brokerDto.getProductCode())
                        .term(brokerDto.getPaymentTerm())
                        .issueNumber(dto.getIssueNumber())
                        .build())
                .executionOrderList(executionOrderList)
                .isNegateExecutionOrderList(true)
                .build());

        //生成保单预计发佣表记录 (非销售佣金基本法)
        generateExpectedFortune(GenerateExpectedFortuneDto.builder()
                //获得积分的用户绑定的基本法列表对应计算值
                .algorithmResDtoList(result.getData())
                //保单发佣批次ID
                //获得积分的用户姓名
                .broker(dto.getName())
                //获得积分的用户业务ID
                .brokerBizId(dto.getClientUserBizId())
                //发佣币种
                .currency(brokerDto.getCurrency())
                //发佣期数
                .fortunePeriod(dto.getIssueNumber())
                //发佣总期数
                .fortuneTotalPeriod(brokerDto.getPaymentTerm())
                //发佣类型 TODO
//                .fortuneType()
                //发佣日期 保单生效日期 + 3个月
                .payoutDate(calculateExpectedPayoutDate(brokerDto.getEffectiveDate(), dto.getIssueNumber()))
                //保单号
                .policyNo(brokerDto.getPolicyNo())
                //佣金发放状态 0=待出账 1=可出账 2=已出账
                .status(FortuneStatusEnum.WAIT.getItemValue())
                //所属团队 TODO
//                .team()
                //所属团队业务ID TODO
//                .teamBizId()
                .build());

        return Result.success();
    }

    private static LocalDate calculateExpectedPayoutDate(LocalDate effectiveDate, Integer issueNumber) {
        return effectiveDate.plusMonths(3).plusYears(issueNumber - 1);
    }

    /**
     * 执行 - 销售佣金（销售佣金基本法）
     * @param dto
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public Result executeBilling(ExecuteBillingDto dto) {
        QueryPolicyAndBrokerDto brokerDto = dto.getPolicyAndBrokerDto();
        //构造销售佣金基本法项目的顺序下标值执行
        List<Integer> executionOrderList = new ArrayList<>();
        executionOrderList.add(1);

        //保单出账先执行销售佣金基本法项目，入完到积分明细表里后，再执行保单绑定人的其他基本法项目
        Result<List<AlgorithmResDto>> result = apiBasicLawCalculateService.policyBrokerAlgorithm(AlgorithmDto.builder()
                .brokerBizId(brokerDto.getBrokerBizId())
                .clientUserBizId(brokerDto.getBrokerBizId())
                .sqlTemplateParamDto(SqlTemplateParamDto.builder()
                        .brokerBizId(brokerDto.getBrokerBizId())
                        .policyNo(brokerDto.getPolicyNo())
                        .productBizId(brokerDto.getPlanBizId())
                        .productCode(brokerDto.getProductCode())
                        .term(brokerDto.getPaymentTerm())
                        .issueNumber(dto.getIssueNumber())
                        .build())
                .executionOrderList(executionOrderList)
                .isNegateExecutionOrderList(false)
                .build());

        //生成保单预计发佣表记录 (销售佣金基本法)
        generateExpectedFortune(GenerateExpectedFortuneDto.builder()
                //转介人绑定的基本法列表对应计算值
                .algorithmResDtoList(result.getData())
                //保单发佣批次ID
                //获得积分的转介人（业务员）姓名
                .broker(dto.getName())
                //获得积分的转介人（业务员）业务ID
                .brokerBizId(brokerDto.getBrokerBizId())
                //发佣币种
                .currency(brokerDto.getCurrency())
                //发佣期数
                .fortunePeriod(dto.getIssueNumber())
                //发佣总期数
                .fortuneTotalPeriod(brokerDto.getPaymentTerm())
                //发佣类型 TODO
//                .fortuneType()
                //发佣日期
                .payoutDate(calculateExpectedPayoutDate(brokerDto.getEffectiveDate(), dto.getIssueNumber()))
                //保单号
                .policyNo(brokerDto.getPolicyNo())
                //佣金发放状态 0=待出账 1=可出账 2=已出账
                .status(FortuneStatusEnum.WAIT.getItemValue())
                //所属团队 TODO
//                .team()
                //所属团队业务ID TODO
//                .teamBizId()
                .build());

        return Result.success();
    }

    /**
     * 生成保单预计发佣表记录
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public Result generateExpectedFortune(GenerateExpectedFortuneDto fortuneDto) {
        List<ExpectedFortune> fortuneList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(fortuneDto.getAlgorithmResDtoList())) {
            //获得积分业务员绑定的基本法列表对应计算值结果集合
            for (AlgorithmResDto algorithmResDto : fortuneDto.getAlgorithmResDtoList()) {
                if (Objects.isNull(algorithmResDto.getCalculatedValue())
                        || algorithmResDto.getCalculatedValue().compareTo(BigDecimal.ZERO) == 0){
                    //判空判0
                    continue;
                }
                PolicyBroker policyBroker = policyBrokerService.queryOne(QueryPolicyBrokerDto.builder()
                        .policyNo(fortuneDto.getPolicyNo())
                        .brokerBizId(fortuneDto.getBrokerBizId())
                        .build());
                ExpectedFortune fortune = new ExpectedFortune();
                //发佣金额 -> 计算值 - sqlAlgorithmResultDtoList集合里面的计算值和值
//                fortune.setAmount(algorithmResDto.getCalculatedValue());
                //保单发佣批次ID
//                fortune.setBatchBizId(fortuneDto.getBatchBizId());
                //转介人姓名 (获得积分的业务员)
                fortune.setBroker(fortuneDto.getBroker());
                //转介人业务ID (获得积分的业务员)
                fortune.setBrokerBizId(fortuneDto.getBrokerBizId());
                //出账项目名称 -> 基本法项目名称
                fortune.setFortuneName(algorithmResDto.getItemName());
                //保单预计发佣业务id
                fortune.setExpectedFortuneBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_FORTUNE.getCode()));
                //基本法项目配置表唯一业务ID
                fortune.setRuleItemBizId(algorithmResDto.getRuleItemBizId());
                //发佣币种
                fortune.setCurrency(fortuneDto.getCurrency());
                //发佣期数
                fortune.setFortunePeriod(fortuneDto.getFortunePeriod());
                //发佣总期数
                fortune.setFortuneTotalPeriod(fortuneDto.getFortuneTotalPeriod());
                //保单号
                fortune.setPolicyNo(fortuneDto.getPolicyNo());
                //预计发佣日期
                fortune.setPayoutDate(fortuneDto.getPayoutDate());

                // 标准发佣金额 -> 计算值
                BigDecimal standardAmount = algorithmResDto.getCalculatedValue();
                fortune.setStandardAmount(standardAmount);

                // 转介人介绍费占比
                String brokerRatioStr = !Objects.isNull(policyBroker) ? policyBroker.getBrokerRatio() : "100";
                fortune.setBrokerRatio(brokerRatioStr);

                try {
                    // 将百分比字符串转换为小数
                    BigDecimal ratio = new BigDecimal(brokerRatioStr);
                    // 除以100
                    BigDecimal decimalRatio = ratio.divide(new BigDecimal("100"), 4, BigDecimal.ROUND_HALF_UP);

                    // 计算实际预计发佣金额 = 标准发佣金额 × decimalRatio
                    BigDecimal actualAmount = standardAmount.multiply(decimalRatio);
                    fortune.setAmount(actualAmount);
                    // 本次出账金额 = 实际预计发佣金额
                    fortune.setCurrentPaymentAmount(actualAmount);

                } catch (NumberFormatException e) {
                    // 如果比例不是有效的数字，使用标准金额
                    fortune.setAmount(standardAmount);
                } catch (Exception e) {
                    // 其他异常处理
                    log.error("计算发佣金额异常: {}", brokerRatioStr, e);
                    fortune.setAmount(standardAmount);
                }

                fortuneList.add(fortune);
            }
        }

        if (!CollectionUtils.isEmpty(fortuneList)) {
            iExpectedFortuneService.saveOrUpdateBatch(fortuneList);
        }

        return Result.success();
    }

    @Override
    public Result<Map<String, Object>> list(ApiExpectedFortunePageRequest request) {
        Page<ExpectedFortune> page = new Page<>(request.getPageNo(), request.getPageSize());

        QueryWrapper<ExpectedFortune> queryWrapper = this.getQueryWrapper(request);
        IPage<ExpectedFortune> iPage = iExpectedFortuneService.page(page, queryWrapper);
        // 查询统计数据
        List<ExpectedFortune> fortuneList = iExpectedFortuneService.list(queryWrapper);
        ExpectedFortuneStatisticsVO statisticsVO = iExpectedFortuneService.getStatistics(fortuneList.stream().map(ExpectedFortune::getId).collect(Collectors.toList()));

        // 组装返回结果
        Map<String, Object> result = new HashMap<>();
        result.put("statisticsVO", statisticsVO);
        result.put("page", iExpectedFortuneService.getVOPage(iPage));
        return Result.success(result);
    }

    @Override
    public Boolean add(ExpectedFortuneAddRequest request) {
        List<ExpectedFortuneDto> addList = request.getAddList();
        if (CollectionUtils.isEmpty(addList)) {
            return true;
        }
        // 查询最新一条记录
        List<ExpectedFortune> latestList = iExpectedFortuneService.list(new QueryWrapper<ExpectedFortune>().orderByDesc("id").last("limit 1"));
        ExpectedFortune latest = latestList.isEmpty() ? null : latestList.get(0);

        ArrayList<ExpectedFortune> fortuneList = new ArrayList<>();
        for (ExpectedFortuneDto expectedFortuneDto : addList) {
            ExpectedFortune expectedFortune = new ExpectedFortune();
            BeanUtil.copyProperties(expectedFortuneDto, expectedFortune);

            // 预计发佣业务id
            expectedFortune.setExpectedFortuneBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_EXPECTED_FORTUNE.getCode()));
            // 应付款编号
            expectedFortune.setPayableNo(this.createPayableNo(expectedFortuneDto, latest));
            fortuneList.add(expectedFortune);
        }
        iExpectedFortuneService.saveOrUpdateBatch(fortuneList);
        return true;
    }

    /**
     * 创建应付款编号 应付款类型-CSF-年份后两位-6位数字（不重复）
     *
     * @param expectedFortuneDto
     * @return
     */
    private String createPayableNo(ExpectedFortuneDto expectedFortuneDto, ExpectedFortune latest) {
        String seq = "000001";
        // 有最新记录，根据最新记录的序号生成下一个序号
        if (!Objects.isNull(latest)) {
            seq = String.format("%06d", Integer.parseInt(latest.getPayableNo().substring(12)) + 1);
        }
        return String.format("%s%s%s",
                expectedFortuneDto.getFortuneBizType() + "-CSF",
                LocalDate.now().getYear() % 100,
                seq);
    }

    /**
     * 分页查询 - 预计发佣
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiExpectedFortunePageResponse>> page(ApiExpectedFortunePageRequest request) {
        Page<ExpectedFortune> page = new Page<>(request.getPageNo(), request.getPageSize());

        QueryWrapper<ExpectedFortune> queryWrapper = this.getQueryWrapper(request);
        IPage<ExpectedFortune> iPage = iExpectedFortuneService.page(page, queryWrapper);

        return Result.success(iPage.convert(ApiExpectedFortuneServiceImpl::transform));
    }

    public static ApiExpectedFortunePageResponse transform(ExpectedFortune expectedFortune) {
        ApiExpectedFortunePageResponse response = new ApiExpectedFortunePageResponse();
        BeanUtil.copyProperties(expectedFortune, response);
        return response;
    }

    public QueryWrapper<ExpectedFortune> getQueryWrapper(ApiExpectedFortunePageRequest request) {
        QueryWrapper<ExpectedFortune> queryWrapper = new QueryWrapper<>();
        if (request == null) {
            return queryWrapper;
        }
        // 获取参数
        String policyNo = request.getPolicyNo();
        LocalDate payoutDateStart = request.getPayoutDateStart();
        LocalDate payoutDateEnd = request.getPayoutDateEnd();
        List<String> statusList = request.getStatusList();
        Integer fortunePeriod = request.getFortunePeriod();
        String fortuneName = request.getFortuneName();
        String team = request.getTeam();
        String teamBizId = request.getTeamBizId();
        String insuranceCompany = request.getInsuranceCompany();
        String productName = request.getProductName();
//        String reconciliationCompany = request.getReconciliationCompany();

        String broker = request.getBroker();
        String signer = request.getSigner();
        String productLaunchBizId = request.getProductLaunchBizId();
        String fortuneBizType = request.getFortuneBizType();

        // 按保单号模糊查询
        queryWrapper.like(StringUtils.isNotBlank(policyNo), "policy_no", policyNo);
        queryWrapper.gt(ObjectUtils.isNotEmpty(payoutDateStart), "payout_date", payoutDateStart);
        queryWrapper.lt(ObjectUtils.isNotEmpty(payoutDateEnd), "payout_date", payoutDateEnd);
        queryWrapper.in(ObjectUtils.isNotEmpty(statusList), "status", statusList);
        queryWrapper.eq(ObjectUtils.isNotEmpty(fortunePeriod), "fortune_period", fortunePeriod);
        queryWrapper.like(StringUtils.isNotBlank(fortuneName), "fortune_name", fortuneName);
        queryWrapper.like(StringUtils.isNotBlank(team), "team", team);
        queryWrapper.eq(StringUtils.isNotBlank(teamBizId), "team_biz_id", teamBizId);
        queryWrapper.like(StringUtils.isNotBlank(insuranceCompany), "insurance_company", insuranceCompany);
        queryWrapper.like(StringUtils.isNotBlank(productName), "product_name", productName);
        queryWrapper.eq(StringUtils.isNotBlank(productLaunchBizId), "product_launch_biz_id", productLaunchBizId);
//        queryWrapper.like(StringUtils.isNotBlank(reconciliationCompany), "reconciliation_company", reconciliationCompany);
        queryWrapper.like(StringUtils.isNotBlank(broker), "broker", broker);
        queryWrapper.like(StringUtils.isNotBlank(signer), "signer", signer);
        queryWrapper.eq(StringUtils.isNotBlank(fortuneBizType), "fortune_biz_type", fortuneBizType);

        // 按id升序排序
        queryWrapper.orderByAsc("id");
        return queryWrapper;
    }

    /**
     * 查询保单是否生成过预计发佣
     * @param policyNo
     * @return
     */
    @Override
    public Result<Boolean> isGenerate(String policyNo) {
        List<ExpectedFortune> list = iExpectedFortuneService.queryList(policyNo);
        if (!CollectionUtils.isEmpty(list)) {
            throw new BusinessException("当前保单已经生成过预计发佣数据，无需再次生成");
        }
        return Result.success();
    }

    /**
     * 根据保单号批量删除预计发佣数据
     * @param policyNo
     * @return
     */
    @Override
    public Result batchDelByPolicyNo(String policyNo) {
        iExpectedFortuneService.batchDelByPolicyNo(policyNo);
        return Result.success();
    }

    @Override
    public ExpectedFortuneStatisticsVO getStatistics(List<Long> expectedFortuneIds) {
        // 自定义统计数据
        ExpectedFortuneStatisticsVO statisticsVO = iExpectedFortuneService.getStatistics(expectedFortuneIds);

        BigDecimal totalAmount = statisticsVO.getTotalExpectedAmount();
        BigDecimal totalPaidAmount = statisticsVO.getTotalPaidAmount();
        // 计算待出账金额
        statisticsVO.setTotalUnpaidAmount(totalAmount.subtract(totalPaidAmount));
        // 计算已出账比例（已出账金额/预计发佣金额）
        BigDecimal divided = BigDecimal.ZERO;
        if (totalAmount.compareTo(BigDecimal.ZERO) > 0) {
            divided = totalPaidAmount.divide(totalAmount, 4, RoundingMode.HALF_UP);
        }
        // 格式化 %
        statisticsVO.setPaidAmountRatio(String.format("%.2f%%", divided.doubleValue() * 100));

        return statisticsVO;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean update(ExpectedFortuneUpdateRequest request) {
        String expectedFortuneBizId = request.getExpectedFortuneBizId();
        ExpectedFortune expectedFortune = iExpectedFortuneService.getOne(new QueryWrapper<ExpectedFortune>().eq("expected_fortune_biz_id", expectedFortuneBizId));
        if (expectedFortune == null) {
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(), "预计发佣数据不存在");
        }
        BeanUtils.copyProperties(request, expectedFortune, "id", "expectedFortuneBizId");

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        expectedFortune.setUpdaterId(loginUserId);
        expectedFortune.setUpdateTime(LocalDateTime.now());

        return iExpectedFortuneService.updateById(expectedFortune);
    }
}
