package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.constant.CodeGeneratorConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.CodeGenerator;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiConditionConfigService;
import com.yd.csf.feign.request.conditionconfig.ApiConditionConfigAddRequest;
import com.yd.csf.feign.request.conditionconfig.ApiConditionConfigEditRequest;
import com.yd.csf.feign.request.conditionconfig.ApiConditionConfigPageRequest;
import com.yd.csf.feign.response.conditionconfig.ApiConditionConfigDetailResponse;
import com.yd.csf.feign.response.conditionconfig.ApiConditionConfigPageResponse;
import com.yd.csf.service.dto.ConditionConfigDto;
import com.yd.csf.service.model.ConditionConfig;
import com.yd.csf.service.model.ConditionType;
import com.yd.csf.service.service.IConditionConfigService;
import com.yd.csf.service.service.IConditionTypeService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

@Service
public class ApiConditionConfigServiceImpl implements ApiConditionConfigService {

    @Autowired
    private IConditionConfigService iConditionConfigService;

    @Autowired
    private IConditionTypeService iConditionTypeService;

    /**
     * 分页查询-条件配置信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiConditionConfigPageResponse>> page(ApiConditionConfigPageRequest request) {
        Page<ApiConditionConfigPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiConditionConfigPageResponse> iPage = iConditionConfigService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 详情-条件配置信息
     * @param conditionBizId
     * @return
     */
    @Override
    public Result<ApiConditionConfigDetailResponse> detail(String conditionBizId) {
        Result<ConditionConfig> result = checkConditionConfigIsExist(conditionBizId);
        ConditionConfig conditionConfig = result.getData();
        ApiConditionConfigDetailResponse response = new ApiConditionConfigDetailResponse();
        BeanUtils.copyProperties(conditionConfig,response);
        ConditionType conditionType = iConditionTypeService.queryOne(conditionConfig.getConditionTypeBizId());
        if (!Objects.isNull(conditionType)) {
            response.setConditionTypeName(conditionType.getName());
        }
        return Result.success(response);
    }

    /**
     * 添加-条件配置信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiConditionConfigAddRequest request) {
        //校验条件名称唯一性
        List<ConditionConfig> conditionConfigList = iConditionConfigService.queryList(ConditionConfigDto.builder()
                .name(request.getName())
                .conditionTypeBizId(request.getConditionTypeBizId())
                .build());
        if (!CollectionUtils.isEmpty(conditionConfigList)) {
            throw new BusinessException("同类型下条件名称已存在");
        }
        ConditionConfig conditionConfig = new ConditionConfig();
        conditionConfig.setConditionBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_CONDITION_CONFIG.getCode()));
        //条件编码
        conditionConfig.setCode(CodeGenerator.generate10Code(CodeGeneratorConstants.PREFIX_CONDITION_CODE));
        iConditionConfigService.saveOrUpdate(conditionConfig);
        return Result.success();
    }

    /**
     * 编辑-条件配置信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiConditionConfigEditRequest request) {
        Result<ConditionConfig> result = checkConditionConfigIsExist(request.getConditionBizId());
        ConditionConfig conditionConfig = result.getData();
        //校验条件名称唯一性
        List<ConditionConfig> conditionConfigList = iConditionConfigService.queryList(ConditionConfigDto.builder()
                .name(request.getName())
                .conditionTypeBizId(request.getConditionTypeBizId())
                .isExcludeMy(true)
                .conditionBizId(request.getConditionBizId())
                .build());
        if (!CollectionUtils.isEmpty(conditionConfigList)) {
            throw new BusinessException("同类型下条件名称已存在");
        }
        iConditionConfigService.saveOrUpdate(conditionConfig);
        return Result.success();
    }

    /**
     * 删除-条件配置信息 TODO
     * @param conditionBizId
     * @return
     */
    @Override
    public Result del(String conditionBizId) {
        return null;
    }

    /**
     * 校验条件配置信息是否存在
     * @param conditionBizId
     * @return
     */
    public Result<ConditionConfig> checkConditionConfigIsExist(String conditionBizId) {
        ConditionConfig conditionConfig = iConditionConfigService.queryOne(conditionBizId);
        if (Objects.isNull(conditionConfig)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(conditionConfig);
    }
}
