package com.yd.csf.api.service.impl;

import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.csf.api.dto.CustomerTypeCompanyCheckCommonDto;
import com.yd.csf.api.dto.CustomerTypeIndividualCheckCommonDto;
import com.yd.csf.api.service.ApiAppointmentCheckService;
import com.yd.csf.feign.dto.appointment.*;
import com.yd.csf.feign.request.appointment.ApiAppointmentAddRequest;
import com.yd.csf.feign.request.appointment.ApiAppointmentEditRequest;
import com.yd.csf.service.enums.CustomerTypeEnum;
import com.yd.csf.service.enums.RelTypeEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

/**
 * 预约参数校验实现类
 */
@Slf4j
@Service
public class ApiAppointmentCheckServiceImpl implements ApiAppointmentCheckService {

    /**
     * 新增预约入参字段校验（非库校验）
     * @param request
     * @return
     */
    public Result checkAddRequest(ApiAppointmentAddRequest request) {

        //新增预约入参字段校验 - 预约信息字段校验
        checkAddApiAppointmentInfoDto(request.getApiAppointmentInfoDto());

        //新增预约入参字段校验 - 产品计划信息字段校验
        checkAddApiProductPlanInfoDto(request.getApiProductPlanInfoDto());

        //新增预约入参字段校验 - 投保人信息字段校验
        checkAddApiPolicyholderInfoDto(request.getApiPolicyholderInfoDto());

        //新增预约入参字段校验 - 受保人信息字段校验
        checkAddApiInsurantInfoDto(request.getApiInsurantInfoDto());

        //新增预约入参字段校验 - 受益人列表信息字段校验
        checkAddApiBeneficiaryInfoDtoList(request.getApiBeneficiaryInfoDtoList());

        //新增预约入参字段校验 - 第二持有人信息字段校验
        checkAddApiSecondHolderInfoDto(request.getApiSecondHolderInfoDto());

        return Result.success();
    }

    /**
     * 编辑预约入参字段校验（非库校验）
     * @param request
     * @return
     */
    @Override
    public Result checkEditRequest(ApiAppointmentEditRequest request) {

        //编辑预约入参字段校验 - 预约信息字段校验
        checkEditApiAppointmentInfoDto(request.getApiAppointmentInfoDto());

        //编辑预约入参字段校验 - 产品计划信息字段校验
        checkEditApiProductPlanInfoDto(request.getApiProductPlanInfoDto());

        //编辑预约入参字段校验 - 投保人信息字段校验
        checkEditApiPolicyholderInfoDto(request.getApiPolicyholderInfoDto());

        //新增预约入参字段校验 - 受保人信息字段校验
        checkEditApiInsurantInfoDto(request.getApiInsurantInfoDto());

        //编辑预约入参字段校验 - 受益人列表信息字段校验 TODO
//        checkEditApiBeneficiaryInfoDtoList(request.getApiBeneficiaryInfoDtoList());

        //编辑预约入参字段校验 - 第二持有人信息字段校验
        checkEditApiSecondHolderInfoDto(request.getApiSecondHolderInfoDto());

        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 预约信息字段校验
     * @param dto
     * @return
     */
    public Result checkAddApiAppointmentInfoDto(ApiAppointmentInfoDto dto) {
        if (Objects.isNull(dto)) {
            dto = new ApiAppointmentInfoDto();
        }
        String tipStr = "预约信息-";
        if (!Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"预约信息表主键id不传值");
        }
        if (StringUtils.isNotBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不传值");
        }
        if (StringUtils.isNotBlank(dto.getAppointmentNo())) {
            throw new BusinessException(tipStr+"预约编号不传值");
        }

        //校验预约信息参数公共方法
        checkApiAppointmentInfoDtoCommon(dto,tipStr);
        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 预约信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiAppointmentInfoDto(ApiAppointmentInfoDto dto) {
        String tipStr = "预约信息-";
        if (Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"预约信息表主键id不能为空");
        }
        if (StringUtils.isBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getAppointmentNo())) {
            throw new BusinessException(tipStr+"预约编号不能为空");
        }

        //校验预约信息参数公共方法
        checkApiAppointmentInfoDtoCommon(dto,tipStr);
        return Result.success();
    }

    /**
     * 新增或者编辑-校验预约信息参数公共方法
     * @param dto
     * @return
     */
    public Result checkApiAppointmentInfoDtoCommon(ApiAppointmentInfoDto dto,String tipStr) {
        if (StringUtils.isBlank(dto.getApplyType())) {
            throw new BusinessException(tipStr+"申请类型不能为空");
        }
        if (Objects.isNull(dto.getIntentionAppointmentTime())) {
            throw new BusinessException(tipStr+"意向预约时间不能为空");
        }

        //校验预约信息-客户和fna入参
        checkCustomerAndFna(dto,tipStr);

        if (StringUtils.isNotBlank(dto.getProposalNo()) && StringUtils.isBlank(dto.getProposalBizId())) {
            //计划书编号不为空并且计划书信息表唯一业务ID为空，提示：计划书信息表唯一业务ID不能为空
            throw new BusinessException(tipStr+"计划书信息表唯一业务ID不能为空");
        }
        if (StringUtils.isNotBlank(dto.getProposalBizId()) && StringUtils.isBlank(dto.getProposalNo())) {
            //计划书编号为空并且计划书信息表唯一业务ID不为空，提示：计划书编号不能为空
            throw new BusinessException(tipStr+"计划书编号不能为空");
        }
        return Result.success();
    }

    /**
     * 校验预约信息-客户和fna入参
     * @param dto
     * @param tipStr
     * @return
     */
    public Result checkCustomerAndFna(ApiAppointmentInfoDto dto,String tipStr) {
        if (StringUtils.isBlank(dto.getCustomerBizId())) {
            throw new BusinessException(tipStr+"客户信息表唯一业务ID不能为空");
        }
//        if (StringUtils.isBlank(dto.getCustomerNo())) {
//            throw new BusinessException(tipStr+"客户编号不能为空");
//        }
        if (StringUtils.isBlank(dto.getFnaBizId())) {
            throw new BusinessException(tipStr+"FNA信息表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getFnaNo())) {
            throw new BusinessException(tipStr+"FNA编号不能为空");
        }
        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 产品计划信息字段校验
     * @param dto
     * @return
     */
    public Result checkAddApiProductPlanInfoDto(ApiProductPlanInfoDto dto) {
        if (Objects.isNull(dto)) {
            dto = new ApiProductPlanInfoDto();
        }
        //新增预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
        checkAddApiProductPlanMainInfoDto(dto.getApiProductPlanMainInfoDto());

        //新增预约入参字段校验 - 产品计划信息字段校验 - 产品附加险信息列表字段校验
        checkAddApiProductPlanAdditionalInfoDtoList(dto.getApiProductPlanAdditionalInfoDtoList());

        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 产品计划信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiProductPlanInfoDto(ApiProductPlanInfoDto dto) {
        //编辑预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
        checkEditApiProductPlanMainInfoDto(dto.getApiProductPlanMainInfoDto());

        //编辑预约入参字段校验 - 产品计划信息字段校验 - 产品附加险信息列表字段校验 TODO
//        checkEditApiProductPlanAdditionalInfoDtoList(dto.getApiProductPlanAdditionalInfoDtoList());

        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
     * @param dto
     * @return
     */
    public Result checkAddApiProductPlanMainInfoDto(ApiProductPlanMainInfoDto dto){
        if (Objects.isNull(dto)) {
            dto = new ApiProductPlanMainInfoDto();
        }
        String tipStr = "产品计划-";
        if (!Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"产品计划表主键id不传值");
        }
        if (StringUtils.isNotBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不传值");
        }
        if (StringUtils.isNotBlank(dto.getPlanBizId())) {
            throw new BusinessException(tipStr+"产品计划信息表唯一业务ID不传值");
        }

        //新增或者编辑预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
        checkApiProductPlanMainInfoDtoCommon(dto,tipStr,0);
        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiProductPlanMainInfoDto(ApiProductPlanMainInfoDto dto){
        String tipStr = "产品计划-";
        if (Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"产品计划表主键id不能为空");
        }
        if (StringUtils.isBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getPlanBizId())) {
            throw new BusinessException(tipStr+"产品计划信息表唯一业务ID不能为空");
        }

        //新增或者编辑预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
        checkApiProductPlanMainInfoDtoCommon(dto,tipStr,1);
        return Result.success();
    }

    /**
     * 新增或者编辑预约入参字段校验 - 产品计划信息字段校验 - 产品计划主信息字段校验
     * @param dto
     * @return
     */
    public Result checkApiProductPlanMainInfoDtoCommon(ApiProductPlanMainInfoDto dto,String tipStr,Integer source) {
        if (StringUtils.isBlank(dto.getProductBizId())) {
            throw new BusinessException(tipStr+"保险产品唯一业务ID（中台保险产品业务id，冗余）不能为空");
        }
        if (StringUtils.isBlank(dto.getProductName())) {
            throw new BusinessException(tipStr+"保险产品名称不能为空");
        }
        if (StringUtils.isBlank(dto.getCurrency())) {
            throw new BusinessException(tipStr+"货币不能为空");
        }
        if (StringUtils.isBlank(dto.getPaymentTerm())) {
            throw new BusinessException(tipStr+"供款年期不能为空");
        }
        if (StringUtils.isBlank(dto.getPaymentFrequency())) {
            throw new BusinessException(tipStr+"付款频率不能为空");
        }
        if (Objects.isNull(dto.getEachIssuePremium())) {
            throw new BusinessException(tipStr+"每期保费不能为空");
        }
        if (Objects.isNull(dto.getSumInsured())) {
            throw new BusinessException(tipStr+"保额不能为空");
        }
        if (Objects.isNull(dto.getIsPrepay())) {
            throw new BusinessException(tipStr+"是否预缴保费不能为空");
        }
        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 产品计划信息字段校验 - 产品附加险信息列表字段校验
     * @param list
     * @return
     */
    public Result checkAddApiProductPlanAdditionalInfoDtoList(List<ApiProductPlanAdditionalInfoDto> list){
        if (CollectionUtils.isEmpty(list)) {
            //为空放行，不校验
            return Result.success();
        }
        String tipStr = "产品计划-产品附加险-";
        //列表下标标记位
        int i = 1;
        for (ApiProductPlanAdditionalInfoDto dto : list) {
            if (!Objects.isNull(dto.getId())) {
                throw new BusinessException(tipStr+"第"+i+"项的产品计划-附加险信息表主键id不传值");
            }
            if (StringUtils.isNotBlank(dto.getAdditionalBizId())) {
                throw new BusinessException(tipStr+"第"+i+"项的附加险信息表唯一业务ID不传值");
            }
            if (StringUtils.isBlank(dto.getPlanBizId())) {
                throw new BusinessException(tipStr+"第"+i+"项的产品计划信息表唯一业务ID不能为空");
            }
            if (StringUtils.isBlank(dto.getAdditionalProductBizId())) {
                throw new BusinessException(tipStr+"第"+i+"项的保险附加产品唯一业务ID不能为空");
            }
            if (StringUtils.isBlank(dto.getCurrency())) {
                throw new BusinessException(tipStr+"第"+i+"项的货币不能为空");
            }
            if (Objects.isNull(dto.getPremium())) {
                throw new BusinessException(tipStr+"第"+i+"项的保费不能为空");
            }
            if (Objects.isNull(dto.getSumInsured())) {
                throw new BusinessException(tipStr+"第"+i+"项的保额不能为空");
            }
            i++;
        }
        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 产品计划信息字段校验 - 编辑产品附加险信息单个对象字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiProductPlanAdditionalInfoDto(ApiProductPlanAdditionalInfoDto dto){
        String tipStr = "产品计划-产品附加险-";
        if (Objects.isNull(dto)) {
            throw new BusinessException(tipStr+"产品计划-附加险信息对象不能为空");
        }
        if (Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"产品计划-附加险信息表主键id不能为空");
        }
        if (StringUtils.isBlank(dto.getAdditionalBizId())) {
            throw new BusinessException(tipStr+"附加险信息表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getPlanBizId())) {
            throw new BusinessException(tipStr+"产品计划信息表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getAdditionalProductBizId())) {
            throw new BusinessException(tipStr+"保险附加产品唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getCurrency())) {
            throw new BusinessException(tipStr+"货币不能为空");
        }
        if (Objects.isNull(dto.getPremium())) {
            throw new BusinessException(tipStr+"保费不能为空");
        }
        if (Objects.isNull(dto.getSumInsured())) {
            throw new BusinessException(tipStr+"保额不能为空");
        }
        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 投保人信息字段校验
     * @param dto
     * @return
     */
    public Result checkAddApiPolicyholderInfoDto(ApiPolicyholderInfoDto dto) {
        if (Objects.isNull(dto)) {
            dto = new ApiPolicyholderInfoDto();
        }
        String tipStr = "投保人-";
        if (StringUtils.isBlank(dto.getCustomerType())) {
            throw new BusinessException(tipStr+"客户类型不能为空");
        }
        if (!Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"投保人信息主键id不传值");
        }
        if (StringUtils.isNotBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不传值");
        }
        if (StringUtils.isNotBlank(dto.getPolicyholderBizId())) {
            throw new BusinessException(tipStr+"投保人信息表唯一业务ID不传值");
        }

        //校验客户类型、客户业务id和客户编号字段
        checkCustomerBizIdAndNo(dto.getCustomerBizId(),dto.getCustomerNo(),tipStr);

        //新增或者编辑-校验投保人信息参数公共方法
        checkApiPolicyholderInfoDtoCommon(dto,tipStr);
        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 投保人信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiPolicyholderInfoDto(ApiPolicyholderInfoDto dto) {
        String tipStr = "投保人-";
        if (StringUtils.isBlank(dto.getCustomerType())) {
            throw new BusinessException(tipStr+"客户类型不能为空");
        }
        if (Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"投保人信息主键id不能为空");
        }
        if (StringUtils.isBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getPolicyholderBizId())) {
            throw new BusinessException(tipStr+"投保人信息表唯一业务ID不能为空");
        }

        //校验客户类型、客户业务id和客户编号字段
        checkCustomerBizIdAndNo(dto.getCustomerBizId(),dto.getCustomerNo(),tipStr);

        //新增或者编辑-校验投保人信息参数公共方法
        checkApiPolicyholderInfoDtoCommon(dto,tipStr);
        return Result.success();
    }

    /**
     * 新增或者编辑-校验投保人或者受保人信息参数公共方法
     * @param dto
     * @return
     */
    public Result checkApiPolicyholderInfoDtoCommon(ApiPolicyholderInfoDto dto,String tipStr) {
        if (CustomerTypeEnum.INDIVIDUAL.getItemValue().equals(dto.getCustomerType())) {
            //客户类型为个人的字段校验
            if (StringUtils.isBlank(dto.getMobile())) {
                throw new BusinessException(tipStr+"移动电话不能为空");
            }
            if (StringUtils.isBlank(dto.getMobileCode())) {
                throw new BusinessException(tipStr+"移动电话区号不能为空");
            }
            if (StringUtils.isBlank(dto.getNationality())) {
                throw new BusinessException(tipStr+"国籍不能为空");
            }

            //校验客户类型为个人类型的公共参数字段
            CustomerTypeIndividualCheckCommonDto individualCheckCommonDto = new CustomerTypeIndividualCheckCommonDto();
            BeanUtils.copyProperties(dto,individualCheckCommonDto);
            checkCustomerTypeIndividualCheckCommonDto(individualCheckCommonDto,tipStr);

        }else if (CustomerTypeEnum.COMPANY.getItemValue().equals(dto.getCustomerType())){
            //客户类型为公司的字段校验
            CustomerTypeCompanyCheckCommonDto companyCheckCommonDto = new CustomerTypeCompanyCheckCommonDto();
            BeanUtils.copyProperties(dto,companyCheckCommonDto);
            //校验客户类型为公司类型的公共参数字段
            checkCustomerTypeCompanyCheckCommonDto(companyCheckCommonDto,tipStr);
        }
        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 受保人信息字段校验
     * @param dto
     * @return
     */
    public Result checkAddApiInsurantInfoDto(ApiInsurantInfoDto dto) {
        if (Objects.isNull(dto)) {
            dto = new ApiInsurantInfoDto();
        }
        String tipStr = "受保人-";
        if (StringUtils.isBlank(dto.getCustomerType())) {
            throw new BusinessException(tipStr+"客户类型不能为空");
        }
        if (StringUtils.isBlank(dto.getPolicyholderRel())) {
            throw new BusinessException(tipStr+"与投保人关系不能为空");
        }
        if (RelTypeEnum.MYSELF.getItemValue().equals(dto.getPolicyholderRel())) {
            //如果对象空或者与投保人关系为本人，就放行不校验
            return Result.success();
        }
        if (!Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"受保人信息表主键id不传值");
        }
        if (StringUtils.isNotBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不传值");
        }
        if (StringUtils.isNotBlank(dto.getInsurantBizId())) {
            throw new BusinessException(tipStr+"受保人信息表唯一业务ID不传值");
        }

        //校验客户业务id和客户编号字段
        checkCustomerBizIdAndNo(dto.getCustomerBizId(),dto.getCustomerNo(),tipStr);

        //新增或者编辑-校验受保人信息参数 --> 复用新增或者编辑-校验投保人信息参数公共方法
        ApiPolicyholderInfoDto apiPolicyholderInfoDto = new ApiPolicyholderInfoDto();
        BeanUtils.copyProperties(dto,apiPolicyholderInfoDto);
        checkApiPolicyholderInfoDtoCommon(apiPolicyholderInfoDto,tipStr);

        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 受保人信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiInsurantInfoDto(ApiInsurantInfoDto dto) {
        String tipStr = "受保人-";
        if (StringUtils.isBlank(dto.getCustomerType())) {
            throw new BusinessException(tipStr+"客户类型不能为空");
        }
        if (StringUtils.isBlank(dto.getPolicyholderRel())) {
            throw new BusinessException(tipStr+"与投保人关系不能为空");
        }
        if (RelTypeEnum.MYSELF.getItemValue().equals(dto.getPolicyholderRel())) {
            //如果对象空或者与投保人关系为本人，就放行不校验
            return Result.success();
        }
        if (Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"受保人信息表主键id不能为空");
        }
        if (StringUtils.isBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getInsurantBizId())) {
            throw new BusinessException(tipStr+"受保人信息表唯一业务ID不能为空");
        }

        //校验客户业务id和客户编号字段
        checkCustomerBizIdAndNo(dto.getCustomerBizId(),dto.getCustomerNo(),tipStr);

        //新增或者编辑-校验受保人信息参数 --> 复用新增或者编辑-校验投保人信息参数公共方法
        ApiPolicyholderInfoDto apiPolicyholderInfoDto = new ApiPolicyholderInfoDto();
        BeanUtils.copyProperties(dto,apiPolicyholderInfoDto);
        checkApiPolicyholderInfoDtoCommon(apiPolicyholderInfoDto,tipStr);

        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 受益人列表信息字段校验
     * @param list
     * @return
     */
    public Result checkAddApiBeneficiaryInfoDtoList(List<ApiBeneficiaryInfoDto> list){
        if (CollectionUtils.isEmpty(list)) {
            //如果对象集合空就放行不校验
            return Result.success();
        }
        String tipStr = "受益人-";
        //列表下标标记位
        int i = 1;
        for (ApiBeneficiaryInfoDto dto : list) {
            if (StringUtils.isBlank(dto.getCustomerType())) {
                throw new BusinessException(tipStr+"客户类型不能为空");
            }
            if (StringUtils.isBlank(dto.getInsurantRel())) {
                throw new BusinessException(tipStr+"与受保人关系不能为空");
            }
            if (!Objects.isNull(dto.getId())) {
                throw new BusinessException(tipStr+"第"+i+"项的受益人信息主键id不传值");
            }
            if (StringUtils.isBlank(dto.getAppointmentBizId())) {
                throw new BusinessException(tipStr+"第"+i+"项的预约信息主表唯一业务ID不能为空");
            }
            if (StringUtils.isNotBlank(dto.getBeneficiaryBizId())) {
                throw new BusinessException(tipStr+"第"+i+"项的受益人信息表唯一业务ID不传值");
            }

            if (RelTypeEnum.MYSELF.getItemValue().equals(dto.getInsurantRel())) {
                //如果对象与受保人关系为本人，就不校验下面的参数，继续循环下一个
                continue;
            }

            if (CustomerTypeEnum.INDIVIDUAL.getItemValue().equals(dto.getCustomerType())) {
                //客户类型为个人的字段校验
                //校验客户类型为个人类型的公共参数字段
                CustomerTypeIndividualCheckCommonDto individualCheckCommonDto = new CustomerTypeIndividualCheckCommonDto();
                BeanUtils.copyProperties(dto,individualCheckCommonDto);
                checkCustomerTypeIndividualCheckCommonDto(individualCheckCommonDto,tipStr);

            }else if (CustomerTypeEnum.COMPANY.getItemValue().equals(dto.getCustomerType())){
                //客户类型为公司的字段校验
                CustomerTypeCompanyCheckCommonDto companyCheckCommonDto = new CustomerTypeCompanyCheckCommonDto();
                BeanUtils.copyProperties(dto,companyCheckCommonDto);
                //校验客户类型为公司类型的公共参数字段
                checkCustomerTypeCompanyCheckCommonDto(companyCheckCommonDto,tipStr);
            }
            i++;
        }
        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 编辑单个受益人信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiBeneficiaryInfoDto(ApiBeneficiaryInfoDto dto){
        if (!Objects.isNull(dto)) {
            throw new BusinessException("入参对象不能为空");
        }
        String tipStr = "受益人-";
        if (StringUtils.isBlank(dto.getCustomerType())) {
            throw new BusinessException(tipStr+"客户类型不能为空");
        }
        if (StringUtils.isBlank(dto.getInsurantRel())) {
            throw new BusinessException(tipStr+"与受保人关系不能为空");
        }
        if (Objects.isNull(dto.getId())) {
            throw new BusinessException(tipStr+"受益人信息主键id不能为空");
        }
        if (StringUtils.isBlank(dto.getAppointmentBizId())) {
            throw new BusinessException(tipStr+"预约信息主表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(dto.getBeneficiaryBizId())) {
            throw new BusinessException(tipStr+"受益人信息表唯一业务ID不能为空");
        }

        if (RelTypeEnum.MYSELF.getItemValue().equals(dto.getInsurantRel())) {
            //如果对象与受保人关系为本人，就不校验下面的了
            return Result.success();
        }

        if (CustomerTypeEnum.INDIVIDUAL.getItemValue().equals(dto.getCustomerType())) {
            //客户类型为个人的字段校验
            //校验客户类型为个人类型的公共参数字段
            CustomerTypeIndividualCheckCommonDto individualCheckCommonDto = new CustomerTypeIndividualCheckCommonDto();
            BeanUtils.copyProperties(dto,individualCheckCommonDto);
            checkCustomerTypeIndividualCheckCommonDto(individualCheckCommonDto,tipStr);

        }else if (CustomerTypeEnum.COMPANY.getItemValue().equals(dto.getCustomerType())){
            //客户类型为公司的字段校验
            CustomerTypeCompanyCheckCommonDto companyCheckCommonDto = new CustomerTypeCompanyCheckCommonDto();
            BeanUtils.copyProperties(dto,companyCheckCommonDto);
            //校验客户类型为公司类型的公共参数字段
            checkCustomerTypeCompanyCheckCommonDto(companyCheckCommonDto,tipStr);
        }
        return Result.success();
    }

    /**
     * 新增预约入参字段校验 - 第二持有人信息字段校验
     * @param dto
     * @return
     */
    public Result checkAddApiSecondHolderInfoDto(ApiSecondHolderInfoDto dto) {
        if (Objects.isNull(dto)) {
            //如果对象空就放行不校验
            return Result.success();
        }
        if (!Objects.isNull(dto.getId())) {
            throw new BusinessException("第二持有人信息表主键id不传值");
        }
        if (StringUtils.isNotBlank(dto.getAppointmentBizId())) {
            throw new BusinessException("预约信息主表唯一业务ID不传值");
        }
        if (StringUtils.isNotBlank(dto.getSecondHolderBizId())) {
            throw new BusinessException("第二持有人信息表唯一业务ID不传值");
        }

        return Result.success();
    }

    /**
     * 编辑预约入参字段校验 - 第二持有人信息字段校验
     * @param dto
     * @return
     */
    public Result checkEditApiSecondHolderInfoDto(ApiSecondHolderInfoDto dto) {
        if (Objects.isNull(dto)) {
            //如果对象空就放行不校验
            return Result.success();
        }
        if (StringUtils.isNotBlank(dto.getSecondHolderBizId())) {
            if (Objects.isNull(dto.getId())) {
                throw new BusinessException("第二持有人信息表主键id不能为空");
            }
            if (StringUtils.isBlank(dto.getAppointmentBizId())) {
                throw new BusinessException("预约信息主表唯一业务ID不能为空");
            }
        }
        return Result.success();
    }

    /**
     * 校验客户业务id和客户编号字段
     * @param customerBizId
     * @param customerNo
     * @return
     */
    public Result checkCustomerBizIdAndNo(String customerBizId,String customerNo,String tipStr) {
//        if (StringUtils.isNotBlank(customerNo) && StringUtils.isBlank(customerBizId)) {
//            //关联客户编号不为空并且关联客户信息表唯一业务ID为空，提示：客户信息表唯一业务ID不能为空
//            throw new BusinessException(tipStr+"客户信息表唯一业务ID不能为空");
//        }
//        if (StringUtils.isNotBlank(customerBizId) && StringUtils.isBlank(customerNo)) {
//            //关联客户编号为空并且关联客户信息表唯一业务ID不为空，提示：客户编号不能为空
//            throw new BusinessException(tipStr+"客户编号不能为空");
//        }
        return Result.success();
    }

    /**
     * 校验客户类型为公司类型的公共参数字段
     * @param dto
     * @return
     */
    public Result checkCustomerTypeCompanyCheckCommonDto(CustomerTypeCompanyCheckCommonDto dto,String tipStr){
        if (StringUtils.isBlank(dto.getCompanyName())) {
            throw new BusinessException(tipStr+"公司名称不能为空");
        }
        if (StringUtils.isBlank(dto.getCompanyNameEn())) {
            throw new BusinessException(tipStr+"公司名称（英文）不能为空");
        }
        if (StringUtils.isBlank(dto.getCompanyBusinessNo())) {
            throw new BusinessException(tipStr+"公司商业登记号码不能为空");
        }
        if (Objects.isNull(dto.getCompanyRegisterTime())) {
            throw new BusinessException(tipStr+"公司注册日期不能为空");
        }
        if (Objects.isNull(dto.getCompanyMobile())) {
            throw new BusinessException(tipStr+"公司电话不能为空");
        }
        if (Objects.isNull(dto.getCompanyMobileCode())) {
            throw new BusinessException(tipStr+"公司电话区号不能为空");
        }
        if (Objects.isNull(dto.getCompanyRegisterRegion())) {
            throw new BusinessException(tipStr+"公司注册地区不能为空");
        }
        return Result.success();
    }

    /**
     * 校验客户类型为个人类型的公共参数字段
     * @param dto
     * @return
     */
    public Result checkCustomerTypeIndividualCheckCommonDto(CustomerTypeIndividualCheckCommonDto dto,String tipStr){
        if (StringUtils.isBlank(dto.getName())) {
            throw new BusinessException(tipStr+"名字不能为空");
        }
        if (StringUtils.isBlank(dto.getNameEn())) {
            throw new BusinessException(tipStr+"名字-英文不能为空");
        }
        if (StringUtils.isBlank(dto.getGender())) {
            throw new BusinessException(tipStr+"性别不能为空");
        }
        return Result.success();
    }
}
