package com.yd.csf.service.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.csf.service.dto.*;
import com.yd.csf.service.model.Fortune;
import com.yd.csf.service.service.FortuneAccountService;
import com.yd.csf.service.service.FortuneService;
import com.yd.csf.service.dao.FortuneMapper;
import com.yd.csf.service.vo.FortuneVO;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author Zhang Jianan
 * @description 针对表【fortune(保单发佣表)】的数据库操作Service实现
 * @createDate 2025-09-22 13:22:08
 */
@Service
public class FortuneServiceImpl extends ServiceImpl<FortuneMapper, Fortune>
        implements FortuneService {

    @Resource
    private FortuneAccountService fortuneAccountService;

    @Override
    public Wrapper<Fortune> getQueryWrapper(FortuneQueryRequest fortuneQueryRequest) {
        QueryWrapper<Fortune> queryWrapper = new QueryWrapper<>();
        if (fortuneQueryRequest == null) {
            return queryWrapper;
        }
        String status = fortuneQueryRequest.getStatus();
        String policyNo = fortuneQueryRequest.getPolicyNo();
        queryWrapper.eq(StringUtils.isNotBlank(status), "status", status);
        queryWrapper.eq(StringUtils.isNotBlank(policyNo), "policy_no", policyNo);

        queryWrapper.orderByAsc("id");
        return queryWrapper;
    }

    @Override
    public Page<FortuneVO> getFortuneVOPage(Page<Fortune> fortunePage) {
        List<Fortune> fortuneList = fortunePage.getRecords();
        Page<FortuneVO> fortuneVOPage = new Page<>(fortunePage.getCurrent(), fortunePage.getSize());
        if (CollUtil.isEmpty(fortuneList)) {
            return fortuneVOPage;
        }

        List<FortuneVO> fortuneVOList = fortuneList.stream().map(fortune -> {
            FortuneVO fortuneVO = new FortuneVO();
            BeanUtils.copyProperties(fortune, fortuneVO);
            return fortuneVO;
        }).collect(Collectors.toList());

        fortuneVOPage.setRecords(fortuneVOList);
        return fortuneVOPage;
    }

    @Override
    public Boolean updateFortune(FortuneUpdateRequest fortuneUpdateRequest) {
        String fortuneBizId = fortuneUpdateRequest.getFortuneBizId();
        Fortune fortune = this.getOne(new QueryWrapper<Fortune>().eq("fortune_biz_id", fortuneBizId));
        if (fortune == null) {
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(), "发佣数据不存在");
        }
        BeanUtils.copyProperties(fortuneUpdateRequest, fortune);

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        fortune.setUpdaterId(loginUserId);
        fortune.setUpdateTime(new Date());

        return this.updateById(fortune);
    }

    @Override
    public void downloadFortune(FortuneDownloadRequest fortuneDownloadRequest, HttpServletResponse response) throws IOException {
        List<String> fortuneBizIdList = fortuneDownloadRequest.getFortuneBizIdList();
        if (CollUtil.isNotEmpty(fortuneBizIdList)) {
            List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));
            if (CollUtil.isNotEmpty(fortuneList)) {
                // 转换为导出DTO
                List<FortuneExportDTO> exportDTOList = fortuneList.stream().map(fortune -> {
                    FortuneExportDTO exportDTO = new FortuneExportDTO();
                    BeanUtils.copyProperties(fortune, exportDTO);
                    return exportDTO;
                }).collect(Collectors.toList());

                // 设置响应头
                response.setContentType("application/vnd.ms-excel;charset=UTF-8");
                response.setCharacterEncoding("utf-8");
                String encode = URLEncoder.encode("保单发佣.xlsx", "UTF-8");
                response.setHeader("Content-Disposition", "attachment;filename=" + "fortune_"+System.currentTimeMillis() + ".xlsx");

                // 导出
                try {
                    EasyExcel.write(response.getOutputStream(), FortuneExportDTO.class).sheet("保单发佣").doWrite(exportDTOList);
                } catch (IOException e) {
                    log.error("导出保单发佣数据失败", e);
                    throw new BusinessException(ResultCode.FAIL.getCode(), "导出失败");
                }
            }
        }
    }

    @Override
    public void downloadAccount(FortuneDownloadRequest fortuneDownloadRequest, HttpServletResponse response) throws UnsupportedEncodingException {
        List<String> fortuneBizIdList = fortuneDownloadRequest.getFortuneBizIdList();
        if (CollUtil.isNotEmpty(fortuneBizIdList)) {
            List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));
            if (CollUtil.isNotEmpty(fortuneList)) {
                // 按人分组
                Map<String, List<Fortune>> fortuneMap = fortuneList.stream().collect(Collectors.groupingBy(Fortune::getBroker));

                // 创建按人分组的导出DTO列表
                List<FortuneAccountExportDTO> accountExportDTOList = new ArrayList<>();

                // 收集所有不同的fortune项目名称
                Set<String> allFortuneNames = fortuneList.stream()
                        .map(Fortune::getFortuneName)
                        .filter(StringUtils::isNotBlank)
                        .collect(Collectors.toSet());

                // 处理每个人的数据
                for (Map.Entry<String, List<Fortune>> entry : fortuneMap.entrySet()) {
                    String broker = entry.getKey();
                    List<Fortune> brokerFortunes = entry.getValue();

                    FortuneAccountExportDTO accountDTO = new FortuneAccountExportDTO();
                    accountDTO.setBroker(broker);

                    // 设置团队、币种（取第一个记录）
                    if (CollUtil.isNotEmpty(brokerFortunes)) {
                        accountDTO.setTeam(brokerFortunes.get(0).getTeam());
                        accountDTO.setCurrency(brokerFortunes.get(0).getCurrency());
                    }

                    // 计算出账总额
                    BigDecimal totalAmount = brokerFortunes.stream()
                            .map(Fortune::getAmount)
                            .reduce(BigDecimal.ZERO, BigDecimal::add);
                    accountDTO.setAmount(totalAmount);

                    // 按fortune项目分组并累加金额
                    Map<String, BigDecimal> fortuneAmountMap = brokerFortunes.stream()
                            .collect(Collectors.groupingBy(
                                    Fortune::getFortuneName,
                                    Collectors.reducing(BigDecimal.ZERO, Fortune::getAmount, BigDecimal::add)
                            ));

                    // 设置各个fortune项目的金额
                    fortuneAmountMap.forEach(accountDTO::addFortuneAmount);

                    accountExportDTOList.add(accountDTO);
                }

                // 设置响应头
                response.setContentType("application/vnd.ms-excel;charset=UTF-8");
                response.setCharacterEncoding("utf-8");
                String fileName = "fortune_account_" + System.currentTimeMillis() + ".xlsx";
                response.setHeader("Content-Disposition", "attachment;filename=" + fileName);

                // 创建动态表头
                List<List<String>> head = new ArrayList<>();
                head.add(Collections.singletonList("转介人"));
                head.add(Collections.singletonList("所属团队"));
                head.add(Collections.singletonList("出账币种"));
                head.add(Collections.singletonList("出账总额"));

                // 添加所有fortune项目作为表头
                for (String fortuneName : allFortuneNames) {
                    head.add(Collections.singletonList(fortuneName + "金额"));
                }

                // 构建数据行
                List<List<Object>> dataList = new ArrayList<>();
                for (FortuneAccountExportDTO dto : accountExportDTOList) {
                    List<Object> row = new ArrayList<>();
                    row.add(dto.getBroker());
                    row.add(dto.getTeam());
                    row.add(dto.getCurrency());
                    row.add(dto.getAmount());
                    for (String fortuneName : allFortuneNames) {
                        row.add(dto.getFortuneAmount(fortuneName));
                    }
                    dataList.add(row);
                }

                // 写入数据库
                fortuneAccountService.saveFortuneAccount(accountExportDTOList);

                // 使用try-with-resources确保流正确关闭
                try (ServletOutputStream outputStream = response.getOutputStream()) {
                    // 写入 Excel
                    EasyExcel.write(outputStream)
                            .head(head)
                            .sheet("出账清单")
                            .doWrite(dataList);

                    outputStream.flush();
                } catch (Exception e) {
                    log.error("导出失败", e);
                    throw new BusinessException(ResultCode.FAIL.getCode(), "导出失败");
                }
            }
        }
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public Boolean updateFortuneStatus(FortuneStatusUpdateRequest fortuneStatusUpdateRequest) {

        List<String> fortuneBizIdList = fortuneStatusUpdateRequest.getFortuneBizIdList();
        List<Fortune> fortuneList = this.list(new QueryWrapper<Fortune>().in("fortune_biz_id", fortuneBizIdList));

        // 获取当前登录用户
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        Date now = new Date();
        List<Fortune> updateList = new ArrayList<>();
        for (Fortune fortune : fortuneList) {
            Fortune updateFortune = new Fortune();
            updateFortune.setId(fortune.getId());
            updateFortune.setUpdaterId(loginUserId);
            updateFortune.setUpdateTime(now);
            updateFortune.setStatus(fortuneStatusUpdateRequest.getStatus());
            updateList.add(updateFortune);
        }

        return this.updateBatchById(updateList);
    }

}




