package com.yd.csf.service.service.impl;

import cn.hutool.core.collection.CollUtil;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.google.gson.Gson;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dao.CustomerMapper;
import com.yd.csf.service.dto.CustomerAddRequest;
import com.yd.csf.service.dto.CustomerQueryRequest;
import com.yd.csf.service.dto.FnaQueryRequest;
import com.yd.csf.service.model.Customer;
import com.yd.csf.service.model.CustomerExpand;
import com.yd.csf.service.service.CustomerExpandService;
import com.yd.csf.service.service.CustomerService;
import com.yd.csf.service.utils.ValidateUtil;
import com.yd.csf.service.vo.AddressVO;
import com.yd.csf.service.vo.CustomerExpandVO;
import com.yd.csf.service.vo.CustomerVO;
import com.yd.csf.service.vo.FnaVO;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

/**
* @author Zhang Jianan
* @description 针对表【customer(客户信息表)】的数据库操作Service实现
* @createDate 2025-08-29 17:38:23
*/
@Service
public class CustomerServiceImpl extends ServiceImpl<CustomerMapper, Customer>
    implements CustomerService{

    private final static Gson GSON = new Gson();

    @Resource
    private CustomerExpandService customerExpandService;

    @Resource
    private CustomerService customerService;



    @Override
    public CustomerVO getCustomerVO(Customer customer) {
        CustomerVO customerVO = CustomerVO.objToVo(customer);

        // 1. 关联查询扩展信息
        CustomerExpand customerExpand = customerExpandService.getByCustomerBizId(customer.getCustomerBizId());
        if (customerExpand != null) {
            CustomerExpandVO customerExpandVO = CustomerExpandVO.objToVo(customerExpand);

            List<AddressVO> addressList = new ArrayList<>();

            if (customerExpand.getResidenceAddress() != null) {
                AddressVO addressVO = GSON.fromJson(customerExpand.getResidenceAddress(), AddressVO.class);
                addressVO.setType("residenceAddress");
                addressList.add(addressVO);
            }
            if (customerExpand.getResidentialAddress() != null) {
                AddressVO addressVO = GSON.fromJson(customerExpand.getResidentialAddress(), AddressVO.class);
                addressVO.setType("residentialAddress");
                addressList.add(addressVO);
            }
            if (customerExpand.getMailingAddress() != null) {
                AddressVO addressVO = GSON.fromJson(customerExpand.getMailingAddress(), AddressVO.class);
                addressVO.setType("mailingAddress");
                addressList.add(addressVO);
            }
            if (customerExpand.getCompanyAddress() != null) {
                AddressVO addressVO = GSON.fromJson(customerExpand.getCompanyAddress(), AddressVO.class);
                addressVO.setType("companyAddress");
                addressList.add(addressVO);
            }
            customerExpandVO.setAddressList(addressList);
            customerVO.setCustomerExpand(customerExpandVO);
        }
        return customerVO;
    }

    @Override
    public Wrapper<Customer> getQueryWrapper(CustomerQueryRequest customerQueryRequest) {
        return null;
    }

    @Override
    public CustomerExpand getCustomerExpand(CustomerAddRequest customerAddRequest) {
        CustomerExpand customerExpand = new CustomerExpand();
        BeanUtils.copyProperties(customerAddRequest, customerExpand);
        List<AddressVO> addressList = customerAddRequest.getAddressList();
        if (CollUtil.isNotEmpty(addressList)) {
            for (AddressVO addressVO : addressList) {
                if ("residenceAddress".equals(addressVO.getType())) {
                    customerExpand.setResidenceAddress(GSON.toJson(addressVO));
                }
                if ("residentialAddress".equals(addressVO.getType())) {
                    customerExpand.setResidentialAddress(GSON.toJson(addressVO));
                }
                if ("mailingAddress".equals(addressVO.getType())) {
                    customerExpand.setMailingAddress(GSON.toJson(addressVO));
                }
                if ("companyAddress".equals(addressVO.getType())) {
                    customerExpand.setCompanyAddress(GSON.toJson(addressVO));
                }
            }
        }

        return customerExpand;
    }

    @Override
    public Page<CustomerVO> getCustomerVOPage(Page<Customer> customerPage) {
        List<Customer> customerList = customerPage.getRecords();
        Page<CustomerVO> customerVOPage = new Page<>(customerPage.getCurrent(), customerPage.getSize(), customerPage.getTotal());
        if (CollUtil.isEmpty(customerList)) {
            return customerVOPage;
        }
        // 对象列表 => 封装对象列表
        List<CustomerVO> customerVOList = customerList.stream().map(CustomerVO::objToVo).collect(Collectors.toList());

        // 1. 关联查询扩展信息
        Set<String> customerIdSet = customerList.stream().map(Customer::getCustomerBizId).collect(Collectors.toSet());
        Map<String, List<CustomerExpand>> customerBizIdCustomerExpandListMap = customerExpandService.listByIds(customerIdSet).stream()
                .collect(Collectors.groupingBy(CustomerExpand::getCustomerBizId));

        // 填充信息
        customerVOList.forEach(customerVO -> {
            String customerBizId = customerVO.getCustomerBizId();
            CustomerExpand customerExpand = null;
            if (customerBizIdCustomerExpandListMap.containsKey(customerBizId)) {
                customerExpand = customerBizIdCustomerExpandListMap.get(customerBizId).get(0);
            }
            customerVO.setCustomerExpand(CustomerExpandVO.objToVo(customerExpand));
        });
        // endregion

        customerVOPage.setRecords(customerVOList);
        return customerVOPage;
    }

    @Override
    public Customer getByCustomerBizId(String customerBizId) {
        return baseMapper.selectOne(new LambdaQueryWrapper<Customer>().eq(Customer::getCustomerBizId, customerBizId));
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    public String addCustomer(CustomerAddRequest customerAddRequest) {
        // 校验 请求 对象中，参数是否全部为空
        if (ValidateUtil.isAllFieldsNull(customerAddRequest)) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }

        // 客户主表信息
        Customer customer = new Customer();
        BeanUtils.copyProperties(customerAddRequest, customer);
        // 客户扩展表信息
        CustomerExpand customerExpand = customerService.getCustomerExpand(customerAddRequest);

        // 获取 userBizId
//        SecurityUtil.getLoginUser();
//        customer.setUserBizId("");

        // 客户主表业务唯一id
        customer.setCustomerBizId(RandomStringGenerator.generateBizId16("customer"));
        boolean result = customerService.save(customer);
        if (!result) {
            throw new BusinessException(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
        }

        // 客户扩展表信息，非空时，写入数据库
        if (!ValidateUtil.isAllFieldsNull(customerAddRequest)) {
            // 主表业务唯一id
            customerExpand.setCustomerBizId(customer.getCustomerBizId());
            // 客户扩展表业务唯一id
            customerExpand.setCustomerExpandBizId(RandomStringGenerator.generateBizId16("customer_expand"));
            // 写入数据库
            boolean resultExpand = customerExpandService.save(customerExpand);
            if (!resultExpand) {
                throw new BusinessException(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
            }
            // 更新客户主表
            Customer updateCustomer = new Customer();
            updateCustomer.setId(customer.getId());
            updateCustomer.setCustomerExpandBizId(customerExpand.getCustomerBizId());
            customerService.updateById(updateCustomer);
        }
        return customer.getCustomerBizId();
    }

}




