package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.sun.org.apache.bcel.internal.generic.I2F;
import com.yd.common.constant.CodeGeneratorConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.CodeGenerator;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.*;
import com.yd.csf.feign.dto.appointment.*;
import com.yd.csf.feign.dto.appointmentfile.ApiAppointmentFileDto;
import com.yd.csf.feign.request.appointment.*;
import com.yd.csf.feign.response.appointment.ApiAppointmentAddResponse;
import com.yd.csf.feign.response.appointment.ApiAppointmentDetailResponse;
import com.yd.csf.feign.response.appointment.ApiAppointmentPageResponse;
import com.yd.csf.service.enums.AppointmentStatusEnum;
import com.yd.csf.service.model.Appointment;
import com.yd.csf.service.model.Fna;
import com.yd.csf.service.model.ProductPlan;
import com.yd.csf.service.service.FnaService;
import com.yd.csf.service.service.IAppointmentService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Objects;

/**
 * 预约信息业务实现类
 */
@Slf4j
@Service
public class ApiAppointmentServiceImpl implements ApiAppointmentService {

    @Autowired
    private ApiAppointmentCheckService apiAppointmentCheckService;

    @Autowired
    private ApiProductPlanService apiProductPlanService;

    @Autowired
    private ApiAdditionalService apiAdditionalService;

    @Autowired
    private ApiPolicyholderService apiPolicyholderService;

    @Autowired
    private ApiInsurantService apiInsurantService;

    @Autowired
    private ApiBeneficiaryService apiBeneficiaryService;

    @Autowired
    private ApiSecondHolderService apiSecondHolderService;

    @Autowired
    private IAppointmentService iAppointmentService;

    @Autowired
    private ApiAppointmentFileService apiAppointmentFileService;

    @Autowired
    private FnaService fnaService;

    /**
     * 预约分页查询
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiAppointmentPageResponse>> page(ApiAppointmentPageRequest request) {
        Page<ApiAppointmentPageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiAppointmentPageResponse> iPage = iAppointmentService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 预约详情（聚合信息详情）
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<ApiAppointmentDetailResponse> detail(String appointmentBizId) {
        ApiAppointmentDetailResponse response = new ApiAppointmentDetailResponse();
        //预约主体信息详情（单个对象详情）
        Result<Appointment> result = checkAppointmentIsExist(appointmentBizId);
        Appointment appointment = result.getData();
        ApiAppointmentInfoDto dto = new ApiAppointmentInfoDto();
        BeanUtils.copyProperties(appointment,dto);
        response.setApiAppointmentInfoDto(dto);
        //产品计划信息
        Result<ApiProductPlanInfoDto> result1 = apiProductPlanService.detail(appointmentBizId,"");
        response.setApiProductPlanInfoDto(result1.getData());
        //投保人信息
        Result<ApiPolicyholderInfoDto> result2 = apiPolicyholderService.detail(appointmentBizId,"");
        response.setApiPolicyholderInfoDto(result2.getData());
        //受保人信息
        Result<ApiInsurantInfoDto> result3 = apiInsurantService.detail(appointmentBizId,"");
        response.setApiInsurantInfoDto(result3.getData());
        //受益人列表信息
        Result<List<ApiBeneficiaryInfoDto>> result4 = apiBeneficiaryService.list(appointmentBizId);
        response.setApiBeneficiaryInfoDtoList(result4.getData());
        //第二持有人信息
        Result<ApiSecondHolderInfoDto> result5 = apiSecondHolderService.detail(appointmentBizId,"");
        response.setApiSecondHolderInfoDto(result5.getData());
        //预约附件信息列表
        Result<List<ApiAppointmentFileDto>> result6 = apiAppointmentFileService.list(appointmentBizId);
        response.setApiAppointmentFileDtoList(result6.getData());
        response.setStatus(!Objects.isNull(result.getData()) ? appointment.getStatus() : null);
        return Result.success(response);
    }

    /**
     * 预约主体信息详情（单个对象详情）
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<ApiAppointmentInfoDto> mainDetail(String appointmentBizId) {
        Result<Appointment> result = checkAppointmentIsExist(appointmentBizId);
        Appointment appointment = result.getData();
        ApiAppointmentInfoDto dto = new ApiAppointmentInfoDto();
        BeanUtils.copyProperties(appointment,dto);
        return Result.success(dto);
    }

    /**
     * 新增预约提交
     * @param request
     * @return
     */
    @Override
    @Transactional(rollbackFor = Exception.class)
    public Result<ApiAppointmentAddResponse> add(ApiAppointmentAddRequest request) {
        //新增预约入参字段校验（非库校验）
        apiAppointmentCheckService.checkAddRequest(request);

        //新增预约-添加预约信息主表数据 - 预约状态为待预约
        Result<Appointment> appointmentResult = addAppointmentData(request.getApiAppointmentInfoDto(),AppointmentStatusEnum.DYY.getItemValue());
        Appointment appointment = appointmentResult.getData();

        //添加产品计划信息表数据
        Result<ProductPlan> productPlanResult = apiProductPlanService.addProductPlanData(request.getApiProductPlanInfoDto(),appointment.getAppointmentBizId());
        ProductPlan productPlan = productPlanResult.getData();
        //批量添加产品计划-附加险信息表数据
        apiAdditionalService.batchAddAdditionalData(request.getApiProductPlanInfoDto(),productPlan.getPlanBizId());

        //添加投保人信息表数据
        apiPolicyholderService.addPolicyholderData(request.getApiPolicyholderInfoDto(),appointment.getAppointmentBizId());

        //添加受保人信息表数据
        apiInsurantService.addInsurantData(request.getApiInsurantInfoDto(),appointment.getAppointmentBizId());

        //批量添加受益人信息表数据
        apiBeneficiaryService.batchAddBeneficiaryData(request.getApiBeneficiaryInfoDtoList(),appointment.getAppointmentBizId());

        //添加第二持有人信息表数据
        apiSecondHolderService.addSecondHolderData(request.getApiSecondHolderInfoDto(),appointment.getAppointmentBizId());

        //预约编号和预约业务id更新到FNA表（提交待预约状态，预约信息的预约业务id和预约编号更新到Fna表的预约业务id和预约编号）
        updateFnaBizIdAndNo(appointment.getFnaBizId(),appointment.getAppointmentBizId(),appointment.getAppointmentNo());
        ApiAppointmentAddResponse response = new ApiAppointmentAddResponse();
        response.setAppointmentBizId(appointment.getAppointmentBizId());
        return Result.success(response);
    }

    /**
     * 提交待预约状态，预约信息更新到Fna表的预约业务id和预约编号
     * @return
     */
    public Result updateFnaBizIdAndNo(String fnaBizId,String appointmentBizId,String appointmentNo) {
        Fna fna = fnaService.queryOne(fnaBizId);
        if (!Objects.isNull(fna)) {
            //预约编号
            fna.setAppointmentBizId(appointmentBizId);
            fna.setAppointmentNo(appointmentNo);
            fnaService.saveOrUpdate(fna);
        }
        return Result.success();
    }

    /**
     * 新增预约暂存
     * @param request
     * @return
     */
    @Override
    public Result addStorage(ApiAppointmentAddStorageRequest request) {
        if (Objects.isNull(request.getApiAppointmentInfoDto())) {
            request.setApiAppointmentInfoDto(new ApiAppointmentInfoDto());
        }
        //校验预约信息-客户和fna入参
        apiAppointmentCheckService.checkCustomerAndFna(request.getApiAppointmentInfoDto(),"预约信息-");

        //新增预约暂存-添加预约信息主表数据为暂存状态
        Result<Appointment> appointmentResult = addAppointmentData(request.getApiAppointmentInfoDto(), AppointmentStatusEnum.ZC.getItemValue());
        Appointment appointment = appointmentResult.getData();

        //添加产品计划信息表数据
        Result<ProductPlan> productPlanResult = apiProductPlanService.addProductPlanData(request.getApiProductPlanInfoDto(),appointment.getAppointmentBizId());
        ProductPlan productPlan = productPlanResult.getData();
        //批量添加产品计划-附加险信息表数据
        apiAdditionalService.batchAddAdditionalData(request.getApiProductPlanInfoDto(),productPlan.getPlanBizId());

        //添加投保人信息表数据
        apiPolicyholderService.addPolicyholderData(request.getApiPolicyholderInfoDto(),appointment.getAppointmentBizId());

        //添加受保人信息表数据
        apiInsurantService.addInsurantData(request.getApiInsurantInfoDto(),appointment.getAppointmentBizId());

        //批量添加受益人信息表数据
        apiBeneficiaryService.batchAddBeneficiaryData(request.getApiBeneficiaryInfoDtoList(),appointment.getAppointmentBizId());

        //添加第二持有人信息表数据
        apiSecondHolderService.addSecondHolderData(request.getApiSecondHolderInfoDto(),appointment.getAppointmentBizId());

        return Result.success();
    }

    /**
     * 新增预约-添加预约信息主表数据
     * @param dto
     * @return
     */
    public Result<Appointment> addAppointmentData(ApiAppointmentInfoDto dto,Integer status) {
        if (Objects.isNull(dto)) {
            //为空设置，方便新建暂存公用方法
            dto = new ApiAppointmentInfoDto();
        }
        Appointment appointment = new Appointment();
        BeanUtils.copyProperties(dto,appointment);
        //生成预约信息主表唯一业务ID
        appointment.setAppointmentBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_APPOINTMENT.getCode()));
        //生成预约编号
        appointment.setAppointmentNo(CodeGenerator.generateCode(CodeGeneratorConstants.APPOINTMENT));
        //预约状态
        appointment.setStatus(status);
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success(appointment);
    }

    /**
     * 编辑预约提交 (聚合信息编辑预约提交)
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiAppointmentEditRequest request) {
        //编辑预约入参字段校验（非库校验）
        apiAppointmentCheckService.checkEditRequest(request);

        Integer status = null;
        Appointment appointmentCheck = null;
        if (!Objects.isNull(request.getApiAppointmentInfoDto())) {
            //校验预约信息是否存在
            Result<Appointment> result = checkAppointmentIsExist(request.getApiAppointmentInfoDto().getAppointmentBizId());
            appointmentCheck = result.getData();
            if (AppointmentStatusEnum.ZC.getItemValue().equals(appointmentCheck.getStatus())){
                //当前为暂存状态——>提交更新为待预约状态
                status = AppointmentStatusEnum.DYY.getItemValue();
            }else if (AppointmentStatusEnum.DYY.getItemValue().equals(appointmentCheck.getStatus())){
                if (Objects.isNull(request.getApiAppointmentInfoDto().getConfirmAppointmentTime())) {
                    //待预约提交到待签署状态，确定预约时间不能为空
                    throw new BusinessException("待预约提交确定预约时间不能为空!");
                }
                //当前为待预约状态——>提交更新为待签署状态-->到新单跟进
                status = AppointmentStatusEnum.DQS.getItemValue();
            }
            //其他状态下都是单纯编辑数据不做状态流转更新
        }

        //编辑预约提交-编辑预约信息主表数据
        Result<Appointment> appointmentResult = editAppointmentData(request.getApiAppointmentInfoDto(),status);
        Appointment appointment = appointmentResult.getData();

        //编辑产品计划信息表数据
        Result<ProductPlan> productPlanResult = apiProductPlanService.editProductPlanData(request.getApiProductPlanInfoDto(),appointment.getAppointmentBizId());
        ProductPlan productPlan = productPlanResult.getData();
        //批量编辑产品计划-附加险信息表数据
        apiAdditionalService.batchEditAdditionalData(request.getApiProductPlanInfoDto(),productPlan.getPlanBizId());

        //编辑投保人信息表数据
        apiPolicyholderService.editPolicyholderData(request.getApiPolicyholderInfoDto(),appointment.getAppointmentBizId());

        //编辑受保人信息表数据
        apiInsurantService.editInsurantData(request.getApiInsurantInfoDto(),appointment.getAppointmentBizId());

        //批量编辑受益人信息表数据
        apiBeneficiaryService.batchEditBeneficiaryData(request.getApiBeneficiaryInfoDtoList(),appointment.getAppointmentBizId());

        //编辑第二持有人信息表数据
        apiSecondHolderService.editSecondHolderData(request.getApiSecondHolderInfoDto(),appointment.getAppointmentBizId());

        if (!Objects.isNull(status) && AppointmentStatusEnum.DYY.getItemValue().equals(status)) {
            //当前为暂存状态——>提交更新为待预约状态需要更新Fna表的编号和业务id完成绑定关联
            //预约编号和预约业务id更新到FNA表（提交待预约状态，预约信息的预约业务id和预约编号更新到Fna表的预约业务id和预约编号）
            updateFnaBizIdAndNo(appointment.getFnaBizId(),appointment.getAppointmentBizId(),appointment.getAppointmentNo());
        }
        return Result.success();
    }

    /**
     * 确定预约时间提交 (流程流转到新单跟进)
     * @param request
     * @return
     */
    @Override
    public Result editConfirmTime(ApiAppointmentEditConfirmTimeRequest request) {
        //校验预约信息是否存在
        Result<Appointment> result = checkAppointmentIsExist(request.getAppointmentBizId());
        Appointment appointment = result.getData();
        appointment.setConfirmAppointmentTime(request.getConfirmAppointmentTime());
        //流转到新单跟进（这里的预约状态为待签署）
        appointment.setStatus(AppointmentStatusEnum.DQS.getItemValue());
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success();
    }

    /**
     * 单个对象编辑-编辑预约主体信息
     * @param apiAppointmentInfoDto
     * @return
     */
    @Override
    public Result singleEdit(ApiAppointmentInfoDto apiAppointmentInfoDto) {

        //编辑预约入参字段校验 - 预约信息字段校验
        apiAppointmentCheckService.checkEditApiAppointmentInfoDto(apiAppointmentInfoDto);

        //编辑预约-编辑预约信息主表数据
        editAppointmentData(apiAppointmentInfoDto,null);
        return Result.success();
    }

    /**
     * 编辑预约-编辑预约信息主表数据
     * @param dto
     * @return
     */
    public Result<Appointment> editAppointmentData(ApiAppointmentInfoDto dto,Integer status) {
        if (Objects.isNull(dto)) {
            //预约信息对象不能为空
            throw new BusinessException("预约信息对象不能为空");
        }
        //校验预约信息是否存在
        Result<Appointment> result = checkAppointmentIsExist(dto.getAppointmentBizId());
        Appointment appointment = result.getData();

        BeanUtils.copyProperties(dto,appointment);
        if (!Objects.isNull(status)) {
            //不为空设置状态
            appointment.setStatus(status);
        }
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success(appointment);
    }

    /**
     * 预约编辑关联FNA
     * @param request
     * @return
     */
    @Override
    public Result editFna(ApiAppointmentEditFnaRequest request) {
        //校验预约信息是否存在
        Result<Appointment> result = checkAppointmentIsExist(request.getAppointmentBizId());
        Appointment appointment = result.getData();

        //修改预约信息表冗余的关联FNA业务id和关联FNA编号
        appointment.setFnaBizId(request.getFnaBizId());
        appointment.setFnaNo(request.getFnaNo());
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success();
    }

    /**
     * 预约解除关联FNA
     * @param request
     * @return
     */
    @Override
    public Result removeFna(ApiAppointmentRemoveFnaRequest request) {
        //校验预约信息是否存在
        Result<Appointment> result = checkAppointmentIsExist(request.getAppointmentBizId());
        Appointment appointment = result.getData();

        //预约解除关联FNA就是清空预约信息冗余的关联FNA业务id和关联FNA编号数据
        appointment.setFnaBizId("");
        appointment.setFnaNo("");
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success();
    }

    /**
     * 预约编辑关联计划书
     * @param request
     * @return
     */
    @Override
    public Result editProposal(ApiAppointmentEditProposalRequest request) {
        //校验预约信息是否存在
        Result<Appointment> result = checkAppointmentIsExist(request.getAppointmentBizId());
        Appointment appointment = result.getData();

        //修改预约信息表冗余的关联计划书业务id和关联计划书编号
        appointment.setProposalBizId(request.getProposalBizId());
        appointment.setProposalNo(request.getProposalNo());
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success();
    }

    /**
     * 预约解除关联计划书
     * @param request
     * @return
     */
    @Override
    public Result removeProposal(ApiAppointmentRemoveProposalRequest request) {
        //校验预约信息是否存在
        Result<Appointment> result = checkAppointmentIsExist(request.getAppointmentBizId());
        Appointment appointment = result.getData();

        //预约预约解除关联计划书就是清空预约信息冗余的关联计划书业务id和关联计划书编号
        appointment.setProposalBizId("");
        appointment.setProposalNo("");
        iAppointmentService.saveOrUpdate(appointment);
        return Result.success();
    }


    /**
     * 校验预约信息是否存在
     * @param appointmentBizId
     * @return
     */
    public Result<Appointment> checkAppointmentIsExist(String appointmentBizId) {
        Appointment appointment = iAppointmentService.queryOne(appointmentBizId);
        if (Objects.isNull(appointment)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(appointment);
    }

}
