package com.yd.csf.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiAppointmentCheckService;
import com.yd.csf.api.service.ApiPolicyholderService;
import com.yd.csf.feign.dto.appointment.ApiPolicyholderInfoDto;
import com.yd.csf.feign.utils.CommonUtil;
import com.yd.csf.service.model.Policyholder;
import com.yd.csf.service.service.IPolicyholderService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.util.Objects;

/**
 * 投保人信息业务实现类
 */
@Slf4j
@Service
public class ApiPolicyholderServiceImpl implements ApiPolicyholderService {

    @Autowired
    private IPolicyholderService iPolicyholderService;

    @Autowired
    private ApiAppointmentCheckService apiAppointmentCheckService;

    /**
     * 投保人信息详情
     * @param appointmentBizId 预约信息主表唯一业务ID
     * @param policyholderBizId 投保人信息表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiPolicyholderInfoDto> detail(String appointmentBizId,
                                                 String policyholderBizId) {
        Policyholder policyholder = iPolicyholderService.queryOne(appointmentBizId,policyholderBizId);
        ApiPolicyholderInfoDto dto = null;
        if (!Objects.isNull(policyholder)) {
            dto = new ApiPolicyholderInfoDto();
            BeanUtils.copyProperties(policyholder,dto);
            //地址列表
            dto.setAddressList(CommonUtil.getAddressList(policyholder.getAddressList()));
        }
        return Result.success(dto);
    }

    /**
     * 编辑投保人信息
     * @param apiPolicyholderInfoDto
     * @return
     */
    @Override
    public Result edit(ApiPolicyholderInfoDto apiPolicyholderInfoDto) {
        //编辑预约入参字段校验 - 投保人信息字段校验
        apiAppointmentCheckService.checkEditApiPolicyholderInfoDto(apiPolicyholderInfoDto);
        //编辑投保人信息表数据
//        editPolicyholderData(apiPolicyholderInfoDto,apiPolicyholderInfoDto.getAppointmentBizId());
        return Result.success();
    }

    /**
     * 添加投保人信息表数据
     * @param dto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<Policyholder> addPolicyholderData(ApiPolicyholderInfoDto dto, String appointmentBizId) {
        if (Objects.isNull(dto)){
            //为空设置，方便新建暂存公用方法
            dto = new ApiPolicyholderInfoDto();
        }
        //投保人信息信息
        Policyholder policyholder = new Policyholder();
        BeanUtils.copyProperties(dto,policyholder);
        //生成投保人信息表唯一业务ID
        policyholder.setPolicyholderBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_POLICYHOLDER.getCode()));
        //预约信息主表唯一业务ID
        policyholder.setAppointmentBizId(appointmentBizId);
        //地址信息（json串）
        policyholder.setAddressList(CommonUtil.getAddressListJsonStr(dto.getAddressList()));
        iPolicyholderService.saveOrUpdate(policyholder);
        return Result.success(policyholder);
    }

    /**
     * 编辑投保人信息表数据
     * @param dto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<Policyholder> editPolicyholderData(ApiPolicyholderInfoDto dto, String appointmentBizId) {
        if (Objects.isNull(dto)) {
            //投保人信息对象不能为空
            throw new BusinessException("投保人信息对象不能为空");
        }
        if (StringUtils.isBlank(dto.getPolicyholderBizId())) {
            throw new BusinessException("投保人信息表唯一业务ID不能为空");
        }
        //校验投保人信息是否存在
        Result<Policyholder> result = checkPolicyholderIsExist(dto.getPolicyholderBizId());
        Policyholder policyholder = result.getData();

        BeanUtils.copyProperties(dto,policyholder);
        policyholder.setAppointmentBizId(appointmentBizId);
        //地址信息（json串）
        policyholder.setAddressList(CommonUtil.getAddressListJsonStr(dto.getAddressList()));
        iPolicyholderService.saveOrUpdate(policyholder);
        return Result.success(policyholder);
    }

    /**
     * 校验投保人信息是否存在
     * @param policyholderBizId
     * @return
     */
    public Result<Policyholder> checkPolicyholderIsExist(String policyholderBizId) {
        Policyholder policyholder = iPolicyholderService.queryOne(policyholderBizId);
        if (Objects.isNull(policyholder)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(policyholder);
    }

}
