package com.yd.csf.api.service.impl;

import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiAppointmentCheckService;
import com.yd.csf.api.service.ApiSecondHolderService;
import com.yd.csf.feign.dto.appointment.ApiSecondHolderInfoDto;
import com.yd.csf.service.model.SecondHolder;
import com.yd.csf.service.service.ISecondHolderService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import java.util.Objects;

/**
 * 第二持有人信息业务实现类
 */
@Slf4j
@Service
public class ApiSecondHolderServiceImpl implements ApiSecondHolderService {

    @Autowired
    private ISecondHolderService iSecondHolderService;

    @Autowired
    private ApiAppointmentCheckService apiAppointmentCheckService;

    /**
     * 第二持有人信息详情
     * @param appointmentBizId 预约信息主表唯一业务ID
     * @param secondHolderBizId 第二持有人信息表唯一业务ID
     * @return
     */
    @Override
    public Result<ApiSecondHolderInfoDto> detail(String appointmentBizId, String secondHolderBizId) {
        SecondHolder secondHolder = iSecondHolderService.queryOne(appointmentBizId,secondHolderBizId);
        ApiSecondHolderInfoDto dto = null;
        if (!Objects.isNull(secondHolder)) {
            dto = new ApiSecondHolderInfoDto();
            BeanUtils.copyProperties(secondHolder,dto);
        }
        return Result.success(dto);
    }

    /**
     * 编辑第二持有人信息
     * @param apiSecondHolderInfoDto
     * @return
     */
    @Override
    public Result edit(ApiSecondHolderInfoDto apiSecondHolderInfoDto) {
        //编辑预约入参字段校验 - 第二持有人信息字段校验
        apiAppointmentCheckService.checkEditApiSecondHolderInfoDto(apiSecondHolderInfoDto);

        if (Objects.isNull(apiSecondHolderInfoDto)) {
            apiSecondHolderInfoDto = new ApiSecondHolderInfoDto();
        }
        editSecondHolderData(apiSecondHolderInfoDto,apiSecondHolderInfoDto.getAppointmentBizId());
        return Result.success();
    }

    /**
     * 添加第二持有人信息表数据
     * @param dto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<SecondHolder> addSecondHolderData(ApiSecondHolderInfoDto dto, String appointmentBizId) {
        if (Objects.isNull(dto)){
            //为空设置，方便新建暂存公用方法
            dto = new ApiSecondHolderInfoDto();
        }
        //投保人信息信息
        SecondHolder secondHolder = new SecondHolder();
        BeanUtils.copyProperties(dto,secondHolder);
        //第二持有人信息表唯一业务ID
        secondHolder.setSecondHolderBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_SECOND_HOLDER.getCode()));
        //预约信息主表唯一业务ID
        secondHolder.setAppointmentBizId(appointmentBizId);
        iSecondHolderService.saveOrUpdate(secondHolder);
        return Result.success(secondHolder);
    }

    /**
     * 编辑第二持有人信息表数据
     * @param dto
     * @param appointmentBizId
     * @return
     */
    @Override
    public Result<SecondHolder> editSecondHolderData(ApiSecondHolderInfoDto dto, String appointmentBizId) {
        if (Objects.isNull(dto)){
            //为空
            dto = new ApiSecondHolderInfoDto();
        }
        if (StringUtils.isBlank(dto.getSecondHolderBizId())) {
            //为空新增（因为首次发起新增预约，第二持有人没有填写为空的，第二次编辑做了修改这种情况是新增操作）
            return addSecondHolderData(dto,appointmentBizId);
        }else {
            //校验第二持有人信息是否存在
            Result<SecondHolder> result = checkSecondHolderIsExist(dto.getSecondHolderBizId());
            //第二持有人信息
            SecondHolder secondHolder = result.getData();
            BeanUtils.copyProperties(dto,secondHolder);
            //预约信息主表唯一业务ID
            secondHolder.setAppointmentBizId(appointmentBizId);
            iSecondHolderService.saveOrUpdate(secondHolder);
            return Result.success(secondHolder);
        }
    }

    /**
     * 校验第二持有人信息是否存在
     * @param secondHolderBizId
     * @return
     */
    public Result<SecondHolder> checkSecondHolderIsExist(String secondHolderBizId) {
        SecondHolder secondHolder = iSecondHolderService.queryOne(secondHolderBizId);
        if (Objects.isNull(secondHolder)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(secondHolder);
    }
}
