package com.yd.csf.api.controller;

import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.csf.api.dto.CommissionExcelDTO;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.*;
import com.yd.csf.service.enums.CommissionExpectedStatusEnum;
import com.yd.csf.service.enums.CommissionStatusEnum;
import com.yd.csf.service.model.Commission;
import com.yd.csf.service.model.CommissionExpected;
import com.yd.csf.service.service.CommissionExpectedService;
import com.yd.csf.service.service.CommissionService;
import com.yd.csf.service.vo.CommissionVO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 保单来佣接口
 *
 * @author jianan
 * @since 2025-09-15
 */
@RestController
@RequestMapping("/commission")
@Tag(name = "保单来佣接口")
public class ApiCommissionController {

    private static final Logger log = LoggerFactory.getLogger(ApiCommissionController.class);

    @Resource
    private CommissionService commissionService;

    @Resource
    private CommissionExpectedService commissionExpectedService;


    @PostMapping("/upload/excel")
    @Transactional(rollbackFor = Exception.class)
    public Result<Boolean> uploadExcel(@RequestParam("file") MultipartFile file) throws IOException {
        List<CommissionExcelDTO> dataList = new ArrayList<>();
        try {
            dataList = EasyExcel.read(file.getInputStream())
                    .head(CommissionExcelDTO.class)
                    .sheet("来佣")
                    .doReadSync();

        } catch (IOException e) {
            throw new RuntimeException("Excel读取失败", e);
        }

        // 获取当前登录用户的ID
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        // 数据验证和处理
        processData(dataList, loginUserId);

        return Result.success(true);
    }

    /**
     * 1.处理导入的数据 2.比对预计来佣 3.保存到数据库
     */
    private void processData(List<CommissionExcelDTO> dataList, String loginUserId) {
        List<Commission> entities = new ArrayList<>();
        for (CommissionExcelDTO data : dataList) {
            // 数据验证
            Commission entity = CommissionExcelDTO.convertToEntity(data, loginUserId);
            entities.add(entity);
        }

        // 根据导入的来佣获取保单号集合
        List<String> policyNoList = entities.stream()
                .map(Commission::getPolicyNo)
                .collect(Collectors.toList());
        // 根据保单号查询预计来佣
        List<CommissionExpected> expectedList = commissionExpectedService.lambdaQuery()
                .in(CommissionExpected::getPolicyNo, policyNoList)
                .list();

        // 校验导入的来佣是否与预计来佣一致
        for (Commission commission : entities) {
            String policyNo = commission.getPolicyNo();
            Integer commissionPeriod = commission.getCommissionPeriod();
            String commissionName = commission.getCommissionName();

            CommissionExpected expected = getCommissionExpected(expectedList, policyNo, commissionPeriod, commissionName);
            if (expected != null) {
                if (expected.getAmount().compareTo(commission.getAmount()) == 0) {
                    // 一致，来佣状态设置为 比对成功
                    commission.setStatus(CommissionStatusEnum.COMPARE_SUCCESS.getItemValue());
                    // 预计来佣的状态设置为 已来佣
                    expected.setStatus(CommissionExpectedStatusEnum.COMPARED.getItemValue());
                } else {
                    // 不一致，来佣状态设置为 比对失败
                    commission.setStatus(CommissionStatusEnum.COMPARE_FAIL.getItemValue());
                    expected.setStatus(CommissionExpectedStatusEnum.PARTIAL.getItemValue());
                }
            }
        }

        // 保存来佣数据
        commissionService.saveBatch(entities);
        // 更新预计来佣状态
        commissionExpectedService.updateBatchById(expectedList);
    }

    private static CommissionExpected getCommissionExpected(List<CommissionExpected> expectedList, String policyNo, Integer commissionPeriod, String commissionName) {
        return expectedList.stream()
                .filter(item -> item.getPolicyNo().equals(policyNo)
                        && commissionPeriod.equals(item.getCommissionPeriod())
                        && commissionName.equals(item.getCommissionName()))
                .findFirst()
                .orElse(null);
    }

    /**
     * 生成可出账（发佣）记录，支持手动复选框选择
     *
     * @param generateFortuneRequest
     * @param request
     * @return
     */
    @PostMapping("/generate/fortune")
    @Operation(summary = "生成可出账（发佣）记录，支持手动复选框选择")
    public Result<Boolean> generateFortune(@RequestBody GenerateFortuneRequest generateFortuneRequest,
                                                            HttpServletRequest request) {
        List<String> commissionBizIdList = generateFortuneRequest.getCommissionBizIdList();
        if (CollectionUtils.isEmpty(commissionBizIdList)) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), "commissionBizIdList不能为空");
        }
        return Result.success(commissionService.generateFortune(generateFortuneRequest));
    }


    /**
     * 创建保单来佣
     *
     * @param customerAddRequest
     * @param request
     * @return
     */
    @Operation(summary = "创建保单来佣")
    @PostMapping("/add")
    public Result<Map<String, Object>> addCommission(@RequestBody CommissionAddRequest customerAddRequest, HttpServletRequest request) {
        if (customerAddRequest == null) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(commissionService.addCommission(customerAddRequest));
    }

    /**
     * 删除来佣
     *
     * @param deleteRequest
     * @param request
     * @return
     */
    @PostMapping("/delete")
    public Result<Boolean> deleteCommission(@RequestBody CommissionDeleteRequest deleteRequest, HttpServletRequest request) {
        if (StringUtils.isBlank(deleteRequest.getCommissionBizId())) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        String commissionBizId = deleteRequest.getCommissionBizId();

        // 获取当前登录用户的ID
        AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
        String loginUserId = currentLoginUser.getId().toString();

        // 判断是否存在
        Commission oldCommission = commissionService.getByCommissionBizId(commissionBizId);
        if (oldCommission == null) {
            return Result.fail(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }
        // 仅本人或管理员可删除
//        if (!oldCommission.getUserId().equals(loginUserId) && !userService.isAdmin(request)) {
//            throw new BusinessException(ErrorCode.NO_AUTH_ERROR);
//        }
        // 操作数据库
        boolean result = commissionService.removeById(oldCommission.getId());
        if (!result) {
            return Result.fail(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
        }
        return Result.success(result);
    }

    /**
     * 更新保单来佣
     *
     * @param commissionUpdateRequest
     * @return
     */
    @PostMapping("/update")
    @Operation(summary = "更新保单来佣信息")
    public Result<Boolean> updateCommission(@RequestBody CommissionUpdateRequest commissionUpdateRequest) {
        if (commissionUpdateRequest == null || commissionUpdateRequest.getCommissionBizId() == null) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(commissionService.updateCommission(commissionUpdateRequest));
    }

    /**
     * 根据 commissionBizId 获取保单来佣（封装类）
     *
     * @param commissionBizId
     * @return
     */
//    @GetMapping("/get/vo")
//    @Operation(summary = "根据 policyBizId 获取保单来佣详情")
//    public Result<PolicyFollowVO> getPolicyFollowByPolicyBizId(@RequestParam("commissionBizId") String commissionBizId, HttpServletRequest request) {
//        if (commissionBizId == null) {
//            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
//        }
//        // 查询数据库
//        PolicyFollow policyFollow = policyFollowService.getByPolicyBizId(policyBizId);
//        if (policyFollow == null) {
//            return Result.fail(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
//        }
//
//        // 获取封装类
//        return Result.success(policyFollowService.getPolicyFollowVO(policyFollow));
//    }

    /**
     * 分页获取保单来佣列表（仅管理员可用）
     *
     * @param fnaQueryRequest
     * @return
     */
//    @PostMapping("/list/page")
//    public Result<Page<Customer>> listFnaByPage(@RequestBody FnaQueryRequest fnaQueryRequest) {
//        long current = fnaQueryRequest.getPageNo();
//        long size = fnaQueryRequest.getPageSize();
//        // 查询数据库
//        Page<Customer> fnaPage = policyFollowService.page(new Page<>(current, size),
//                policyFollowService.getQueryWrapper(fnaQueryRequest));
//        return Result.success(fnaPage);
//    }

    /**
     * 分页获取保单来佣列表（VO）
     *
     * @param commissionQueryRequest
     * @param request
     * @return
     */
    @PostMapping("/list/page/vo")
    @Operation(summary = "分页获取保单来佣列表")
    public Result<Page<CommissionVO>> listPolicyFollowByPage(@RequestBody CommissionQueryRequest commissionQueryRequest,
                                                             HttpServletRequest request) {
        long current = commissionQueryRequest.getPageNo();
        long size = commissionQueryRequest.getPageSize();

        // 查询数据库
        Page<Commission> commissionPage = commissionService.page(new Page<>(current, size),
                commissionService.getQueryWrapper(commissionQueryRequest));
        // 获取封装类
        return Result.success(commissionService.getCommissionVOPage(commissionPage));
    }

}
