package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.constant.CodeGeneratorConstants;
import com.yd.common.enums.CommonEnum;
import com.yd.common.enums.ResultCode;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.CodeGenerator;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiConditionTypeService;
import com.yd.csf.feign.request.conditiontype.ApiConditionTypeAddRequest;
import com.yd.csf.feign.request.conditiontype.ApiConditionTypeEditRequest;
import com.yd.csf.feign.request.conditiontype.ApiConditionTypePageRequest;
import com.yd.csf.feign.response.conditiontype.ApiConditionTypeDetailResponse;
import com.yd.csf.feign.response.conditiontype.ApiConditionTypePageResponse;
import com.yd.csf.service.dto.ConditionTypeDto;
import com.yd.csf.service.model.ConditionType;
import com.yd.csf.service.service.IConditionTypeService;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.List;
import java.util.Objects;

@Service
public class ApiConditionTypeServiceImpl implements ApiConditionTypeService {

    @Autowired
    private IConditionTypeService iConditionTypeService;

    /**
     * 分页查询-条件类型信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiConditionTypePageResponse>> page(ApiConditionTypePageRequest request) {
        Page<ApiConditionTypePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiConditionTypePageResponse> iPage = iConditionTypeService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 详情-条件类型信息
     * @param conditionTypeBizId
     * @return
     */
    @Override
    public Result<ApiConditionTypeDetailResponse> detail(String conditionTypeBizId) {
        Result<ConditionType> result = checkConditionTypeIsExist(conditionTypeBizId);
        ConditionType conditionType = result.getData();
        ApiConditionTypeDetailResponse response = new ApiConditionTypeDetailResponse();
        BeanUtils.copyProperties(conditionType,response);
        return Result.success(response);
    }

    /**
     * 添加-条件类型信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiConditionTypeAddRequest request) {
        //校验条件类型名称唯一性
        List<ConditionType> conditionTypeList = iConditionTypeService.queryList(ConditionTypeDto.builder()
                .name(request.getName())
                .build());
        if (!CollectionUtils.isEmpty(conditionTypeList)) {
            throw new BusinessException("条件类型名称已存在");
        }
        ConditionType conditionType = new ConditionType();
        conditionType.setConditionTypeBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_CONDITION_TYPE.getCode()));
        //编码
        conditionType.setCode(CodeGenerator.generate10Code(CodeGeneratorConstants.PREFIX_CONDITION_TYPE_CODE));
        iConditionTypeService.saveOrUpdate(conditionType);
        return Result.success();
    }

    /**
     * 编辑-条件类型信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiConditionTypeEditRequest request) {
        Result<ConditionType> result = checkConditionTypeIsExist(request.getConditionTypeBizId());
        ConditionType conditionType = result.getData();
        //校验条件类型名称唯一性
        List<ConditionType> conditionTypeList = iConditionTypeService.queryList(ConditionTypeDto.builder()
                .name(request.getName())
                .isExcludeMy(true)
                .conditionTypeBizId(request.getConditionTypeBizId())
                .build());
        if (!CollectionUtils.isEmpty(conditionTypeList)) {
            throw new BusinessException("条件类型名称已存在");
        }
        iConditionTypeService.saveOrUpdate(conditionType);
        return Result.success();
    }

    /**
     * 删除-条件类型信息 TODO
     * @param conditionTypeBizId
     * @return
     */
    @Override
    public Result del(String conditionTypeBizId) {
        return null;
    }

    /**
     * 校验条件类型信息是否存在
     * @param conditionTypeBizId
     * @return
     */
    public Result<ConditionType> checkConditionTypeIsExist(String conditionTypeBizId) {
        ConditionType conditionType = iConditionTypeService.queryOne(conditionTypeBizId);
        if (Objects.isNull(conditionType)) {
            //数据不存在
            throw new BusinessException(ResultCode.NULL_ERROR.getCode(),ResultCode.NULL_ERROR.getMessage());
        }
        return Result.success(conditionType);
    }
}
