package com.yd.csf.api.controller;

import com.alibaba.excel.EasyExcel;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.auth.core.dto.AuthUserDto;
import com.yd.auth.core.utils.SecurityUtil;
import com.yd.common.result.Result;
import com.yd.csf.api.dto.CommissionExcelDTO;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.CommissionQueryRequest;
import com.yd.csf.service.dto.CommissionUpdateRequest;
import com.yd.csf.service.dto.GenerateFortuneRequest;
import com.yd.csf.service.model.Commission;
import com.yd.csf.service.service.CommissionService;
import com.yd.csf.service.vo.CommissionVO;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.apache.commons.collections4.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * 保单来佣接口
 *
 * @author jianan
 * @since 2025-09-15
 */
@RestController
@RequestMapping("/commission")
@Tag(name = "保单来佣接口")
public class ApiCommissionController {

    private static final Logger log = LoggerFactory.getLogger(ApiCommissionController.class);

    @Resource
    private CommissionService commissionService;


    @PostMapping("/upload/excel")
    @Transactional(rollbackFor = Exception.class)
    public Result<Boolean> uploadExcel(@RequestParam("file") MultipartFile file) throws IOException {
        try {
            List<CommissionExcelDTO> dataList = EasyExcel.read(file.getInputStream())
                    .head(CommissionExcelDTO.class)
                    .sheet("来佣")
                    .doReadSync();

            // 获取当前登录用户的ID
            AuthUserDto currentLoginUser = SecurityUtil.getCurrentLoginUser();
            String loginUserId = currentLoginUser.getId().toString();

            // 数据验证和处理
            processData(dataList, loginUserId);

        } catch (IOException e) {
            throw new RuntimeException("Excel读取失败", e);
        }

        return Result.success(true);
    }

    /**
     * 处理导入的数据
     */
    private void processData(List<CommissionExcelDTO> dataList, String loginUserId) {
        List<Commission> entities = new ArrayList<>();
        for (CommissionExcelDTO data : dataList) {
            // 数据验证
            Commission entity = CommissionExcelDTO.convertToEntity(data, loginUserId);
            entities.add(entity);
        }
        // 批量保存
        commissionService.saveBatch(entities);
    }

    /**
     * 生成可出账（发佣）记录，支持手动复选框选择
     *
     * @param generateFortuneRequest
     * @param request
     * @return
     */
    @PostMapping("/generate/fortune")
    @Operation(summary = "生成可出账（发佣）记录，支持手动复选框选择")
    public Result<Boolean> generateFortune(@RequestBody GenerateFortuneRequest generateFortuneRequest,
                                                            HttpServletRequest request) {
        List<String> commissionBizIdList = generateFortuneRequest.getCommissionBizIdList();
        if (CollectionUtils.isEmpty(commissionBizIdList)) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), "commissionBizIdList不能为空");
        }
        return Result.success(commissionService.generateFortune(generateFortuneRequest));
    }


    /**
     * 创建保单来佣
     *
     * @param customerAddRequest
     * @param request
     * @return
     */
//    @Operation(summary = "创建保单来佣")
//    @PostMapping("/add")
//    @Transactional(rollbackFor = Exception.class)
//    public Result<Map<String, Object>> addCustomer(@RequestBody CustomerAddRequest customerAddRequest, HttpServletRequest request) {
//        if (customerAddRequest == null) {
//            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
//        }
//        return Result.success(policyFollowService.addCustomer(customerAddRequest));
//    }

    /**
     * 删除fna
     *
     * @param deleteRequest
     * @param request
     * @return
     */
//    @PostMapping("/delete")
//    public Result<Boolean> deleteFna(@RequestBody DeleteRequest deleteRequest, HttpServletRequest request) {
//        if (deleteRequest == null || deleteRequest.getId() <= 0) {
//            throw new BusinessException(ErrorCode.PARAMS_ERROR);
//        }
//        User user = userService.getLoginUser(request);
//        long id = deleteRequest.getId();
//        // 判断是否存在
//        Customer oldFna = policyFollowService.getById(id);
//        ThrowUtils.throwIf(oldFna == null, ErrorCode.NOT_FOUND_ERROR);
//        // 仅本人或管理员可删除
//        if (!oldFna.getUserId().equals(user.getId()) && !userService.isAdmin(request)) {
//            throw new BusinessException(ErrorCode.NO_AUTH_ERROR);
//        }
//        // 操作数据库
//        boolean result = policyFollowService.removeById(id);
//        ThrowUtils.throwIf(!result, ErrorCode.OPERATION_ERROR);
//        return Result.success(true);
//    }

    /**
     * 更新保单来佣
     *
     * @param commissionUpdateRequest
     * @return
     */
    @PostMapping("/update")
    @Operation(summary = "更新保单来佣信息")
    public Result<Boolean> updateCommission(@RequestBody CommissionUpdateRequest commissionUpdateRequest) {
        if (commissionUpdateRequest == null || commissionUpdateRequest.getCommissionBizId() == null) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(commissionService.updateCommission(commissionUpdateRequest));
    }

    /**
     * 根据 commissionBizId 获取保单来佣（封装类）
     *
     * @param commissionBizId
     * @return
     */
//    @GetMapping("/get/vo")
//    @Operation(summary = "根据 policyBizId 获取保单来佣详情")
//    public Result<PolicyFollowVO> getPolicyFollowByPolicyBizId(@RequestParam("commissionBizId") String commissionBizId, HttpServletRequest request) {
//        if (commissionBizId == null) {
//            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
//        }
//        // 查询数据库
//        PolicyFollow policyFollow = policyFollowService.getByPolicyBizId(policyBizId);
//        if (policyFollow == null) {
//            return Result.fail(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
//        }
//
//        // 获取封装类
//        return Result.success(policyFollowService.getPolicyFollowVO(policyFollow));
//    }

    /**
     * 分页获取保单来佣列表（仅管理员可用）
     *
     * @param fnaQueryRequest
     * @return
     */
//    @PostMapping("/list/page")
//    public Result<Page<Customer>> listFnaByPage(@RequestBody FnaQueryRequest fnaQueryRequest) {
//        long current = fnaQueryRequest.getPageNo();
//        long size = fnaQueryRequest.getPageSize();
//        // 查询数据库
//        Page<Customer> fnaPage = policyFollowService.page(new Page<>(current, size),
//                policyFollowService.getQueryWrapper(fnaQueryRequest));
//        return Result.success(fnaPage);
//    }

    /**
     * 分页获取保单来佣列表（VO）
     *
     * @param commissionQueryRequest
     * @param request
     * @return
     */
    @PostMapping("/list/page/vo")
    @Operation(summary = "分页获取保单来佣列表")
    public Result<Page<CommissionVO>> listPolicyFollowByPage(@RequestBody CommissionQueryRequest commissionQueryRequest,
                                                             HttpServletRequest request) {
        long current = commissionQueryRequest.getPageNo();
        long size = commissionQueryRequest.getPageSize();

        // 查询数据库
        Page<Commission> commissionPage = commissionService.page(new Page<>(current, size),
                commissionService.getQueryWrapper(commissionQueryRequest));
        // 获取封装类
        return Result.success(commissionService.getCommissionVOPage(commissionPage));
    }

}
