package com.yd.csf.api.service.impl;

import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.csf.api.dto.*;
import com.yd.csf.api.service.ApiAgentDetailFycService;
import com.yd.csf.api.service.ApiBasicLawCalculateService;
import com.yd.csf.api.utils.FormulaParser;
import com.yd.csf.feign.enums.VariableEnum;
import com.yd.csf.feign.request.basiclawcalculate.ApiGenerateBillingRequest;
import com.yd.csf.feign.response.basiclawcalculate.ApiGenerateBillingResponse;
import com.yd.csf.service.dto.*;
import com.yd.csf.service.model.*;
import com.yd.csf.service.service.*;
import com.yd.insurance.base.feign.client.ApiRelProductAnnouncementFeignClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import javax.script.ScriptException;
import javax.sql.DataSource;
import java.math.BigDecimal;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiBasicLawCalculateServiceImpl implements ApiBasicLawCalculateService {

    @Autowired
    private PolicyService policyService;

    @Autowired
    private CommissionService commissionService;

    @Autowired
    private ApiRelProductAnnouncementFeignClient apiRelProductAnnouncementFeignClient;

    @Autowired
    private PolicyBrokerService policyBrokerService;

    @Autowired
    private IFormulaConfigService iFormulaConfigService;

    @Autowired
    private IVariableService iVariableService;

    @Autowired
    private IRelObjectConditionService iRelObjectConditionService;

    @Autowired
    private IRelObjectSqlService iRelObjectSqlService;

    @Autowired
    private ICommissionSqlTemplateService iCommissionSqlTemplateService;

    @Autowired
    private IRuleItemConfigService iRuleItemConfigService;

    @Autowired
    private IRelObjectFormulaService iRelObjectFormulaService;

    @Autowired
    private ICommissionRuleBindingService iCommissionRuleBindingService;

    @Autowired
    private ApiAgentDetailFycService apiAgentDetailFycService;

    /**
     * 获取JdbcTemplate（需要注入）
     */
    @Autowired
    private DataSource dataSource;

    private NamedParameterJdbcTemplate getJdbcTemplate() {
        return new NamedParameterJdbcTemplate(dataSource);
    }

    /**
     * 生成出账
     * @param request
     * @return
     */
    @Override
    public Result<ApiGenerateBillingResponse> generateBilling(ApiGenerateBillingRequest request) {
        //根据保单来佣业务id列表查询保单业务ID列表
        List<Commission> commissionList = commissionService.queryList(CommissionDto.builder().commissionBizIdList(request.getCommissionBizIdList()).build());
        if (CollectionUtils.isEmpty(commissionList)) {
            throw new BusinessException("来佣数据不存在");
        }
        //根据保单号分组，取每组的第一条, 获取最后的去重的来佣列表
        List<Commission> distinctCommissionList = commissionList.stream()
                .collect(Collectors.collectingAndThen(
                        Collectors.toMap(
                                Commission::getPolicyNo,  //保单号作为key
                                Function.identity(),      //Commission对象本身作为value
                                (existing, replacement) -> existing  //遇到重复key时保留已存在的
                        ),
                        map -> new ArrayList<>(map.values())));
        //获取保单号列表
        List<String> policyNoList = distinctCommissionList.stream().map(Commission::getPolicyNo).collect(Collectors.toList());
        //查询保单列表信息
        List<Policy> policyList =  policyService.queryList(PolicyDto.builder().policyNoList(policyNoList).build());

        if (!CollectionUtils.isEmpty(policyList)) {
            //查询保单转介人（销售人员）关系数据
            List<PolicyBroker> policyBrokerList = policyBrokerService.queryList(QueryPolicyBrokerDto.builder().policyNoList(policyNoList).build());
            for (Policy policy : policyList) {
                List<PolicyBroker> thisPolicyBrokerList = new ArrayList<>();
                //当前保单的绑定的转介人（销售人员）列表
                if (!CollectionUtils.isEmpty(policyBrokerList)) {
                    thisPolicyBrokerList = policyBrokerList.stream().filter(d -> d.getPolicyNo().equals(policy.getPolicyNo())).collect(Collectors.toList());
                }
                if (!CollectionUtils.isEmpty(thisPolicyBrokerList)) {
                    //遍历当前保单绑定的转介人列表，根据人绑定的基本法项目（计算销售佣金，推荐佣金，辅导员佣金，终身推荐奖等。）绑定什么项目算什么金额，按照佣金项目执行顺序来计算。
                    for (PolicyBroker policyBroker : thisPolicyBrokerList) {
                        //构造销售佣金基本法项目的顺序下标值执行
                        List<Integer> executionOrderList = new ArrayList<>();
                        executionOrderList.add(1);

                        //保单出账先执行销售佣金基本法项目，入完到积分明细表里后，再执行保单绑定人的其他基本法项目
                        Result<List<AlgorithmResDto>> result = policyBrokerAlgorithm(AlgorithmDto.builder()
                                .brokerBizId(policyBroker.getBrokerBizId())
                                .executionOrderList(executionOrderList)
                                .isNegateExecutionOrderList(false)
                                .build());

                        //生成业务员（转介人）积分明细表记录（销售积分）
                        generateAgentDetailFyc(GenerateAgentDetailFycDto.builder()
                                .agentId(policyBroker.getBrokerBizId())
                                .policyNo(policy.getPolicyNo())
                                .algorithmResDtoList(result.getData())
//                                .sourceType()
                                .build());

                        //生成保单发佣表记录

                    }
                }
            }
        }
        return null;
    }

    public Result saveBatchFortune() {

    }
    /**
     * 生成业务员（转介人）积分明细表记录
     * @return
     */
    public Result generateAgentDetailFyc(GenerateAgentDetailFycDto dto){
        //基本法计算 - 保存积分明细表
        apiAgentDetailFycService.saveAgentDetailFyc(dto);
        return Result.success();
    }

    /**
     * 算法-计算-转介人（销售业务员）绑定的基本法类型（基本法项目列表）对应的积分值
     * @param algorithmDto
     * @return
     */
    public Result<List<AlgorithmResDto>> policyBrokerAlgorithm(AlgorithmDto algorithmDto) {
        //校验 - 转介人业务ID（客户端用户表唯一业务ID）不能为空
        if (StringUtils.isBlank(algorithmDto.getBrokerBizId())) {
            throw new BusinessException("转介人业务ID（客户端用户表唯一业务ID）不能为空");
        }
        List<CommissionRuleBinding> commissionRuleBindingList = iCommissionRuleBindingService.queryList(CommissionRuleBindingDto.builder().targetId(algorithmDto.getBrokerBizId()).build());
        if (CollectionUtils.isEmpty(commissionRuleBindingList)) {
            throw new BusinessException("业务员基本法绑定信息不存在");
        }
        CommissionRuleBinding commissionRuleBinding = commissionRuleBindingList.get(0);
        algorithmDto.setRuleBizId(commissionRuleBinding.getRuleBizId());
        return commissionRuleAlgorithm(algorithmDto);
    }

    /**
     * 算法-计算-基本法类型
     * @param algorithmDto 算法DTO - 公共入参封装
     * @return
     */
    public Result<List<AlgorithmResDto>> commissionRuleAlgorithm(AlgorithmDto algorithmDto) {
        //校验 - 基本法配置表唯一业务ID不能为空
        if (StringUtils.isBlank(algorithmDto.getRuleBizId())) {
            throw new BusinessException("基本法配置表唯一业务ID不能为空");
        }
        //查询基本法类型绑定的基本法项目列表 - 执行顺序，数值越小越先执行，用于控制佣金项目的计算顺序
        List<RuleItemConfig> ruleItemConfigList = iRuleItemConfigService.queryList(RuleItemConfigDto.builder().ruleBizId(algorithmDto.getRuleBizId()).build());
        if (CollectionUtils.isEmpty(ruleItemConfigList)) {
            throw new BusinessException("基本法项目列表不存在");
        }
        List<AlgorithmResDto> algorithmResDtoList = new ArrayList<>();
        //遍历基本法项目 - 计算对应值
        for (RuleItemConfig ruleItemConfig : ruleItemConfigList) {
            if (!CollectionUtils.isEmpty(algorithmDto.getExecutionOrderList())) {
                //基本法项目执行顺序和是否在需要执行的基本法项目的执行顺序下标值列表中
                List<Integer> isExistList = algorithmDto.getExecutionOrderList().stream().filter(s -> s.equals(ruleItemConfig.getExecutionOrder())).collect(Collectors.toList());
                //需要执行的基本法项目的执行顺序下标值列表（为空就是执行全部项目）
                if (algorithmDto.getIsNegateExecutionOrderList() && !CollectionUtils.isEmpty(isExistList)) {
                    //取反并且isExistList非空 -> 跳出当前循环
                    continue;
                }
                if (!algorithmDto.getIsNegateExecutionOrderList() && CollectionUtils.isEmpty(isExistList)) {
                    //非取反并且isExistList空 -> 跳出当前循环
                    continue;
                }
            }
            algorithmDto.setRuleItemBizId(ruleItemConfig.getRuleItemBizId());
            //算法-计算-基本法项目
            Result<AlgorithmResDto> result = ruleItemAlgorithm(algorithmDto);
            algorithmResDtoList.add(result.getData());
        }
        return Result.success(algorithmResDtoList);
    }

    /**
     * 算法-计算-基本法项目
     * @param algorithmDto 算法DTO - 公共入参封装
     * @return
     */
    public Result<AlgorithmResDto> ruleItemAlgorithm(AlgorithmDto algorithmDto) {
        //校验 - 基本法项目配置表唯一业务ID不能为空
        if (StringUtils.isBlank(algorithmDto.getRuleItemBizId())) {
            throw new BusinessException("基本法项目配置表唯一业务ID不能为空");
        }
        //查询对象公式关系 - 即基本法项目和公式关系数据
        List<RelObjectFormula> relObjectFormulaList = iRelObjectFormulaService.queryList(RelObjectFormulaDto.builder().objectBizId(algorithmDto.getRuleItemBizId()).build());
        if (CollectionUtils.isEmpty(relObjectFormulaList)) {
            throw new BusinessException("基本法项目和公式关系数据不存在");
        }
        RelObjectFormula relObjectFormula = relObjectFormulaList.get(0);
        //公式配置表唯一业务ID
        algorithmDto.setFormulaBizId(relObjectFormula.getFormulaBizId());
        Result<AlgorithmResDto> result = calculationFormulaAlgorithm(algorithmDto);
        AlgorithmResDto dto = result.getData();
        dto.setRuleBizId(algorithmDto.getRuleBizId());
        dto.setRuleItemBizId(algorithmDto.getRuleItemBizId());
        return Result.success(dto);
    }

    /**
     * 算法-计算-公式
     * @param algorithmDto 算法DTO - 公共入参封装
     * @return
     */
    public Result<AlgorithmResDto> calculationFormulaAlgorithm(AlgorithmDto algorithmDto) {
        AlgorithmResDto resDto = new AlgorithmResDto();
        //校验算法DTO入参
        checkAlgorithmDto(algorithmDto);
        //获取计算公式对象
        FormulaConfig formulaConfig = iFormulaConfigService.queryOne(algorithmDto.getFormulaBizId());
        if (Objects.isNull(formulaConfig)) {
            throw new BusinessException("计算公式对象不存在");
        }
        //获取计算公式（变量表唯一业务ID组合）
        String calculationFormulaBizId = formulaConfig.getCalculationFormulaBizId();
        if (StringUtils.isBlank(calculationFormulaBizId)) {
            throw new BusinessException("获取计算公式（变量表唯一业务ID组合）不存在");
        }
        //解析计算公式-获取用到的变量表业务ID集合列表
        //从公式字符串中解析出所有变量业务ID（返回列表，保持顺序）
        List<String> variableBizIdList = new ArrayList<>();
        try {
            variableBizIdList = FormulaParser.parseVariableBizIdsOrdered(calculationFormulaBizId);
        }catch (Exception e) {
            throw new BusinessException("从公式字符串中解析出所有变量业务ID，解析失败");
        }
        //检测解析出来的变量表业务ID集合列表是否存在
        if (CollectionUtils.isEmpty(variableBizIdList)) {
            throw new BusinessException("解析计算公式-获取用到的变量表业务ID集合列表不存在");
        }
        //变量表业务ID集合列表去重
        variableBizIdList = variableBizIdList.stream().distinct().collect(Collectors.toList());
        algorithmDto.setVariableBizIdList(variableBizIdList);
        //算法-计算-变量
        Result<List<VariableAlgorithmDto>> variableResult = variableAlgorithm(algorithmDto);
        if (variableResult.getCode() != 200) {
            throw new BusinessException("变量计算失败: " + variableResult.getMsg());
        }

        List<VariableAlgorithmDto> variableValues = variableResult.getData();

        //将变量值代入计算公式进行计算
        BigDecimal result = evaluateFormula(calculationFormulaBizId, variableValues);

        resDto.setCalculatedValue(result);

        //TODO 前变量绑定的SQL模板计算结果集合列表（含计算值，以及计算值的提供者等等）
//        resDto.setSqlAlgorithmResultDtoList();
        return Result.success(resDto);
    }

    /**
     * 计算公式 - 将变量值代入公式进行计算
     * @param formula 公式字符串，如："(variable_1001 + variable_2123) * variable_4455"
     * @param variableValues 变量值列表
     * @return 计算结果
     */
    private BigDecimal evaluateFormula(String formula, List<VariableAlgorithmDto> variableValues) {
        try {
            //创建变量值映射表
            Map<String, BigDecimal> variableMap = new HashMap<>();
            for (VariableAlgorithmDto variable : variableValues) {
                if (StringUtils.isNotBlank(variable.getCalculatedValue())) {
                    try {
                        BigDecimal value = new BigDecimal(variable.getCalculatedValue());
                        variableMap.put(variable.getVariableBizId(), value);
                    } catch (NumberFormatException e) {
                        throw new BusinessException("变量值格式错误: " + variable.getVariableBizId() + " = " + variable.getCalculatedValue());
                    }
                }
            }

            //替换公式中的变量业务ID为实际数值
            String expression = replaceVariablesWithValues(formula, variableMap);

            //计算表达式
            return calculateExpression(expression);

        } catch (Exception e) {
            log.error("计算公式失败, formula: {}, variables: {}", formula, variableValues, e);
            throw new BusinessException("计算公式失败: " + e.getMessage());
        }
    }

    /**
     * 将公式中的变量业务ID替换为实际数值
     * @param formula 原始公式
     * @param variableMap 变量映射表
     * @return 替换后的表达式
     */
    private String replaceVariablesWithValues(String formula, Map<String, BigDecimal> variableMap) {
        String expression = formula;

        //按变量业务ID长度从长到短排序，避免替换时出现部分匹配的问题
        List<String> sortedKeys = variableMap.keySet().stream()
                .sorted((a, b) -> Integer.compare(b.length(), a.length()))
                .collect(Collectors.toList());

        for (String variableBizId : sortedKeys) {
            BigDecimal value = variableMap.get(variableBizId);
            if (value != null) {
                //将变量业务ID替换为数值，确保完全匹配
                expression = expression.replace(variableBizId, value.toString());
            }
        }

        //检查是否还有未替换的变量
        if (containsUnreplacedVariables(expression)) {
            throw new BusinessException("公式中存在未计算的变量: " + expression);
        }

        return expression;
    }

    /**
     * 检查表达式中是否还有未替换的变量
     */
    private boolean containsUnreplacedVariables(String expression) {
        //假设变量业务ID的格式是 "variable_" 开头
        return expression.matches(".*variable_\\w+.*");
    }

    /**
     * 计算数学表达式
     * @param expression 数学表达式，如："(100.5 + 200.3) * 50.2"
     * @return 计算结果
     */
    private BigDecimal calculateExpression(String expression) {
        try {
            //使用JavaScript引擎计算表达式
            ScriptEngineManager manager = new ScriptEngineManager();
            ScriptEngine engine = manager.getEngineByName("JavaScript");

            //计算表达式
            Object result = engine.eval(expression);

            //转换为BigDecimal
            if (result instanceof Number) {
                return new BigDecimal(result.toString());
            } else {
                throw new BusinessException("表达式计算结果不是数字类型: " + result);
            }

        } catch (ScriptException e) {
            throw new BusinessException("表达式计算失败: " + expression + ", 错误: " + e.getMessage());
        }
    }

    /**
     * 算法-计算-变量
     * @param algorithmDto 算法DTO - 公共入参封装
     * @return
     */
    public Result<List<VariableAlgorithmDto>> variableAlgorithm(AlgorithmDto algorithmDto) {
        //校验算法DTO入参
        checkAlgorithmDto(algorithmDto);
        //查询变量表列表信息
        List<Variable> variableList = iVariableService.queryList(VariableDto.builder()
                .variableBizIdList(algorithmDto.getVariableBizIdList())
                .build());
        if (CollectionUtils.isEmpty(variableList)) {
            throw new BusinessException("变量表列表信息不存在");
        }
        List<VariableAlgorithmDto> variableAlgorithmDtoList = new ArrayList<>();
        //遍历变量列表 - 计算对应的值
        for (Variable variable : variableList) {
            VariableAlgorithmDto dto = new VariableAlgorithmDto();
            BeanUtils.copyProperties(variable,dto);
            //变量类型 - 固定值
            if (VariableEnum.FIXED_VALUE.getItemValue().equals(variable.getType())) {
                //固定值，直接取出固定值返回作为公式计算的值
                dto.setCalculatedValue(variable.getValue());
            }else if (VariableEnum.DYNAMIC_COMPUTATION.getItemValue().equals(variable.getType())){
                //动态计算值 - 结合通用入参参数、条件、SQL模板计算出值
                //校验 - 动态计算必须绑定SQL模板。查询对象和SQL模板配置关系表
                List<RelObjectSql> relObjectSqlList = iRelObjectSqlService.queryList(RelObjectSqlDto.builder().objectBizId(variable.getVariableBizId()).build());
                if (CollectionUtils.isEmpty(relObjectSqlList)) {
                    throw new BusinessException("动态计算的变量必须要绑定SQL模板");
                }
                RelObjectSql relObjectSql = relObjectSqlList.get(0);
                //SQL模板表唯一业务ID
                algorithmDto.setSqlTemplateBizId(relObjectSql.getSqlTemplateBizId());
                if (Objects.isNull(algorithmDto.getSqlTemplateParamDto())) {
                    //对象空构造空对象
                    algorithmDto.setSqlTemplateParamDto(new SqlTemplateParamDto());
                }
                //先查询是否有关联条件：有关联条件作为SQL模板的判断依据，有关联条件就查询关联条件类型业务ID带入到SQL模板关联查询判断条件情况。
                List<RelObjectCondition> relObjectConditionList = iRelObjectConditionService.queryList(RelObjectConditionDto.builder()
                        .objectBizId(variable.getVariableBizId())
                        .build());
                if (!CollectionUtils.isEmpty(relObjectConditionList)) {
                    RelObjectCondition relObjectCondition = relObjectConditionList.get(0);
                    //有关联条件就查询关联条件类型业务ID带入到SQL模板关联查询判断条件情况
                    algorithmDto.getSqlTemplateParamDto().setConditionTypeBizId(relObjectCondition.getConditionTypeBizId());
                }

                //执行 - 算法 - SQL模板
                Result<List<SqlAlgorithmResultDto>> result = sqlAlgorithm(algorithmDto);

                // 设置SQL模板计算结果列表
                dto.setSqlAlgorithmResultDtoList(result.getData());

                // 计算值 - 对sqlAlgorithmResultDtoList中所有calculatedValue求和
                if (result.getData() != null && !result.getData().isEmpty()) {
                    BigDecimal sum = BigDecimal.ZERO;
                    for (SqlAlgorithmResultDto sqlResult : result.getData()) {
                        if (StringUtils.isNotBlank(sqlResult.getCalculatedValue())) {
                            try {
                                BigDecimal value = new BigDecimal(sqlResult.getCalculatedValue());
                                sum = sum.add(value);
                            } catch (NumberFormatException e) {
                                log.warn("计算值格式错误，跳过: {}", sqlResult.getCalculatedValue());
                            }
                        }
                    }
                    dto.setCalculatedValue(sum.toString());
                } else {
                    dto.setCalculatedValue("0"); // 默认值
                }
            }
            variableAlgorithmDtoList.add(dto);
        }
        return Result.success(variableAlgorithmDtoList);
    }

    /**
     * 算法 - 执行 - SQL模板
     * @param algorithmDto
     * @return
     */
    public Result<List<SqlAlgorithmResultDto>> sqlAlgorithm(AlgorithmDto algorithmDto) {
        //SQL模板表唯一业务ID非空
        if (StringUtils.isBlank(algorithmDto.getSqlTemplateBizId())) {
            throw new BusinessException("SQL模板表唯一业务ID不能为空");
        }
        //查询SQL模板对象
        CommissionSqlTemplate commissionSqlTemplate = iCommissionSqlTemplateService.queryOne(algorithmDto.getSqlTemplateBizId());
        if (Objects.isNull(commissionSqlTemplate)) {
            throw new BusinessException("SQL模板对象不存在");
        }
        //SQL模板内容不能为空
        if (StringUtils.isBlank(commissionSqlTemplate.getSqlTemplate())) {
            throw new BusinessException("SQL模板内容SQL语句不能为空");
        }
        try {
            //获取SQL模板内容
            String sqlTemplate = commissionSqlTemplate.getSqlTemplate();

            //构建SQL模板入参参数Map
            Map<String, Object> paramMap = buildParamMap(algorithmDto);

            //执行SQL查询
            Object result = executeParameterizedQuery(sqlTemplate, paramMap);

            //将查询结果转换为 List<SqlAlgorithmResultDto>
            List<SqlAlgorithmResultDto> resultDtoList = convertToSqlAlgorithmResultDto(result);

            return Result.success(resultDtoList);

        } catch (Exception e) {
            log.error("执行SQL模板失败, sqlTemplateBizId: {}", algorithmDto.getSqlTemplateBizId(), e);
            throw new BusinessException("执行SQL模板失败: " + e.getMessage());
        }
    }

    /**
     * 将查询结果转换为 List<SqlAlgorithmResultDto>
     * @param result 原始查询结果
     * @return 转换后的结果列表
     */
    private List<SqlAlgorithmResultDto> convertToSqlAlgorithmResultDto(Object result) {
        List<SqlAlgorithmResultDto> resultDtoList = new ArrayList<>();

        if (result == null) {
            return resultDtoList;
        }

        if (result instanceof List) {
            List<?> resultList = (List<?>) result;
            for (Object item : resultList) {
                SqlAlgorithmResultDto dto = convertSingleResult(item);
                if (dto != null) {
                    resultDtoList.add(dto);
                }
            }
        } else {
            // 单个结果的情况
            SqlAlgorithmResultDto dto = convertSingleResult(result);
            if (dto != null) {
                resultDtoList.add(dto);
            }
        }

        return resultDtoList;
    }

    /**
     * 转换单个查询结果
     * @param item 查询结果项
     * @return 转换后的DTO
     */
    private SqlAlgorithmResultDto convertSingleResult(Object item) {
        SqlAlgorithmResultDto dto = new SqlAlgorithmResultDto();

        if (item instanceof Map) {
            // 处理Map类型的结果（多列）
            Map<String, Object> row = (Map<String, Object>) item;

            // 根据实际数据库列名设置provider和calculatedValue
            if (row.containsKey("provider")) {
                dto.setProvider(row.get("provider") != null ? row.get("provider").toString() : null);
            }
            if (row.containsKey("calculated_value")) {
                dto.setCalculatedValue(row.get("calculated_value") != null ? row.get("calculated_value").toString() : null);
            }
            // 可以添加其他字段的映射...

        } else if (item instanceof String || item instanceof Number) {
            // 处理单列查询结果
            dto.setCalculatedValue(item.toString());
            // 单列情况下，provider可以为空或设置默认值
            dto.setProvider("default_provider");
        }

        return dto;
    }

    /**
     * 构建参数Map
     * @param algorithmDto
     * @return
     */
    private Map<String, Object> buildParamMap(AlgorithmDto algorithmDto) {
        SqlTemplateParamDto sqlTemplateParamDto = algorithmDto.getSqlTemplateParamDto();
        if (Objects.isNull(sqlTemplateParamDto)) {
            throw new BusinessException("SQL模板条件入参参数及参数值对象不能为空");
        }
        Map<String, Object> paramMap = new HashMap<>();
        //添加参数 - 把所有SQL模板涉及到的参数都列出来put进去，SQL模板内容有对应的就替换，没有就不替换，就能达到高效通用性
        //条件类型表唯一业务ID
        paramMap.put("conditionTypeBizId", sqlTemplateParamDto.getConditionTypeBizId());
        //保单号
        paramMap.put("policyNo", sqlTemplateParamDto.getPolicyNo());
        //产品代码
        paramMap.put("productCode", sqlTemplateParamDto.getProductCode());
        //保单绑定转介人业务ID
        paramMap.put("brokerBizId", sqlTemplateParamDto.getBrokerBizId());
        return paramMap;
    }

    /**
     * 执行参数化查询
     * @param sqlTemplate
     * @param paramMap
     * @return
     */
    private Object executeParameterizedQuery(String sqlTemplate, Map<String, Object> paramMap) {
        //使用NamedParameterJdbcTemplate执行参数化查询
        NamedParameterJdbcTemplate jdbcTemplate = getJdbcTemplate();

        //判断SQL类型：查询还是更新
        if (isSelectQuery(sqlTemplate)) {
            //查询操作
            List<Map<String, Object>> resultList = jdbcTemplate.queryForList(sqlTemplate, paramMap);

            //根据业务需求处理结果
            if (resultList.isEmpty()) {
                return null;
            } else if (resultList.size() == 1) {
                Map<String, Object> singleResult = resultList.get(0);
                //如果只有一列，直接返回值
                if (singleResult.size() == 1) {
                    return singleResult.values().iterator().next();
                }
                return singleResult;
            } else {
                return resultList;
            }
        } else {
            //更新操作（INSERT/UPDATE/DELETE）
            int affectedRows = jdbcTemplate.update(sqlTemplate, paramMap);
            return affectedRows;
        }
    }

    /**
     * 判断是否为SELECT查询
     * @param sql
     * @return
     */
    private boolean isSelectQuery(String sql) {
        String trimmedSql = sql.trim().toLowerCase();
        return trimmedSql.startsWith("select");
    }

    /**
     * 校验算法DTO入参
     * @param algorithmDto
     * @return
     */
    public Result checkAlgorithmDto(AlgorithmDto algorithmDto) {
        if (Objects.isNull(algorithmDto)) {
            throw new BusinessException("算法DTO入参对象不能为空");
        }
        if (StringUtils.isBlank(algorithmDto.getFormulaBizId())) {
            throw new BusinessException("公式配置表唯一业务ID不能为空");
        }
        if (CollectionUtils.isEmpty(algorithmDto.getVariableBizIdList())) {
            throw new BusinessException("变量表业务ID集合列表不能为空");
        }
        return Result.success();
    }
}
