package com.yd.csf.api.service.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.enums.CommonEnum;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;
import com.yd.common.utils.RandomStringGenerator;
import com.yd.csf.api.service.ApiPremiumRemittanceFileService;
import com.yd.csf.feign.dto.premiumremittancefile.ApiPremiumRemittanceFileDto;
import com.yd.csf.feign.request.premiumremittancefile.ApiPremiumRemittanceFilePageRequest;
import com.yd.csf.feign.response.premiumremittancefile.ApiPremiumRemittanceFilePageResponse;
import com.yd.csf.service.model.PremiumRemittanceFile;
import com.yd.csf.service.service.IPremiumRemittanceFileService;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

@Slf4j
@Service
public class ApiPremiumRemittanceFileServiceImpl implements ApiPremiumRemittanceFileService {

    @Autowired
    private IPremiumRemittanceFileService iPremiumRemittanceFileService;

    /**
     * 分页列表查询-保费汇款记录附件信息
     * @param request
     * @return
     */
    @Override
    public Result<IPage<ApiPremiumRemittanceFilePageResponse>> page(ApiPremiumRemittanceFilePageRequest request) {
        Page<ApiPremiumRemittanceFilePageResponse> page = new Page<>(request.getPageNo(), request.getPageSize());
        IPage<ApiPremiumRemittanceFilePageResponse> iPage = iPremiumRemittanceFileService.page(page, request);
        return Result.success(iPage);
    }

    /**
     * 添加-保费汇款记录附件信息
     * @param request
     * @return
     */
    @Override
    public Result add(ApiPremiumRemittanceFileDto request) {
        if (StringUtils.isBlank(request.getPremiumRemittanceBizId())) {
            throw new BusinessException("保费对账-汇款记录表唯一业务ID不能为空");
        }
        PremiumRemittanceFile premiumRemittanceFile = new PremiumRemittanceFile();
        BeanUtils.copyProperties(request,premiumRemittanceFile);
        premiumRemittanceFile.setPremiumRemittanceBizId(request.getPremiumRemittanceBizId());
        premiumRemittanceFile.setPremiumRemittanceFileBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PREMIUM_REMITTANCE_FILE.getCode()));
        iPremiumRemittanceFileService.saveOrUpdate(premiumRemittanceFile);
        return Result.success();
    }

    /**
     * 编辑-保费汇款记录附件信息
     * @param request
     * @return
     */
    @Override
    public Result edit(ApiPremiumRemittanceFileDto request) {
        if (StringUtils.isBlank(request.getPremiumRemittanceBizId())) {
            throw new BusinessException("保费对账-汇款记录表唯一业务ID不能为空");
        }
        if (StringUtils.isBlank(request.getPremiumRemittanceFileBizId())) {
            throw new BusinessException("保费汇款记录附件表唯一业务ID不能为空");
        }
        Result<PremiumRemittanceFile> result = checkPremiumRemittanceFileIsExist(request.getPremiumRemittanceFileBizId());
        PremiumRemittanceFile file = result.getData();
        BeanUtils.copyProperties(request,file);
        iPremiumRemittanceFileService.saveOrUpdate(file);
        return Result.success();
    }

    /**
     * 删除-保费汇款记录附件信息
     * @param premiumRemittanceFileBizId
     * @return
     */
    @Override
    public Result del(String premiumRemittanceFileBizId) {
        Result<PremiumRemittanceFile> result = checkPremiumRemittanceFileIsExist(premiumRemittanceFileBizId);
        PremiumRemittanceFile file = result.getData();
        iPremiumRemittanceFileService.removeById(file.getId());
        return Result.success();
    }

    /**
     * 添加-保费对账汇款记录附件列表
     * @param apiPremiumRemittanceFileDtoList
     * @param premiumRemittanceBizId
     * @return
     */
    @Override
    public Result addPremiumRemittanceFileList(List<ApiPremiumRemittanceFileDto> apiPremiumRemittanceFileDtoList,
                                               String premiumRemittanceBizId) {
        if (CollectionUtils.isEmpty(apiPremiumRemittanceFileDtoList)) {
            return Result.success();
        }
        List<PremiumRemittanceFile> addPremiumRemittanceFileList = apiPremiumRemittanceFileDtoList
                .stream().map(dto -> {
                    PremiumRemittanceFile file = new PremiumRemittanceFile();
                    BeanUtils.copyProperties(dto,file);
                    file.setPremiumRemittanceBizId(premiumRemittanceBizId);
                    file.setPremiumRemittanceFileBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PREMIUM_REMITTANCE_FILE.getCode()));
                    return file;
                }).collect(Collectors.toList());
        iPremiumRemittanceFileService.saveOrUpdateBatch(addPremiumRemittanceFileList);
        return Result.success();
    }

    /**
     * 保存-保费对账汇款记录附件列表
     * @param apiPremiumRemittanceFileDtoList
     * @param premiumRemittanceBizId
     * @return
     */
    @Override
    public Result savePremiumRemittanceFileList(List<ApiPremiumRemittanceFileDto> apiPremiumRemittanceFileDtoList,
                                                String premiumRemittanceBizId) {
        //先删后新增
        iPremiumRemittanceFileService.delByPremiumRemittanceBizId(premiumRemittanceBizId);
        //新增
        if (CollectionUtils.isEmpty(apiPremiumRemittanceFileDtoList)) {
            return Result.success();
        }
        List<PremiumRemittanceFile> saveList = apiPremiumRemittanceFileDtoList.stream().map(dto -> {
            PremiumRemittanceFile file = new PremiumRemittanceFile();
            BeanUtils.copyProperties(dto,file);
            file.setPremiumRemittanceBizId(premiumRemittanceBizId);
            file.setPremiumRemittanceFileBizId(RandomStringGenerator.generateBizId16(CommonEnum.UID_TYPE_PREMIUM_REMITTANCE_FILE.getCode()));
            return file;
        }).collect(Collectors.toList());
        iPremiumRemittanceFileService.saveOrUpdateBatch(saveList);
        return Result.success();
    }

    /**
     * 根据保费对账-汇款记录表唯一业务ID删除
     * @param premiumRemittanceBizId
     * @return
     */
    @Override
    public Result delByPremiumRemittanceBizId(String premiumRemittanceBizId) {
        iPremiumRemittanceFileService.delByPremiumRemittanceBizId(premiumRemittanceBizId);
        return Result.success();
    }

    /**
     * 根据保费对账-汇款记录表唯一业务ID查询保费汇款记录附件列表
     * @param premiumRemittanceBizId
     * @return
     */
    @Override
    public List<ApiPremiumRemittanceFileDto> apiPremiumRemittanceFileDtoList(String premiumRemittanceBizId) {
        List<PremiumRemittanceFile> fileList = iPremiumRemittanceFileService.queryList(premiumRemittanceBizId);
        if (CollectionUtils.isEmpty(fileList)) {
            return new ArrayList<>();
        }
        List<ApiPremiumRemittanceFileDto> fileDtoList = fileList.stream().map(dto -> {
            ApiPremiumRemittanceFileDto fileDto = new ApiPremiumRemittanceFileDto();
            BeanUtils.copyProperties(dto,fileDto);
            return fileDto;
        }).collect(Collectors.toList());
        return fileDtoList;
    }

    /**
     * 校验保费汇款记录附件信息是否存在
     * @param premiumRemittanceFileBizId
     * @return
     */
    @Override
    public Result<PremiumRemittanceFile> checkPremiumRemittanceFileIsExist(String premiumRemittanceFileBizId) {
        PremiumRemittanceFile premiumRemittanceFile = iPremiumRemittanceFileService.queryOne(premiumRemittanceFileBizId);
        if (Objects.isNull(premiumRemittanceFile)) {
            //数据不存在
            throw new BusinessException("保费汇款记录附件信息不存在");
        }
        return Result.success(premiumRemittanceFile);
    }
}
