package com.yd.csf.api.controller;

import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.yd.common.exception.BusinessException;
import com.yd.common.result.Result;

import com.yd.csf.api.service.ApiAppointmentService;
import com.yd.csf.feign.dto.appointment.ApiAppointmentInfoDto;
import com.yd.csf.service.common.ErrorCode;
import com.yd.csf.service.dto.DeleteFnaRequest;
import com.yd.csf.service.dto.FnaAddRequest;
import com.yd.csf.service.dto.FnaQueryRequest;
import com.yd.csf.service.dto.FnaUpdateRequest;
import com.yd.csf.service.model.Customer;
import com.yd.csf.service.model.Fna;
import com.yd.csf.service.service.CustomerService;
import com.yd.csf.service.service.FnaService;
import com.yd.csf.service.vo.FnaVO;
import com.yd.user.feign.client.sysuser.ApiSysUserFeignClient;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

/**
 * Fna接口
 *
 * @author jianan
 * @since 2025-08-31
 */
@RestController
@RequestMapping("/Fna")
@Tag(name = "流程接口")
public class ApiFnaController {

    @Resource
    private FnaService fnaService;
    @Resource
    private CustomerService customerService;
    @Resource
    private ApiAppointmentService appointmentService;
    @Resource
    private ApiSysUserFeignClient apiSysUserFeignClient;

    /**
     * 新建流程
     *
     * @param fnaAddRequest
     * @param request
     * @return
     */
    @PostMapping("/add")
    @Operation(summary = "新建流程")
    public Result<String> addFna(@RequestBody FnaAddRequest fnaAddRequest, HttpServletRequest request) {
        if (fnaAddRequest == null) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(fnaService.addFna(fnaAddRequest));
    }

    /**
     * 删除流程
     *
     * @param deleteRequest
     * @param request
     * @return
     */
    @PostMapping("/delete")
    @Operation(summary = "删除流程")
    public Result<Boolean> deleteFna(@RequestBody DeleteFnaRequest deleteRequest, HttpServletRequest request) {
        if (deleteRequest == null || StringUtils.isBlank(deleteRequest.getFnaBizId())) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        return Result.success(fnaService.deleteFna(deleteRequest));
    }

    /**
     * 更新流程
     *
     * @param fnaUpdateRequest
     * @return
     */
    @PostMapping("/update")
    @Operation(summary = "更新流程")
    public Result<Boolean> updateFna(@RequestBody FnaUpdateRequest fnaUpdateRequest) {
        if (fnaUpdateRequest == null || StringUtils.isBlank(fnaUpdateRequest.getFnaBizId())) {
            throw new BusinessException(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }

        // 判断是否存在
        String fnaBizId = fnaUpdateRequest.getFnaBizId();
        Fna fna = fnaService.getByBizId(fnaBizId);
        if (fna == null) {
            throw new BusinessException(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }

        BeanUtils.copyProperties(fnaUpdateRequest, fna,"fnaBizId");

        if (StringUtils.isNotBlank(fnaUpdateRequest.getCustomerBizId())) {
            Customer customer = customerService.getByCustomerBizId(fnaUpdateRequest.getCustomerBizId());
            fna.setCustomerName(customer.getName());
        }
        if (StringUtils.isNotBlank(fnaUpdateRequest.getAppointmentBizId())) {
            Result<ApiAppointmentInfoDto> apiAppointmentInfoDtoResult = appointmentService.mainDetail(fnaUpdateRequest.getAppointmentBizId());
            if (apiAppointmentInfoDtoResult.getData() != null) {
                ApiAppointmentInfoDto apiAppointmentInfoDto = apiAppointmentInfoDtoResult.getData();
                fna.setAppointmentNo(apiAppointmentInfoDto.getAppointmentNo());
            }
        }

        // 操作数据库
        boolean result = fnaService.updateById(fna);
        if (!result) {
            return Result.fail(ErrorCode.OPERATION_ERROR.getCode(), ErrorCode.OPERATION_ERROR.getMessage());
        }
        return Result.success(true);
    }

    /**
     * 根据 fnaBizId 获取fna（封装类）
     *
     * @param fnaBizId
     * @return
     */
    @GetMapping("/get/vo")
    @Operation(summary = "流程详情")
    public Result<FnaVO> getFnaVOByBizId(String fnaBizId, HttpServletRequest request) {
        if (StringUtils.isBlank(fnaBizId)) {
            return Result.fail(ErrorCode.PARAMS_ERROR.getCode(), ErrorCode.PARAMS_ERROR.getMessage());
        }
        // 查询数据库
        Fna fna = fnaService.getByBizId(fnaBizId);
        if (fna == null) {
            return Result.fail(ErrorCode.NOT_FOUND_ERROR.getCode(), ErrorCode.NOT_FOUND_ERROR.getMessage());
        }

        // 获取封装类
        return Result.success(fnaService.getFnaVO(fna));
    }

    /**
     * 分页获取fna列表（仅管理员可用）
     *
     * @param fnaQueryRequest
     * @return
     */
    @PostMapping("/list/page")
    @Operation(summary = "分页获取流程列表")
    public Result<Page<FnaVO>> listFnaByPage(@RequestBody FnaQueryRequest fnaQueryRequest) {
        long current = fnaQueryRequest.getPageNo();
        long size = fnaQueryRequest.getPageSize();
        // 查询数据库
        Page<Fna> fnaPage = fnaService.page(new Page<>(current, size),
                fnaService.getQueryWrapper(fnaQueryRequest));
        return Result.success(fnaService.getFnaVOPage(fnaPage));
    }

    /**
     * 分页获取fna列表（封装类）
     *
     * @param fnaQueryRequest
     * @param request
     * @return
     */
//    @PostMapping("/list/page/vo")
//    public Result<Page<Fna>> listFnaVOByPage(@RequestBody FnaQueryRequest fnaQueryRequest,
//                                             HttpServletRequest request) {
//        long current = fnaQueryRequest.getPageNo();
//        long size = fnaQueryRequest.getPageSize();
//
//        // 查询数据库
//        Page<Fna> fnaPage = fnaService.page(new Page<>(current, size),
//                fnaService.getQueryWrapper(fnaQueryRequest));
//        // 获取封装类
////        return Result.success(fnaService.getFnaVOPage(fnaPage, request));
//
//        return Result.success(fnaPage);
//    }


}
