package com.yd.gateway.utils;

import com.yd.gateway.exception.JwtAuthException;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.UnsupportedJwtException;
import io.jsonwebtoken.io.Decoders;
import io.jsonwebtoken.security.Keys;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import javax.crypto.SecretKey;

/**
 * JWT令牌提供者
 * 修复了密钥长度不足的问题，并优化了异常处理
 */
@Component
public class JwtTokenUtil {

    // 使用512位长度的密钥（通过配置文件设置）
    @Value("${jwt.secret}")
    private String jwtSecret;

    @Value("${jwt.expiration}") // 默认24小时(单位毫秒)
    private int jwtExpiration;

    /**
     * 验证令牌有效性
     */
    public boolean validateToken(String token) {
        try {
            Jwts.parserBuilder().setSigningKey(getSigningKey()).build().parseClaimsJws(token);
            return true;
        } catch (SecurityException | MalformedJwtException e) {
            throw new JwtAuthException("无效的JWT签名");
        } catch (ExpiredJwtException e) {
            throw new JwtAuthException("JWT令牌已过期");
        } catch (UnsupportedJwtException e) {
            throw new JwtAuthException("不支持的JWT令牌");
        } catch (IllegalArgumentException e) {
            throw new JwtAuthException("JWT令牌无效");
        } catch (Exception e) {
            throw new JwtAuthException("无效的JWT令牌: " + e.getMessage());
        }
    }

    /**
     * 获取签名密钥
     */
    private SecretKey getSigningKey() {
        try {
            if (jwtSecret == null || jwtSecret.trim().isEmpty()) {
                throw new JwtAuthException("JWT密钥未配置");
            }

            byte[] keyBytes = Decoders.BASE64.decode(jwtSecret);
            return Keys.hmacShaKeyFor(keyBytes);
        } catch (Exception e) {
            throw new JwtAuthException("获取签名密钥失败: " + e.getMessage());
        }
    }


}
