package com.yd.auth.core.dto;

import lombok.Data;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 * 用户认证信息数据传输对象
 * 实现Spring Security的UserDetails接口，用于封装用户认证信息
 */
@Data
public class AuthUserDto implements UserDetails {

    /**
     * 系统用户主键id（数据库唯一标识）
     */
    private Long id;

    /**
     * 用户唯一标识(业务ID)，用于业务层面的用户识别
     */
    private String userUid;

    /**
     * 登录账号（实现UserDetails接口必需）
     */
    private String username;

    /**
     * 加密后的密码（实现UserDetails接口必需）
     */
    private String password;

    // ============== Spring Security账户状态属性 ==============
    /**
     * 账户是否启用（默认true启用）
     */
    private Boolean enabled = true;

    /**
     * 账户是否未过期（默认true未过期）
     */
    private Boolean accountNonExpired = true;

    /**
     * 凭证(密码)是否未过期（默认true未过期）
     */
    private Boolean credentialsNonExpired = true;

    /**
     * 账户是否未锁定（默认true未锁定）
     */
    private Boolean accountNonLocked = true;

    /**
     * 用户关联的角色列表
     */
    private List<AuthRoleDto> roles;

    /**
     * 获取用户权限集合（实现UserDetails接口必需）
     * Spring Security通过此方法获取用户的权限/角色信息
     *
     * @return 权限集合，每个权限以GrantedAuthority对象表示
     */
    @Override
    public Collection<? extends GrantedAuthority> getAuthorities() {
        // 创建权限集合
        List<GrantedAuthority> authorities = new ArrayList<GrantedAuthority>();

        // 检查角色列表是否有效
        if (roles != null && !roles.isEmpty()) {
            // 遍历所有角色
            for (AuthRoleDto role : roles) {
                // 将角色名转换为Spring Security认可的权限标识（添加ROLE_前缀）
                authorities.add(new SimpleGrantedAuthority("ROLE_" + role.getName()));
            }
        }
        return authorities;
    }


    /**
     * 检查账户是否未过期
     * @return true=未过期，false=已过期
     */
    @Override
    public boolean isAccountNonExpired() {
        return accountNonExpired;
    }

    /**
     * 检查账户是否未锁定
     * @return true=未锁定，false=已锁定
     */
    @Override
    public boolean isAccountNonLocked() {
        return accountNonLocked;
    }

    /**
     * 检查凭证(密码)是否未过期
     * @return true=未过期，false=已过期
     */
    @Override
    public boolean isCredentialsNonExpired() {
        return credentialsNonExpired;
    }

    /**
     * 检查账户是否启用
     * @return true=启用，false=禁用
     */
    @Override
    public boolean isEnabled() {
        return enabled;
    }
}
