package com.yd.common.utils;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Random;

/**
 * 编号生成工具类
 */
public class CodeGenerator {
    
    // 单例模式确保随机数生成器效率
    private static final Random random = new Random();
    private static final DateTimeFormatter DATE_FORMATTER = DateTimeFormatter.ofPattern("yyMMdd");

    /**
     * 生成编号
     * @param letter 首字母
     * @return 格式：首字母 + 6位日期（250903） + 7位随机数字
     */
    public static String generateCode(String letter) {
        //6位日期
        String datePart = LocalDateTime.now().format(DATE_FORMATTER);
        //7位随机数字
        String randomPart = generateRandomDigits(7);
        return letter + datePart + randomPart;
    }
    
    /**
     * 生成指定长度的随机数字字符串
     * @param length 数字字符串长度
     * @return 随机数字字符串
     */
    private static String generateRandomDigits(int length) {
        if (length <= 0) {
            throw new IllegalArgumentException("长度必须为正整数");
        }
        
        // 生成指定位数的随机数字
        int maxValue = (int) Math.pow(10, length);
        int randomNumber = random.nextInt(maxValue);
        
        // 格式化为固定长度，前面补零
        return String.format("%0" + length + "d", randomNumber);
    }


    /**
     * 生成编码：前缀 + 指定位数的随机纯数字
     * @param prefix
     * @param length
     * @return
     */
    public static String generateCode(String prefix,int length) {
        return prefix + generateRandomNumbers(length);
    }

    /**
     * 生成编码：前缀 + 10位的随机纯数字
     * @param prefix
     * @return
     */
    public static String generate10Code(String prefix) {
        return prefix + generateRandomNumbers(10);
    }

    /**
     * 生成指定位数的随机纯数字字符串
     * @param length 数字位数
     * @return 随机数字字符串
     */
    private static String generateRandomNumbers(int length) {
        if (length <= 0) {
            throw new IllegalArgumentException("长度必须大于0");
        }

        Random random = new Random();
        StringBuilder sb = new StringBuilder(length);

        // 第一位不能为0
        sb.append(random.nextInt(9) + 1);

        // 生成剩余位数的随机数字
        for (int i = 1; i < length; i++) {
            sb.append(random.nextInt(10));
        }

        return sb.toString();
    }

    public static void main(String[] args) {
        System.out.println(generate10Code("RULE"));
    }
}
