package com.yd.common.utils;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;

import java.io.IOException;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;

public class CustomLocalDateTimeDeserializer extends JsonDeserializer<LocalDateTime> {

    // 定义多种可能的格式
    private static final DateTimeFormatter[] FORMATTERS = {
            // ISO 格式: yyyy-MM-dd'T'HH:mm:ss
            DateTimeFormatter.ISO_LOCAL_DATE_TIME,
            // 标准格式: yyyy-MM-dd HH:mm:ss
            DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"),
            // 只有日期: yyyy-MM-dd
            DateTimeFormatter.ofPattern("yyyy-MM-dd"),
            // 其他可能的格式
            DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm:ss"),
            DateTimeFormatter.ofPattern("yyyy/MM/dd"),
            DateTimeFormatter.ofPattern("yyyy.MM.dd HH:mm:ss"),
            DateTimeFormatter.ofPattern("yyyy.MM.dd")
    };

    @Override
    public LocalDateTime deserialize(JsonParser p, DeserializationContext ctxt) throws IOException {
        String text = p.getText();
        if (text == null || text.trim().isEmpty()) {
            return null;
        }

        text = text.trim();

        // 尝试用所有格式解析
        for (DateTimeFormatter formatter : FORMATTERS) {
            try {
                // 如果是日期格式，转换为 LocalDateTime
                if (formatter.toString().contains("yyyy-MM-dd") &&
                        !formatter.toString().contains("HH:mm:ss")) {
                    // 这是纯日期格式，需要转换为 LocalDateTime
                    return LocalDateTime.parse(text + "T00:00:00", DateTimeFormatter.ISO_LOCAL_DATE_TIME);
                }
                return LocalDateTime.parse(text, formatter);
            } catch (Exception e) {
                // 继续尝试下一个格式
                continue;
            }
        }

        // 如果所有格式都失败，尝试更灵活的解析
        return parseFlexible(text);
    }

    private LocalDateTime parseFlexible(String text) throws IOException {
        try {
            // 尝试处理带T的日期时间
            if (text.contains("T")) {
                // 移除可能的多余空格
                text = text.replace(" ", "");
                // 确保T后面有时间部分
                if (text.indexOf('T') + 1 < text.length()) {
                    String timePart = text.substring(text.indexOf('T') + 1);
                    if (timePart.isEmpty()) {
                        text = text + "00:00:00";
                    } else if (timePart.split(":").length == 2) {
                        text = text + ":00";
                    }
                    return LocalDateTime.parse(text, DateTimeFormatter.ISO_LOCAL_DATE_TIME);
                }
            }

            // 尝试处理只有日期的格式
            if (text.length() == 10 && text.charAt(4) == '-' && text.charAt(7) == '-') {
                return LocalDateTime.parse(text + "T00:00:00", DateTimeFormatter.ISO_LOCAL_DATE_TIME);
            }

            // 尝试处理包含空格的日期时间
            if (text.length() >= 10 && text.charAt(4) == '-' && text.charAt(7) == '-') {
                if (text.length() > 10 && text.charAt(10) == ' ') {
                    // 格式: yyyy-MM-dd HH:mm:ss
                    return LocalDateTime.parse(text, DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss"));
                }
            }

            throw new IOException("无法解析日期时间: " + text);
        } catch (Exception e) {
            throw new IOException("无法解析日期时间: " + text, e);
        }
    }
}