package com.yd.auth.core.security;

import com.yd.auth.core.dto.AuthUserDto;
import io.jsonwebtoken.*;
import io.jsonwebtoken.io.Decoders;
import io.jsonwebtoken.security.Keys;
import io.jsonwebtoken.security.SecurityException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.stereotype.Component;
import javax.crypto.SecretKey;
import java.util.*;
import java.util.stream.Collectors;

/**
 * JWT令牌提供者
 * 修复了密钥长度不足的问题，并优化了异常处理
 */
@Slf4j
@Component
public class JwtTokenProvider {

    // 使用512位长度的密钥（通过配置文件设置）
    @Value("${jwt.secret}")
    private String jwtSecret;

    @Value("${jwt.expiration}") // 默认24小时(单位毫秒)
    private long jwtExpiration;

    /**
     * 生成JWT令牌
     */
    public String generateToken(Authentication authentication) {
        if (authentication == null || authentication.getPrincipal() == null) {
            throw new JwtAuthenticationException("无法从空认证信息生成令牌");
        }

        log.info("jwtExpiration毫秒值：{}",jwtExpiration);
        AuthUserDto authUserDto = (AuthUserDto) authentication.getPrincipal();

        try {
            return Jwts.builder()
                    .setSubject(authUserDto.getUserBizId())
                    .claim("roles", getRoles(authUserDto.getAuthorities()))
                    .setIssuedAt(new Date())
                    .setExpiration(new Date(System.currentTimeMillis() + jwtExpiration))
                    .signWith(getSigningKey(), SignatureAlgorithm.HS512)
                    .compact();
        } catch (Exception e) {
            throw new JwtAuthenticationException("生成JWT令牌失败: " + e.getMessage());
        }
    }

    /**
     * 提取角色信息
     */
    private Collection<String> getRoles(Collection<? extends GrantedAuthority> authorities) {
        return authorities.stream()
                .map(GrantedAuthority::getAuthority)
                .collect(Collectors.toList());
    }

    /**
     * 从令牌中获取用户名
     */
    public String getUsernameFromToken(String token) {
        try {
            return Jwts.parserBuilder()
                    .setSigningKey(getSigningKey())
                    .build()
                    .parseClaimsJws(token)
                    .getBody()
                    .getSubject();
        } catch (Exception e) {
            throw new JwtAuthenticationException("从令牌获取用户名失败: " + e.getMessage());
        }
    }

    /**
     * 从令牌中获取用户业务id
     */
    public String getUserBizIdFromToken(String token) {
        try {
            return Jwts.parserBuilder()
                    .setSigningKey(getSigningKey())
                    .build()
                    .parseClaimsJws(token)
                    .getBody()
                    .getSubject();
        } catch (Exception e) {
            throw new JwtAuthenticationException("从令牌获取用户名失败: " + e.getMessage());
        }
    }

    /**
     * 从令牌中获取认证信息
     */
    public Authentication getAuthentication(String token) {
        try {
            if (validateToken(token)) {
                String username = getUsernameFromToken(token);
                return new UsernamePasswordAuthenticationToken(
                        username,
                        null,
                        getAuthoritiesFromToken(token)
                );
            }
            return null;
        } catch (Exception e) {
            throw new JwtAuthenticationException("获取认证信息失败: " + e.getMessage());
        }
    }

    /**
     * 从令牌中获取权限信息
     */
    private Collection<? extends GrantedAuthority> getAuthoritiesFromToken(String token) {
        try {
            Claims claims = Jwts.parserBuilder()
                    .setSigningKey(getSigningKey())
                    .build()
                    .parseClaimsJws(token)
                    .getBody();

            @SuppressWarnings("unchecked")
            Collection<String> roles = (Collection<String>) claims.get("roles");

            return roles.stream()
                    .map(role -> (GrantedAuthority) () -> role)
                    .collect(Collectors.toList());
        } catch (Exception e) {
            throw new JwtAuthenticationException("从令牌获取权限信息失败: " + e.getMessage());
        }
    }

    /**
     * 验证令牌有效性
     */
    public boolean validateToken(String token) {
        try {
            Jwts.parserBuilder().setSigningKey(getSigningKey()).build().parseClaimsJws(token);
            return true;
        } catch (SecurityException | MalformedJwtException e) {
            throw new JwtAuthenticationException("无效的JWT签名");
        } catch (ExpiredJwtException e) {
            throw new JwtAuthenticationException("JWT令牌已过期");
        } catch (UnsupportedJwtException e) {
            throw new JwtAuthenticationException("不支持的JWT令牌");
        } catch (IllegalArgumentException e) {
            throw new JwtAuthenticationException("JWT令牌无效");
        } catch (Exception e) {
            throw new JwtAuthenticationException("无效的JWT令牌: " + e.getMessage());
        }
    }

    /**
     * 获取签名密钥
     */
    private SecretKey getSigningKey() {
        try {
            if (jwtSecret == null || jwtSecret.trim().isEmpty()) {
                throw new JwtAuthenticationException("JWT密钥未配置");
            }

            byte[] keyBytes = Decoders.BASE64.decode(jwtSecret);
            return Keys.hmacShaKeyFor(keyBytes);
        } catch (Exception e) {
            throw new JwtAuthenticationException("获取签名密钥失败: " + e.getMessage());
        }
    }

    /**
     * 生成符合HS512要求的密钥（用于生成配置文件中的密钥）
     */
    public static String generateSecureKey() {
        try {
            SecretKey key = Keys.secretKeyFor(SignatureAlgorithm.HS512);
            return Base64.getEncoder().encodeToString(key.getEncoded());
        } catch (Exception e) {
            throw new RuntimeException("生成安全密钥失败", e);
        }
    }

    public long getJwtExpiration() {
        return jwtExpiration;
    }

    /**
     * JWT认证异常
     */
    public static class JwtAuthenticationException extends RuntimeException {
        public JwtAuthenticationException(String message) {
            super(message);
        }

        public JwtAuthenticationException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    // 解析 JWT 声明
    public Claims parseClaims(String token) {
        return Jwts.parserBuilder()
                .setSigningKey(getSigningKey())
                .build()
                .parseClaimsJws(token)
                .getBody();
    }

    // 获取 JWT 头部信息
    public Map<String, Object> getHeaders(String token) {
        return Jwts.parserBuilder()
                .setSigningKey(getSigningKey())
                .build()
                .parseClaimsJws(token)
                .getHeader();
    }

// 在 JwtTokenProvider 类中添加以下方法

    /**
     * 解析 JWT token 并获取过期时间
     * 这个方法不需要 Spring 依赖注入，可以直接调用
     */
    public static void analyzeToken(String token, String secretKey) {
        try {
            // 使用给定的密钥进行解析
            byte[] keyBytes = Base64.getDecoder().decode(secretKey);
            SecretKey key = Keys.hmacShaKeyFor(keyBytes);

            // 解析 token
            Jws<Claims> claimsJws = Jwts.parserBuilder()
                    .setSigningKey(key)
                    .build()
                    .parseClaimsJws(token);

            Claims claims = claimsJws.getBody();

            System.out.println("Token 解析成功:");
            System.out.println("Subject (用户ID): " + claims.getSubject());
            System.out.println("Issued At (签发时间): " + new Date(claims.getIssuedAt().getTime()));
            System.out.println("Expiration (过期时间): " + new Date(claims.getExpiration().getTime()));
            System.out.println("Roles: " + claims.get("roles"));

            // 计算剩余有效期
            long currentTime = System.currentTimeMillis();
            long expirationTime = claims.getExpiration().getTime();
            long remainingTime = expirationTime - currentTime;

            System.out.println("当前时间: " + new Date(currentTime));
            System.out.println("距离过期还有: " + formatDuration(remainingTime));

        } catch (ExpiredJwtException e) {
            // 即使过期了，也能获取到 claims
            Claims claims = e.getClaims();
            System.out.println("Token 已过期:");
            System.out.println("Subject (用户ID): " + claims.getSubject());
            System.out.println("Issued At (签发时间): " + new Date(claims.getIssuedAt().getTime()));
            System.out.println("Expiration (过期时间): " + new Date(claims.getExpiration().getTime()));
            System.out.println("过期于: " + formatDuration(System.currentTimeMillis() - claims.getExpiration().getTime()) + "前");

        } catch (Exception e) {
            System.err.println("解析 Token 失败: " + e.getMessage());
            e.printStackTrace();
        }
    }

    /**
     * 格式化时间间隔
     */
    private static String formatDuration(long millis) {
        if (millis < 0) {
            return "已过期 " + formatDuration(-millis);
        }

        long seconds = millis / 1000;
        long minutes = seconds / 60;
        long hours = minutes / 60;
        long days = hours / 24;

        if (days > 0) {
            return days + "天 " + (hours % 24) + "小时";
        } else if (hours > 0) {
            return hours + "小时 " + (minutes % 60) + "分钟";
        } else if (minutes > 0) {
            return minutes + "分钟 " + (seconds % 60) + "秒";
        } else {
            return seconds + "秒";
        }
    }

    /**
     * 修改 main 方法，使用配置文件中的密钥
     */
    public static void main(String[] args) {
        // 使用配置文件中的密钥
        String configSecret = "v92cuMDOZEd0WIbmX9LHQpZ0HgMIfRPXvDXGmP+5ThbADd2YdD0mCbGujIvLSBZR2ZWby8PKYRT2ReBQ7wIDPQ==";

        // 要解析的 token
        String token = "eyJhbGciOiJIUzUxMiJ9.eyJzdWIiOiJ1c2VyXzEwMDEiLCJyb2xlcyI6W10sImlhdCI6MTc2ODU2MjQwMSwiZXhwIjozMzQ2NDEwMDAxfQ.izMcWJfmhgi3rpyHn1_fNTbSYjWQMAjRN21hNWTBsbp84tsJSWaYew2Zr_dyyKf3woKoH_QsQ74FZl4gAs7Fug";

        // 分析 token
        System.out.println("使用配置文件中的密钥分析 Token:");
        analyzeToken(token, configSecret);

        System.out.println("\n--- 生成新的测试 Token ---");

        // 使用配置文件中的密钥生成一个新的测试 token
        try {
            byte[] keyBytes = Base64.getDecoder().decode(configSecret);
            SecretKey key = Keys.hmacShaKeyFor(keyBytes);

            // 生成一个有效期更合理的测试 token（24小时）
            long expiration = 1577847600000L; // 24小时

            String newToken = Jwts.builder()
                    .setSubject("test_user_001")
                    .claim("roles", Arrays.asList("ROLE_USER"))
                    .setIssuedAt(new Date())
                    .setExpiration(new Date(System.currentTimeMillis() + expiration))
                    .signWith(key, SignatureAlgorithm.HS512)
                    .compact();

            System.out.println("新生成的 Token: " + newToken);
            System.out.println("\n解析新生成的 Token:");
            analyzeToken(newToken, configSecret);

        } catch (Exception e) {
            System.err.println("生成测试 Token 失败: " + e.getMessage());
        }
    }
}
